/**
 * SLA Utility Functions
 */

/**
 * Calculate SLA deadline based on job type and priority
 */
const calculateSLADeadline = (jobType, priority, createdAt) => {
  const created = new Date(createdAt);
  let deadline;

  if (jobType === 'PERIODIC') {
    // Periodic Maintenance: Within 24 hours of scheduled date
    deadline = new Date(created);
    deadline.setHours(deadline.getHours() + 24);
  } else if (jobType === 'COMPLIANCE') {
    if (priority === 'IMPORTANT') {
      // Important: 4-hour response time
      deadline = new Date(created);
      deadline.setHours(deadline.getHours() + 4);
    } else {
      // Not Important: Same day if before 11 AM, else next day by 7 PM
      const hour = created.getHours();
      if (hour < 11) {
        // Before 11 AM: Complete by 7 PM same day
        deadline = new Date(created);
        deadline.setHours(19, 0, 0, 0);
      } else {
        // After 11 AM: Complete by 7 PM next day
        deadline = new Date(created);
        deadline.setDate(deadline.getDate() + 1);
        deadline.setHours(19, 0, 0, 0);
      }
    }
  }

  return deadline;
};

/**
 * Check if SLA is breached
 */
const isSLABreached = (deadline) => {
  return new Date() > new Date(deadline);
};

/**
 * Get time remaining until SLA deadline
 */
const getTimeRemaining = (deadline) => {
  const now = new Date();
  const deadlineDate = new Date(deadline);
  const diff = deadlineDate - now;

  if (diff <= 0) {
    return { breached: true, minutes: 0, hours: 0 };
  }

  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(minutes / 60);

  return {
    breached: false,
    minutes: minutes % 60,
    hours,
    totalMinutes: minutes,
  };
};

module.exports = {
  calculateSLADeadline,
  isSLABreached,
  getTimeRemaining,
};

