/**
 * JWT Utility Functions
 */

const jwt = require('jsonwebtoken');
const logger = require('./logger');

/**
 * Get JWT secret with fallback
 */
const getJWTSecret = () => {
  const secret = process.env.JWT_SECRET;
  if (!secret) {
    logger.warn('JWT_SECRET not set in environment, using default (NOT SECURE FOR PRODUCTION)');
    return 'hrq-default-jwt-secret-key-change-in-production-minimum-32-characters';
  }
  return secret;
};

/**
 * Generate access token
 */
const generateAccessToken = (payload) => {
  try {
    return jwt.sign(payload, getJWTSecret(), {
      expiresIn: process.env.JWT_EXPIRATION || '2h',
    });
  } catch (error) {
    logger.error('Error generating access token:', error);
    throw error;
  }
};

/**
 * Generate refresh token
 */
const generateRefreshToken = (payload) => {
  try {
    const secret = process.env.REFRESH_TOKEN_SECRET || getJWTSecret();
    return jwt.sign(payload, secret, {
      expiresIn: process.env.REFRESH_TOKEN_EXPIRATION || '7d',
    });
  } catch (error) {
    logger.error('Error generating refresh token:', error);
    throw error;
  }
};

/**
 * Verify access token
 */
const verifyAccessToken = (token) => {
  try {
    return jwt.verify(token, getJWTSecret());
  } catch (error) {
    logger.error('Error verifying access token:', error);
    throw error;
  }
};

/**
 * Verify refresh token
 */
const verifyRefreshToken = (token) => {
  try {
    const secret = process.env.REFRESH_TOKEN_SECRET || getJWTSecret();
    return jwt.verify(token, secret);
  } catch (error) {
    logger.error('Error verifying refresh token:', error);
    throw error;
  }
};

module.exports = {
  generateAccessToken,
  generateRefreshToken,
  verifyAccessToken,
  verifyRefreshToken,
};

