/**
 * Geocoding Utility
 * Uses Google Maps Geocoding API to convert addresses to coordinates
 */

const axios = require('axios');
const logger = require('./logger');

/**
 * Geocode an address to get coordinates
 * @param {string} address - Full address string
 * @returns {Promise<{lat: number, lng: number} | null>}
 */
const geocodeAddress = async (address) => {
  try {
    const apiKey = process.env.GOOGLE_MAPS_API_KEY;
    if (!apiKey) {
      logger.warn('Google Maps API key not configured. Geocoding unavailable.');
      return null;
    }

    if (!address || address.trim() === '') {
      return null;
    }

    const encodedAddress = encodeURIComponent(address);
    const url = `https://maps.googleapis.com/maps/api/geocode/json?address=${encodedAddress}&key=${apiKey}`;

    const response = await axios.get(url, { timeout: 5000 });

    if (response.data.status === 'OK' && response.data.results.length > 0) {
      const location = response.data.results[0].geometry.location;
      return {
        lat: location.lat,
        lng: location.lng,
      };
    } else {
      logger.warn(`Geocoding failed for address: ${address}. Status: ${response.data.status}`);
      return null;
    }
  } catch (error) {
    logger.error('Geocoding error:', error);
    return null;
  }
};

/**
 * Calculate distance between two coordinates using Haversine formula
 * @param {number} lat1 - Latitude of first point
 * @param {number} lon1 - Longitude of first point
 * @param {number} lat2 - Latitude of second point
 * @param {number} lon2 - Longitude of second point
 * @returns {number} Distance in meters
 */
const calculateDistance = (lat1, lon1, lat2, lon2) => {
  if (!lat1 || !lon1 || !lat2 || !lon2) {
    return Infinity;
  }

  const R = 6371e3; // Earth radius in meters
  const φ1 = lat1 * Math.PI / 180;
  const φ2 = lat2 * Math.PI / 180;
  const Δφ = (lat2 - lat1) * Math.PI / 180;
  const Δλ = (lon2 - lon1) * Math.PI / 180;

  const a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
    Math.cos(φ1) * Math.cos(φ2) *
    Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

  return R * c; // Distance in meters
};

/**
 * Resolve job location from coordinates or address
 * @param {number} lat - Latitude
 * @param {number} lng - Longitude
 * @param {string} address - Full address string
 * @returns {Promise<{lat: number, lng: number} | null>}
 */
const resolveJobLocation = async (lat, lng, address) => {
  // If valid coordinates are provided, use them
  if (lat && lng && lat !== 0 && lng !== 0) {
    return { lat, lng };
  }

  // Otherwise, try to geocode the address
  if (address && address.trim() !== '') {
    const coordinates = await geocodeAddress(address);
    return coordinates;
  }

  return null;
};

module.exports = {
  geocodeAddress,
  calculateDistance,
  resolveJobLocation,
};

