/**
 * User Service
 */

const User = require('../models/User.model');
const Employee = require('../models/Employee.model');
const Customer = require('../models/Customer.model');
const UserActivity = require('../models/UserActivity.model');
const { hashPassword } = require('../utils/password');
const logger = require('../utils/logger');

/**
 * Get users with proper isolation
 */
const getUsers = async (filters, tenantId, userRole, userId) => {
  try {
    const query = { tenantId };

    // Role-based filtering
    if (filters.role) {
      query.role = filters.role;
    }

    // Manager can only see assigned agents
    if (userRole === 'MANAGER') {
      const manager = await User.findById(userId);
      if (manager && manager.assignedAgents && manager.assignedAgents.length > 0) {
        query._id = { $in: manager.assignedAgents };
      } else {
        return [];
      }
    }

    // Admin can see all users in their org
    // Super Admin can see all users

    const users = await User.find(query)
      .select('-password')
      .populate('tenantId', 'name')
      .populate({
        path: 'managerId',
        select: 'name email',
        // This will set managerId to null if the referenced document doesn't exist
        strictPopulate: false
      })
      .sort({ createdAt: -1 });

    // Clean up any invalid managerId references
    // Mongoose populate might return an empty object or invalid reference
    users.forEach(user => {
      if (user.managerId) {
        // If managerId is an object but has no _id or name, it's invalid
        if (typeof user.managerId === 'object' && 
            !user.managerId._id && 
            !user.managerId.name && 
            !user.managerId.email) {
          user.managerId = null;
        }
      }
    });

    return users;
  } catch (error) {
    logger.error('Get users error:', error);
    throw error;
  }
};

/**
 * Get user by ID
 */
const getUserById = async (userId, tenantId, userRole, requesterId) => {
  try {
    const user = await User.findOne({ _id: userId, tenantId })
      .select('-password')
      .populate('tenantId', 'name')
      .populate('managerId', 'name email')
      .populate('assignedAgents', 'name email phone');

    if (!user) {
      throw new Error('User not found');
    }

    // Manager can only see assigned agents
    if (userRole === 'MANAGER') {
      const manager = await User.findById(requesterId);
      if (!manager.assignedAgents.includes(userId)) {
        throw new Error('Access denied');
      }
    }

    return user;
  } catch (error) {
    logger.error('Get user by ID error:', error);
    throw error;
  }
};

/**
 * Create user
 */
const createUser = async (userData, tenantId, creatorRole) => {
  try {
    // Only Admin and Super Admin can create users
    if (!['ADMIN', 'SUPER_ADMIN'].includes(creatorRole)) {
      throw new Error('Insufficient permissions');
    }

    // Hash password
    if (userData.password) {
      userData.password = await hashPassword(userData.password);
    }

    const user = new User({
      ...userData,
      tenantId,
    });

    await user.save();
    user.password = undefined;

    return user.populate(['tenantId', 'managerId']);
  } catch (error) {
    logger.error('Create user error:', error);
    throw error;
  }
};

/**
 * Update user
 */
const updateUser = async (userId, userData, tenantId, userRole, requesterId) => {
  try {
    const user = await User.findOne({ _id: userId, tenantId });

    if (!user) {
      throw new Error('User not found');
    }

    // Permission checks
    if (userRole === 'MANAGER') {
      const manager = await User.findById(requesterId);
      if (!manager.assignedAgents.includes(userId)) {
        throw new Error('Access denied');
      }
    }

    // Hash password if provided
    if (userData.password) {
      userData.password = await hashPassword(userData.password);
    }

    Object.assign(user, userData);
    await user.save();
    user.password = undefined;

    return user.populate(['tenantId', 'managerId']);
  } catch (error) {
    logger.error('Update user error:', error);
    throw error;
  }
};

/**
 * Delete user
 */
const deleteUser = async (userId, tenantId, userRole) => {
  try {
    if (!['ADMIN', 'SUPER_ADMIN'].includes(userRole)) {
      throw new Error('Insufficient permissions');
    }

    const user = await User.findOneAndDelete({ _id: userId, tenantId });

    if (!user) {
      throw new Error('User not found');
    }

    return { message: 'User deleted successfully' };
  } catch (error) {
    logger.error('Delete user error:', error);
    throw error;
  }
};

/**
 * Assign agents to manager
 * Supports both User accounts and Employee accounts (by creating User accounts for employees)
 */
const assignAgentsToManager = async (managerId, agentIds, tenantId) => {
  try {
    const manager = await User.findOne({ _id: managerId, tenantId, role: 'MANAGER' });

    if (!manager) {
      throw new Error('Manager not found');
    }

    // Check if agentIds are Employee IDs or User IDs
    // First, try to find them as User accounts
    let userAgents = await User.find({
      _id: { $in: agentIds },
      tenantId,
      role: 'SERVICE_AGENT',
    });

    // If some agents are not found as Users, they might be Employees
    const foundUserIds = userAgents.map(a => a._id.toString());
    const missingIds = agentIds.filter(id => !foundUserIds.includes(id.toString()));

    // For missing IDs, check if they are Employees and create User accounts for them
    if (missingIds.length > 0) {
      const Employee = require('../models/Employee.model');
      const Role = require('../models/Role.model');
      
      // Get Agent role
      const agentRole = await Role.findOne({ 
        tenantId, 
        name: { $regex: /^agent$/i },
        status: 'ACTIVE'
      });

      if (agentRole) {
        const employeeAgents = await Employee.find({
          _id: { $in: missingIds },
          tenantId,
          roleId: agentRole._id,
          status: 'ACTIVE'
        }).populate('roleId', 'name');

        // Create User accounts for Employee agents
        for (const emp of employeeAgents) {
          // Check if User account already exists
          let userAgent = await User.findOne({ email: emp.email, tenantId });
          
          if (!userAgent) {
            // Create User account for employee
            // Employee password is already hashed, but User model will hash it again
            // So we need to set it directly after creation or use a workaround
            const { hashPassword } = require('../utils/password');
            const tempPassword = 'TempPassword123!';
            
            userAgent = new User({
              email: emp.email,
              password: tempPassword, // Temporary, will be replaced
              name: `${emp.firstName} ${emp.lastName}`,
              phone: emp.contactNumber,
              role: 'SERVICE_AGENT',
              status: emp.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE',
              tenantId: tenantId,
            });
            
            // Save first to get _id
            await userAgent.save();
            
            // If employee password is already hashed, use it directly
            // We need to bypass the pre-save hook
            if (emp.password && emp.password.startsWith('$2')) {
              await User.updateOne(
                { _id: userAgent._id },
                { $set: { password: emp.password } }
              );
              // Reload to get updated password
              userAgent = await User.findById(userAgent._id);
            }
            logger.info(`Created User account for Employee agent: ${emp.email}`);
          }
          
          userAgents.push(userAgent);
        }
      }
    }

    // Now assign all user agents to manager
    const finalAgentIds = userAgents.map(a => a._id);
    
    manager.assignedAgents = finalAgentIds;
    await manager.save();

    // Update agents' managerId
    await User.updateMany(
      { _id: { $in: finalAgentIds } },
      { managerId: managerId }
    );

    return manager.populate('assignedAgents', 'name email phone');
  } catch (error) {
    logger.error('Assign agents error:', error);
    throw error;
  }
};

/**
 * Update user location activity
 */
const updateUserLocation = async (userId, userRole, tenantId, locationData) => {
  try {
    // Determine user model based on role
    let userModel = 'User';
    if (userRole === 'EMPLOYEE') {
      userModel = 'Employee';
    } else if (userRole === 'CUSTOMER') {
      userModel = 'Customer';
    }

    // Create activity record
    const activity = new UserActivity({
      userId,
      userModel,
      tenantId,
      activityType: 'LOCATION_UPDATE',
      location: {
        latitude: locationData.latitude,
        longitude: locationData.longitude,
        locationName: locationData.locationName || null,
        address: locationData.locationName || null,
      },
      metadata: {
        timestamp: locationData.timestamp,
      },
      ipAddress: locationData.ipAddress,
      userAgent: locationData.userAgent,
    });

    await activity.save();

    logger.info(`Location updated for user: ${userId}`, {
      userId,
      userRole,
      tenantId,
      latitude: locationData.latitude,
      longitude: locationData.longitude,
      locationName: locationData.locationName,
    });

    return {
      success: true,
      activityId: activity._id,
      location: activity.location,
      timestamp: activity.createdAt,
    };
  } catch (error) {
    logger.error('Update user location error:', error);
    throw error;
  }
};

/**
 * Get manager's assigned agents with Employee details
 */
const getManagerAssignedAgents = async (managerId, tenantId) => {
  try {
    logger.info('🔵 [GET MANAGER AGENTS] Starting', { managerId, tenantId });
    
    // First, try to find manager User account
    let manager = await User.findOne({ _id: managerId, tenantId, role: 'MANAGER' });
    
    // If not found as User, check if it's an Employee with Manager role
    if (!manager) {
      logger.info('📱 [GET MANAGER AGENTS] Manager not found as User, checking Employee collection...');
      const employee = await Employee.findById(managerId).populate('roleId', 'name');
      
      if (!employee) {
        logger.error('❌ [GET MANAGER AGENTS] Employee not found with ID:', managerId);
        throw new Error('Manager not found');
      }
      
      logger.info('✅ [GET MANAGER AGENTS] Employee found', {
        employeeId: employee._id,
        email: employee.email,
        roleId: employee.roleId?._id,
        roleName: employee.roleId?.name,
      });
      
      // Check if Employee has Manager role
      if (employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
        logger.info('✅ [GET MANAGER AGENTS] Employee has Manager role, looking for corresponding User account...');
        // Find corresponding User account with MANAGER role
        manager = await User.findOne({
          email: employee.email,
          tenantId,
          role: 'MANAGER',
        });
        
        if (!manager) {
          logger.warn('⚠️ [GET MANAGER AGENTS] Employee manager found but no corresponding User account');
          logger.warn('   Employee Email:', employee.email);
          logger.warn('   This manager has no assigned agents (agents are assigned via User model)');
          // Return empty array if no User account exists (agents are assigned via User model)
          return [];
        }
        logger.info('✅ [GET MANAGER AGENTS] Found corresponding User account for Employee manager', {
          userId: manager._id,
          email: manager.email,
        });
      } else {
        logger.error('❌ [GET MANAGER AGENTS] Employee does not have Manager role', {
          employeeId: employee._id,
          roleId: employee.roleId?._id,
          roleName: employee.roleId?.name,
        });
        throw new Error('Manager not found - Employee does not have Manager role');
      }
    } else {
      logger.info('✅ [GET MANAGER AGENTS] Manager found as User', {
        userId: manager._id,
        email: manager.email,
      });
    }

    logger.info('📊 [GET MANAGER AGENTS] Manager details', { 
      managerId: manager._id, 
      email: manager.email,
      assignedAgentsCount: manager.assignedAgents?.length || 0 
    });

    if (!manager.assignedAgents || manager.assignedAgents.length === 0) {
      logger.info('ℹ️ [GET MANAGER AGENTS] Manager has no assigned agents');
      return [];
    }

    // Get assigned agent User accounts
    const assignedUserAgents = await User.find({
      _id: { $in: manager.assignedAgents },
      tenantId,
      role: 'SERVICE_AGENT',
    }).select('email name phone');

    logger.info('👥 [GET MANAGER AGENTS] Found assigned User agents', { count: assignedUserAgents.length });

    // Get Role model to validate Agent role
    const Role = require('../models/Role.model');
    
    // Find Agent role from roles collection
    const agentRole = await Role.findOne({
      tenantId,
      name: { $regex: new RegExp('^Agent$', 'i') }, // Case-insensitive exact match
      status: 'ACTIVE'
    });
    
    if (!agentRole) {
      logger.warn('⚠️ [GET MANAGER AGENTS] Agent role not found in roles collection');
    }

    // For each User agent, find corresponding Employee record
    const agentsWithEmployeeData = [];
    
    for (const userAgent of assignedUserAgents) {
      const agentEmail = userAgent.email;
      
      // Find Employee record by email
      const employee = await Employee.findOne({
        email: agentEmail,
        tenantId,
      }).populate('roleId', 'name');

      if (employee) {
        // Validate that employee has Agent role from roles collection
        const employeeRoleName = employee.roleId?.name;
        const isAgent = agentRole && employee.roleId && 
                       employee.roleId._id.toString() === agentRole._id.toString();
        
        if (!isAgent && employeeRoleName && employeeRoleName.toLowerCase() !== 'agent') {
          logger.warn('⚠️ [GET MANAGER AGENTS] Employee does not have Agent role', {
            email: agentEmail,
            roleId: employee.roleId?._id,
            roleName: employeeRoleName
          });
          // Skip this employee if they don't have Agent role
          continue;
        }
        
        // Use Employee data (firstName, lastName, email, contactNumber, employeeId)
        agentsWithEmployeeData.push({
          _id: employee._id,
          firstName: employee.firstName,
          lastName: employee.lastName,
          name: `${employee.firstName} ${employee.lastName}`,
          email: employee.email,
          phone: employee.contactNumber || userAgent.phone || '',
          contactNumber: employee.contactNumber || userAgent.phone || '',
          employeeId: employee.employeeId,
          roleId: employee.roleId,
        });
        logger.info('✅ [GET MANAGER AGENTS] Found Employee for agent', { 
          email: agentEmail, 
          name: `${employee.firstName} ${employee.lastName}`,
          employeeId: employee.employeeId,
          roleName: employeeRoleName
        });
      } else {
        // Fallback to User data if Employee not found
        agentsWithEmployeeData.push({
          _id: userAgent._id,
          name: userAgent.name || agentEmail,
          email: agentEmail,
          phone: userAgent.phone || '',
          contactNumber: userAgent.phone || '',
        });
        logger.warn('⚠️ [GET MANAGER AGENTS] Employee not found for agent, using User data', { email: agentEmail });
      }
    }

    logger.info('✅ [GET MANAGER AGENTS] Returning agents with Employee data', { count: agentsWithEmployeeData.length });
    return agentsWithEmployeeData;
  } catch (error) {
    logger.error('❌ [GET MANAGER AGENTS] Error:', error);
    throw error;
  }
};

module.exports = {
  getUsers,
  getUserById,
  createUser,
  updateUser,
  deleteUser,
  assignAgentsToManager,
  updateUserLocation,
  getManagerAssignedAgents,
};

