/**
 * Tenant Service
 */

const Tenant = require('../models/Tenant.model');
const User = require('../models/User.model');
const Role = require('../models/Role.model');
const { hashPassword } = require('../utils/password');
const logger = require('../utils/logger');

/**
 * Get companies from companies collection (Super Admin only)
 */
const getTenants = async (filters) => {
  try {
    const query = {};

    if (filters.status) {
      query.status = filters.status;
    }
    
    if (filters.isActive !== undefined) {
      query.isActive = filters.isActive;
    }
    
    if (filters.search) {
      query.$or = [
        { name: { $regex: filters.search, $options: 'i' } },
        { email: { $regex: filters.search, $options: 'i' } },
      ];
    }

    // Get companies from companies collection
    const companies = await Tenant.find(query).sort({ createdAt: -1 });
    return companies;
  } catch (error) {
    logger.error('Get companies error:', error);
    throw error;
  }
};

/**
 * Get company by ID from companies collection
 */
const getTenantById = async (tenantId) => {
  try {
    const company = await Tenant.findById(tenantId);

    if (!company) {
      throw new Error('Company not found');
    }

    return company;
  } catch (error) {
    logger.error('Get company by ID error:', error);
    throw error;
  }
};

/**
 * Create company in companies collection (password stored in company)
 */
const createTenant = async (tenantData) => {
  try {
    // Password is required and validated in controller
    if (!tenantData.adminPassword) {
      throw new Error('Password is required');
    }
    
    // Store password directly in company (will be hashed by pre-save hook)
    const companyData = {
      ...tenantData,
      password: tenantData.adminPassword, // Store password in company
    };
    
    // Remove adminPassword from company data (already stored as password)
    delete companyData.adminPassword;
    delete companyData.adminEmail;
    delete companyData.adminName;
    delete companyData.adminPhone;
    
    // Create company (password will be hashed automatically by pre-save hook)
    const company = new Tenant(companyData);
    await company.save();

    logger.info(`Company created: ${company.name}`, { companyId: company._id, email: company.email });

    // Create default Manager and Agent roles for the new company
    try {
      await createDefaultRoles(company._id);
      logger.info(`Default roles created for company: ${company.name}`, { companyId: company._id });
    } catch (roleError) {
      logger.error(`Failed to create default roles for company ${company.name}:`, roleError);
      // Don't fail company creation if role creation fails, but log it
    }

    return company;
  } catch (error) {
    logger.error('Create company error:', error);
    throw error;
  }
};

/**
 * Update company in companies collection
 */
const updateTenant = async (tenantId, tenantData) => {
  try {
    // Extract password if provided
    const { adminPassword, ...companyUpdateData } = tenantData;

    // Update company in companies collection
    const company = await Tenant.findById(tenantId);

    if (!company) {
      throw new Error('Company not found');
    }

    // Update company fields
    Object.assign(company, companyUpdateData);

    // Update password if provided (will be hashed by pre-save hook)
    if (adminPassword && adminPassword.trim() !== '') {
      company.password = adminPassword; // Pre-save hook will hash it
      company.markModified('password');
    }

    await company.save();
    
    if (adminPassword && adminPassword.trim() !== '') {
      logger.info(`Password updated for company: ${company.name}`, { companyId: company._id });
    }

    return company;
  } catch (error) {
    logger.error('Update company error:', error);
    throw error;
  }
};

/**
 * Update white-label configuration
 */
const updateWhiteLabelConfig = async (tenantId, config) => {
  try {
    const tenant = await Tenant.findById(tenantId);

    if (!tenant) {
      throw new Error('Tenant not found');
    }

    tenant.logoUrl = config.logoUrl || tenant.logoUrl;
    tenant.primaryColor = config.primaryColor || tenant.primaryColor;
    tenant.secondaryColor = config.secondaryColor || tenant.secondaryColor;

    await tenant.save();
    return tenant;
  } catch (error) {
    logger.error('Update white-label config error:', error);
    throw error;
  }
};

/**
 * Activate tenant
 */
const activateTenant = async (tenantId) => {
  try {
    const tenant = await Tenant.findByIdAndUpdate(
      tenantId,
      { 
        $set: { 
          isActive: true,
          status: 'ACTIVE'
        } 
      },
      { new: true, runValidators: true }
    );

    if (!tenant) {
      throw new Error('Tenant not found');
    }

    // Activate all users of this tenant
    await User.updateMany(
      { tenantId },
      { status: 'ACTIVE' }
    );

    logger.info(`Tenant activated: ${tenant.name}`, { tenantId });
    return tenant;
  } catch (error) {
    logger.error('Activate tenant error:', error);
    throw error;
  }
};

/**
 * Deactivate tenant
 */
const deactivateTenant = async (tenantId) => {
  try {
    const tenant = await Tenant.findByIdAndUpdate(
      tenantId,
      { 
        $set: { 
          isActive: false,
          status: 'INACTIVE'
        } 
      },
      { new: true, runValidators: true }
    );

    if (!tenant) {
      throw new Error('Tenant not found');
    }

    // Deactivate all users of this tenant
    await User.updateMany(
      { tenantId },
      { status: 'INACTIVE' }
    );

    logger.info(`Tenant deactivated: ${tenant.name}`, { tenantId });
    return tenant;
  } catch (error) {
    logger.error('Deactivate tenant error:', error);
    throw error;
  }
};

/**
 * Delete tenant
 */
const deleteTenant = async (tenantId) => {
  try {
    // Delete all users of this tenant first
    await User.deleteMany({ tenantId });

    const tenant = await Tenant.findByIdAndDelete(tenantId);

    if (!tenant) {
      throw new Error('Tenant not found');
    }

    logger.info(`Tenant deleted: ${tenant.name}`, { tenantId });
    return { message: 'Tenant deleted successfully' };
  } catch (error) {
    logger.error('Delete tenant error:', error);
    throw error;
  }
};

/**
 * Create default Manager and Agent roles for a company
 */
const createDefaultRoles = async (tenantId) => {
  try {
    // Check if roles already exist
    const existingManager = await Role.findOne({ tenantId, name: 'Manager' });
    const existingAgent = await Role.findOne({ tenantId, name: 'Agent' });

    // Create Manager role if it doesn't exist
    if (!existingManager) {
      const managerRole = new Role({
        tenantId,
        name: 'Manager',
        description: 'Default Manager role with full management permissions',
        permissions: {
          reports: {
            create: true,
            view: true,
            editUpdate: true,
            delete: true,
          },
          customerManagement: {
            create: true,
            view: true,
            editUpdate: true,
            delete: true,
          },
          agentsManagement: {
            create: true,
            view: true,
            editUpdate: true,
            delete: true,
          },
          jobsManagement: {
            create: true,
            view: true,
            edit: true,
            update: true,
            assignToAgents: true,
            approveReject: true,
          },
        },
        isDefault: false,
        status: 'ACTIVE',
        createdBy: tenantId,
      });
      await managerRole.save();
      logger.info(`Manager role created for tenant: ${tenantId}`);
    }

    // Create Agent role if it doesn't exist
    if (!existingAgent) {
      const agentRole = new Role({
        tenantId,
        name: 'Agent',
        description: 'Default Agent role with job execution permissions',
        permissions: {
          reports: {
            create: true,
            view: true,
            editUpdate: true,
            delete: false,
          },
          customerManagement: {
            create: false,
            view: true,
            editUpdate: false,
            delete: false,
          },
          agentsManagement: {
            create: false,
            view: false,
            editUpdate: false,
            delete: false,
          },
          jobsManagement: {
            create: false,
            view: true,
            edit: true,
            update: true,
            assignToAgents: false,
            approveReject: false,
          },
        },
        isDefault: false,
        status: 'ACTIVE',
        createdBy: tenantId,
      });
      await agentRole.save();
      logger.info(`Agent role created for tenant: ${tenantId}`);
    }
  } catch (error) {
    logger.error('Create default roles error:', error);
    throw error;
  }
};

/**
 * Create default roles for all existing companies
 */
const createDefaultRolesForAllCompanies = async () => {
  try {
    const companies = await Tenant.find({});
    let createdCount = 0;
    let skippedCount = 0;

    for (const company of companies) {
      try {
        const existingManager = await Role.findOne({ tenantId: company._id, name: 'Manager' });
        const existingAgent = await Role.findOne({ tenantId: company._id, name: 'Agent' });

        if (!existingManager || !existingAgent) {
          await createDefaultRoles(company._id);
          createdCount++;
          logger.info(`Default roles created/updated for company: ${company.name}`);
        } else {
          skippedCount++;
        }
      } catch (error) {
        logger.error(`Failed to create roles for company ${company.name}:`, error);
      }
    }

    return {
      success: true,
      message: `Default roles creation completed. Created: ${createdCount}, Skipped: ${skippedCount}`,
      created: createdCount,
      skipped: skippedCount,
    };
  } catch (error) {
    logger.error('Create default roles for all companies error:', error);
    throw error;
  }
};

module.exports = {
  getTenants,
  getTenantById,
  createTenant,
  updateTenant,
  updateWhiteLabelConfig,
  activateTenant,
  deactivateTenant,
  deleteTenant,
  createDefaultRoles,
  createDefaultRolesForAllCompanies,
};

