/**
 * SLA Service
 */

const Job = require('../models/Job.model');
const Notification = require('../models/Notification.model');
const User = require('../models/User.model');
const { isSLABreached, getTimeRemaining } = require('../utils/sla');
const logger = require('../utils/logger');

/**
 * Monitor SLA and detect breaches
 */
const monitorSLA = async (tenantId) => {
  try {
    const now = new Date();
    
    // Find jobs with approaching or breached SLA
    const jobs = await Job.find({
      tenantId,
      status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS', 'PENDING_APPROVAL'] },
      slaDeadline: { $exists: true },
      slaBreached: false,
    })
      .populate('assignedAgentId', 'name email managerId')
      .populate('managerId', 'name email')
      .populate('customerId', 'name email');

    const breaches = [];
    const warnings = [];

    for (const job of jobs) {
      const timeRemaining = getTimeRemaining(job.slaDeadline);
      
      if (timeRemaining.breached) {
        // SLA breached
        job.slaBreached = true;
        await job.save();

        breaches.push(job);

        // Create escalation notifications
        await escalateSLABreach(job, tenantId);
      } else if (timeRemaining.totalMinutes <= 30 && timeRemaining.totalMinutes > 0) {
        // Warning: Less than 30 minutes remaining
        warnings.push({
          job,
          timeRemaining,
        });

        // Send warning notification if not already sent
        await sendSLAWarning(job, timeRemaining, tenantId);
      }
    }

    return {
      breaches,
      warnings,
      checkedAt: now,
    };
  } catch (error) {
    logger.error('Monitor SLA error:', error);
    throw error;
  }
};

/**
 * Escalate SLA breach
 */
const escalateSLABreach = async (job, tenantId) => {
  try {
    const emailService = require('./email.service');
    
    // Notify agent
    if (job.assignedAgentId) {
      await Notification.create({
        tenantId,
        userId: job.assignedAgentId._id,
        jobId: job._id,
        title: 'SLA BREACHED',
        message: `Job ${job.ticketNumber} has breached SLA deadline`,
        notificationType: 'SLA_BREACH',
      });

      // Send email notification
      try {
        await emailService.sendSLABreachEmail(job.assignedAgentId, job);
      } catch (error) {
        logger.warn('Failed to send SLA breach email to agent:', error);
      }
    }

    // Notify manager
    if (job.managerId) {
      await Notification.create({
        tenantId,
        userId: job.managerId._id,
        jobId: job._id,
        title: 'SLA BREACHED - Escalation Required',
        message: `Job ${job.ticketNumber} assigned to ${job.assignedAgentId?.name || 'agent'} has breached SLA`,
        notificationType: 'SLA_BREACH',
      });

      // Send email notification
      try {
        await emailService.sendSLABreachEmail(job.managerId, job);
      } catch (error) {
        logger.warn('Failed to send SLA breach email to manager:', error);
      }
    }

    // Notify admin
    const admins = await User.find({
      tenantId,
      role: 'ADMIN',
      status: 'ACTIVE',
    });

    for (const admin of admins) {
      await Notification.create({
        tenantId,
        userId: admin._id,
        jobId: job._id,
        title: 'SLA BREACHED - Admin Alert',
        message: `Job ${job.ticketNumber} has breached SLA and requires attention`,
        notificationType: 'SLA_BREACH',
      });

      // Send email notification
      try {
        await emailService.sendSLABreachEmail(admin, job);
      } catch (error) {
        logger.warn('Failed to send SLA breach email to admin:', error);
      }
    }
  } catch (error) {
    logger.error('Escalate SLA breach error:', error);
  }
};

/**
 * Send SLA warning
 */
const sendSLAWarning = async (job, timeRemaining, tenantId) => {
  try {
    // Check if warning already sent (simple check - can be enhanced)
    const existingWarning = await Notification.findOne({
      tenantId,
      jobId: job._id,
      notificationType: 'SLA_BREACH',
      createdAt: { $gte: new Date(Date.now() - 30 * 60000) }, // Last 30 minutes
    });

    if (existingWarning) {
      return; // Warning already sent
    }

    // Notify agent
    if (job.assignedAgentId) {
      await Notification.create({
        tenantId,
        userId: job.assignedAgentId._id,
        jobId: job._id,
        title: `SLA Warning: ${timeRemaining.minutes} minutes remaining`,
        message: `Job ${job.ticketNumber} SLA deadline approaching`,
        notificationType: 'SLA_BREACH',
      });
    }

    // Notify manager
    if (job.managerId) {
      await Notification.create({
        tenantId,
        userId: job.managerId._id,
        jobId: job._id,
        title: `SLA Warning: ${timeRemaining.minutes} minutes remaining`,
        message: `Job ${job.ticketNumber} SLA deadline approaching`,
        notificationType: 'SLA_BREACH',
      });
    }
  } catch (error) {
    logger.error('Send SLA warning error:', error);
  }
};

/**
 * Get SLA breaches
 */
const getSLABreaches = async (tenantId, filters = {}) => {
  try {
    const query = {
      tenantId,
      slaBreached: true,
      status: { $ne: 'CLOSED' },
    };

    if (filters.priority) {
      query.priority = filters.priority;
    }

    const breaches = await Job.find(query)
      .populate('customerId', 'name email phone')
      .populate('assignedAgentId', 'name email')
      .populate('managerId', 'name email')
      .sort({ slaDeadline: 1 });

    return breaches;
  } catch (error) {
    logger.error('Get SLA breaches error:', error);
    throw error;
  }
};

/**
 * Get job SLA details
 */
const getJobSLA = async (jobId, tenantId) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    const timeRemaining = job.slaDeadline ? getTimeRemaining(job.slaDeadline) : null;

    return {
      jobId: job._id,
      ticketNumber: job.ticketNumber,
      slaDeadline: job.slaDeadline,
      slaBreached: job.slaBreached,
      timeRemaining,
      status: job.status,
      priority: job.priority,
    };
  } catch (error) {
    logger.error('Get job SLA error:', error);
    throw error;
  }
};

/**
 * Manually escalate job
 */
const escalateJob = async (jobId, tenantId, escalatedBy) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    // Create escalation notification to admin
    const admins = await User.find({
      tenantId,
      role: 'ADMIN',
      status: 'ACTIVE',
    });

    for (const admin of admins) {
      await Notification.create({
        tenantId,
        userId: admin._id,
        jobId: job._id,
        title: 'Manual Escalation',
        message: `Job ${job.ticketNumber} has been manually escalated by ${escalatedBy}`,
        notificationType: 'SYSTEM',
      });
    }

    return { message: 'Job escalated successfully' };
  } catch (error) {
    logger.error('Escalate job error:', error);
    throw error;
  }
};

module.exports = {
  monitorSLA,
  getSLABreaches,
  getJobSLA,
  escalateJob,
};

