/**
 * Role Service
 * Business logic for role management
 */

const Role = require('../models/Role.model');
const Employee = require('../models/Employee.model');
const logger = require('../utils/logger');

/**
 * Get roles for a company
 */
const getRoles = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };

    if (filters.status) {
      query.status = filters.status;
    }

    if (filters.search) {
      query.$or = [
        { name: { $regex: filters.search, $options: 'i' } },
        { description: { $regex: filters.search, $options: 'i' } },
      ];
    }

    const roles = await Role.find(query).sort({ createdAt: -1 });

    return roles;
  } catch (error) {
    logger.error('Get roles error:', error);
    throw error;
  }
};

/**
 * Get role by ID
 */
const getRoleById = async (roleId, tenantId) => {
  try {
    const role = await Role.findOne({ _id: roleId, tenantId });

    if (!role) {
      throw new Error('Role not found');
    }

    return role;
  } catch (error) {
    logger.error('Get role by ID error:', error);
    throw error;
  }
};

/**
 * Create role
 */
const createRole = async (roleData, tenantId, createdBy) => {
  try {
    // Check if role name already exists for this tenant
    const existingRole = await Role.findOne({ 
      tenantId, 
      name: roleData.name.trim() 
    });
    
    if (existingRole) {
      throw new Error('Role with this name already exists');
    }

    // If this is set as default, unset other defaults
    if (roleData.isDefault) {
      await Role.updateMany(
        { tenantId, isDefault: true },
        { $set: { isDefault: false } }
      );
    }

    const role = new Role({
      ...roleData,
      tenantId,
      createdBy,
      name: roleData.name.trim(),
    });

    await role.save();
    logger.info(`Role created: ${role.name}`, { roleId: role._id, tenantId });

    return role;
  } catch (error) {
    logger.error('Create role error:', error);
    throw error;
  }
};

/**
 * Update role
 */
const updateRole = async (roleId, roleData, tenantId) => {
  try {
    const role = await Role.findOne({ _id: roleId, tenantId });

    if (!role) {
      throw new Error('Role not found');
    }

    // If name is being updated, check for duplicates
    if (roleData.name && roleData.name.trim() !== role.name) {
      const existingRole = await Role.findOne({ 
        tenantId, 
        name: roleData.name.trim(),
        _id: { $ne: roleId }
      });
      if (existingRole) {
        throw new Error('Role with this name already exists');
      }
      roleData.name = roleData.name.trim();
    }

    // If this is set as default, unset other defaults
    if (roleData.isDefault && !role.isDefault) {
      await Role.updateMany(
        { tenantId, isDefault: true, _id: { $ne: roleId } },
        { $set: { isDefault: false } }
      );
    }

    // Update fields
    Object.assign(role, roleData);
    await role.save();

    logger.info(`Role updated: ${role.name}`, { roleId: role._id, tenantId });

    return role;
  } catch (error) {
    logger.error('Update role error:', error);
    throw error;
  }
};

/**
 * Delete role
 */
const deleteRole = async (roleId, tenantId) => {
  try {
    // Check if any employees are using this role
    const employeesUsingRole = await Employee.countDocuments({ 
      roleId, 
      tenantId 
    });

    if (employeesUsingRole > 0) {
      throw new Error(`Cannot delete role. ${employeesUsingRole} employee(s) are using this role.`);
    }

    const role = await Role.findOneAndDelete({ _id: roleId, tenantId });

    if (!role) {
      throw new Error('Role not found');
    }

    logger.info(`Role deleted: ${role.name}`, { roleId: role._id, tenantId });

    return { success: true };
  } catch (error) {
    logger.error('Delete role error:', error);
    throw error;
  }
};

module.exports = {
  getRoles,
  getRoleById,
  createRole,
  updateRole,
  deleteRole,
};

