/**
 * Report Service
 */

const Report = require('../models/Report.model');
const Job = require('../models/Job.model');
const JobWorkDetails = require('../models/JobWorkDetails.model');
const Media = require('../models/Media.model');
const JobApproval = require('../models/JobApproval.model');
const User = require('../models/User.model');
const PDFDocument = require('pdfkit');
const fs = require('fs');
const fsPromises = require('fs').promises;
const path = require('path');
const logger = require('../utils/logger');

/**
 * Generate PDF report for job
 */
const generateReport = async (jobId, tenantId) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId })
      .populate('customerId', 'name email phone address')
      .populate('assignedAgentId', 'name email phone')
      .populate('managerId', 'name email');

    if (!job) {
      throw new Error('Job not found');
    }

    if (job.status !== 'APPROVED') {
      throw new Error('Job must be approved before generating report');
    }

    // Check if report already exists
    let report = await Report.findOne({ jobId });
    if (report) {
      return report;
    }

    // Get work details
    const workDetails = await JobWorkDetails.findOne({ jobId });

    // Get media
    const media = await Media.find({ jobId, tenantId }).sort({ createdAt: 1 });

    // Get approval details
    const approval = await JobApproval.findOne({ jobId });

    // Get tenant for branding
    const Tenant = require('../models/Tenant.model');
    const tenant = await Tenant.findById(tenantId);

    // Generate PDF
    const reportNumber = `RPT-${Date.now()}`;
    const reportDir = 'uploads/reports';
    await fsPromises.mkdir(reportDir, { recursive: true });
    
    const pdfPath = path.join(reportDir, `${reportNumber}.pdf`);
    const pdfFileUrl = `/uploads/reports/${reportNumber}.pdf`;

    await createPDFReport({
      job,
      workDetails,
      media,
      approval,
      tenant,
      pdfPath,
    });

    // Create report record
    report = new Report({
      jobId,
      tenantId,
      reportNumber,
      pdfFileUrl,
      generatedAt: new Date(),
    });

    await report.save();

    // Send email to customer
    try {
      const emailService = require('./email.service');
      await emailService.sendReportEmail(job.customerId, report, job);
    } catch (error) {
      logger.warn('Failed to send report email:', error);
    }

    return report;
  } catch (error) {
    logger.error('Generate report error:', error);
    throw error;
  }
};

/**
 * Create PDF report
 */
const createPDFReport = async ({ job, workDetails, media, approval, tenant, pdfPath }) => {
  return new Promise((resolve, reject) => {
    try {
      const doc = new PDFDocument({ margin: 50 });

      const stream = fs.createWriteStream(pdfPath);
      doc.pipe(stream);

      // Header with tenant branding
      doc.fontSize(20).text(tenant?.name || 'Service Report', { align: 'center' });
      doc.moveDown();
      doc.fontSize(14).text(`Report Number: ${job.ticketNumber}`, { align: 'center' });
      doc.moveDown(2);

      // Job Information
      doc.fontSize(16).text('Job Information', { underline: true });
      doc.moveDown();
      doc.fontSize(12);
      doc.text(`Ticket Number: ${job.ticketNumber}`);
      doc.text(`Service Type: ${job.ticketType}`);
      doc.text(`Status: ${job.status}`);
      doc.text(`Date: ${new Date(job.createdAt).toLocaleDateString()}`);
      doc.moveDown();

      // Customer Information
      doc.fontSize(16).text('Customer Information', { underline: true });
      doc.moveDown();
      doc.fontSize(12);
      doc.text(`Name: ${job.customerName}`);
      doc.text(`Email: ${job.customerEmail || 'N/A'}`);
      doc.text(`Phone: ${job.customerPhone || 'N/A'}`);
      doc.text(`Address: ${job.jobLocationAddress}`);
      doc.moveDown();

      // Agent Information
      if (job.assignedAgentId) {
        doc.fontSize(16).text('Service Agent', { underline: true });
        doc.moveDown();
        doc.fontSize(12);
        doc.text(`Name: ${job.assignedAgentId.name}`);
        doc.text(`Email: ${job.assignedAgentId.email}`);
        doc.moveDown();
      }

      // Work Details
      if (workDetails) {
        doc.fontSize(16).text('Work Details', { underline: true });
        doc.moveDown();
        doc.fontSize(12);
        doc.text(`Work Notes: ${workDetails.workNotes}`);
        doc.text(`Work Completed: ${workDetails.workCompletedFully ? 'Yes' : 'Partial'}`);
        
        if (workDetails.sparesRequired && workDetails.sparesRequired.length > 0) {
          doc.moveDown();
          doc.text('Spares Required:');
          workDetails.sparesRequired.forEach((spare, index) => {
            doc.text(`  ${index + 1}. ${spare.name} - Qty: ${spare.quantity}`);
          });
        }
        doc.moveDown();
      }

      // Approval Information
      if (approval && approval.approvalStatus === 'APPROVED') {
        doc.fontSize(16).text('Approval Information', { underline: true });
        doc.moveDown();
        doc.fontSize(12);
        doc.text(`Approved By: ${job.managerId?.name || 'N/A'}`);
        doc.text(`Approved At: ${approval.approvedAt ? new Date(approval.approvedAt).toLocaleString() : 'N/A'}`);
        if (approval.comments) {
          doc.text(`Comments: ${approval.comments}`);
        }
        doc.moveDown();
      }

      // Media note
      if (media && media.length > 0) {
        doc.fontSize(16).text('Attachments', { underline: true });
        doc.moveDown();
        doc.fontSize(12);
        doc.text(`${media.length} photo(s) attached. See digital copy for full details.`);
        doc.moveDown();
      }

      // Footer
      doc.moveDown(3);
      doc.fontSize(10).text(
        `Generated on ${new Date().toLocaleString()}`,
        { align: 'center' }
      );

      doc.end();

      stream.on('finish', () => {
        resolve();
      });

      stream.on('error', (error) => {
        reject(error);
      });
    } catch (error) {
      reject(error);
    }
  });
};

/**
 * Get report by job ID
 */
const getReportByJobId = async (jobId, tenantId) => {
  try {
    const report = await Report.findOne({ jobId, tenantId })
      .populate('jobId', 'ticketNumber status');

    return report;
  } catch (error) {
    logger.error('Get report by job ID error:', error);
    throw error;
  }
};

/**
 * Get reports
 */
const getReports = async (filters, tenantId, userRole, userId) => {
  try {
    const query = { tenantId };

    // Customer can only see their own reports
    if (userRole === 'CUSTOMER') {
      const jobs = await Job.find({ tenantId, customerId: userId }).select('_id');
      query.jobId = { $in: jobs.map(j => j._id) };
    }

    const reports = await Report.find(query)
      .populate('jobId', 'ticketNumber status customerName')
      .sort({ generatedAt: -1 });

    return reports;
  } catch (error) {
    logger.error('Get reports error:', error);
    throw error;
  }
};

module.exports = {
  generateReport,
  getReportByJobId,
  getReports,
};

