/**
 * Product Service
 * Business logic for product management
 */

const Product = require('../models/Product.model');
const logger = require('../utils/logger');

/**
 * Get products for a company
 */
const getProducts = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };

    if (filters.status) {
      query.status = filters.status;
    }

    if (filters.productType) {
      query.productType = filters.productType;
    }

    if (filters.search) {
      query.$or = [
        { name: { $regex: filters.search, $options: 'i' } },
        { hsnCode: { $regex: filters.search, $options: 'i' } },
        { productId: { $regex: filters.search, $options: 'i' } },
      ];
    }

    const products = await Product.find(query).sort({ createdAt: -1 });

    return products;
  } catch (error) {
    logger.error('Get products error:', error);
    throw error;
  }
};

/**
 * Get product by ID
 */
const getProductById = async (productId, tenantId) => {
  try {
    const product = await Product.findOne({ _id: productId, tenantId });

    if (!product) {
      throw new Error('Product not found');
    }

    return product;
  } catch (error) {
    logger.error('Get product by ID error:', error);
    throw error;
  }
};

/**
 * Create product
 */
const createProduct = async (productData, tenantId, createdBy) => {
  try {
    const product = new Product({
      ...productData,
      tenantId,
      createdBy,
    });

    await product.save();
    logger.info(`Product created: ${product.productId}`, { productId: product._id, tenantId });

    return product;
  } catch (error) {
    logger.error('Create product error:', error);
    throw error;
  }
};

/**
 * Update product
 */
const updateProduct = async (productId, productData, tenantId) => {
  try {
    const product = await Product.findOne({ _id: productId, tenantId });

    if (!product) {
      throw new Error('Product not found');
    }

    Object.assign(product, productData);
    await product.save();

    logger.info(`Product updated: ${product.productId}`, { productId: product._id, tenantId });

    return product;
  } catch (error) {
    logger.error('Update product error:', error);
    throw error;
  }
};

/**
 * Delete product
 */
const deleteProduct = async (productId, tenantId) => {
  try {
    // Check if product is assigned to any customers
    const Customer = require('../models/Customer.model');
    const customersUsingProduct = await Customer.countDocuments({
      tenantId,
      products: productId,
    });

    if (customersUsingProduct > 0) {
      throw new Error(`Cannot delete product. ${customersUsingProduct} customer(s) have this product assigned.`);
    }

    const product = await Product.findOneAndDelete({ _id: productId, tenantId });

    if (!product) {
      throw new Error('Product not found');
    }

    logger.info(`Product deleted: ${product.productId}`, { productId: product._id, tenantId });

    return { success: true };
  } catch (error) {
    logger.error('Delete product error:', error);
    throw error;
  }
};

/**
 * Get products assigned to a customer
 */
const getCustomerProducts = async (customerId, tenantId) => {
  try {
    const Customer = require('../models/Customer.model');
    
    // Find customer and get their assigned products
    const customer = await Customer.findOne({
      _id: customerId,
      tenantId,
    }).populate('products');

    if (!customer) {
      throw new Error('Customer not found');
    }

    // Return only the products assigned to this customer
    const products = customer.products || [];
    
    // Filter to only return active products
    const activeProducts = products.filter(product => 
      product && (product.status === 'ACTIVE' || !product.status)
    );

    return activeProducts;
  } catch (error) {
    logger.error('Get customer products error:', error);
    throw error;
  }
};

module.exports = {
  getProducts,
  getProductById,
  getCustomerProducts,
  createProduct,
  updateProduct,
  deleteProduct,
};

