/**
 * Periodic Maintenance Configuration Service
 */

const PeriodicMaintenanceConfig = require('../models/PeriodicMaintenanceConfig.model');
const Product = require('../models/Product.model');
const logger = require('../utils/logger');

/**
 * Get all configurations
 */
const getConfigs = async (tenantId, query = {}) => {
  try {
    const filter = { tenantId };
    
    if (query.search) {
      filter.name = { $regex: query.search, $options: 'i' };
    }
    
    if (query.isActive !== undefined) {
      filter.isActive = query.isActive === 'true';
    }

    const configs = await PeriodicMaintenanceConfig.find(filter)
      .populate('productId', 'name productId hsnCode productType')
      .populate('createdBy', 'companyName email')
      .sort({ createdAt: -1 });

    return configs;
  } catch (error) {
    logger.error('Get configs error:', error);
    throw error;
  }
};

/**
 * Get configuration by ID
 */
const getConfigById = async (id, tenantId) => {
  try {
    const config = await PeriodicMaintenanceConfig.findOne({ _id: id, tenantId })
      .populate('productId', 'name productId hsnCode productType')
      .populate('createdBy', 'companyName email');

    if (!config) {
      throw new Error('Configuration not found');
    }

    return config;
  } catch (error) {
    logger.error('Get config by ID error:', error);
    throw error;
  }
};

/**
 * Create new configuration
 */
const createConfig = async (configData, tenantId, userId) => {
  try {
    // Check if product already has a configuration
    const existingConfig = await PeriodicMaintenanceConfig.findOne({
      tenantId,
      productId: configData.productId,
    });

    if (existingConfig) {
      throw new Error('This product already has a periodic maintenance configuration');
    }

    // Verify product exists and belongs to tenant
    const product = await Product.findOne({ _id: configData.productId, tenantId });
    if (!product) {
      throw new Error('Product not found');
    }

    // Validate and process fields
    const processedFields = (configData.fields || []).map((field, index) => {
      const processedField = {
        fieldType: field.fieldType,
        fieldName: field.fieldName,
        isRequired: field.isRequired || false,
        order: field.order || index,
      };

      // Process based on field type
      if (field.fieldType === 'CHECKLIST') {
        processedField.checklistItems = (field.checklistItems || []).map(item => ({
          name: item.name,
        }));
      } else if (field.fieldType === 'NUMBER_INPUT') {
        processedField.numberInput = {
          minValue: field.numberInput?.minValue,
          maxValue: field.numberInput?.maxValue,
          rangeName: field.numberInput?.rangeName || '',
          noRange: field.numberInput?.noRange || false,
        };
      } else if (field.fieldType === 'RADIO_BUTTON') {
        processedField.radioButton = {
          groupName: field.radioButton?.groupName || field.fieldName,
          options: (field.radioButton?.options || []).map(option => ({
            label: option.label,
            value: option.value || option.label,
          })),
        };
      }

      return processedField;
    });

    const config = new PeriodicMaintenanceConfig({
      tenantId,
      name: configData.name,
      productId: configData.productId,
      fields: processedFields,
      createdBy: userId,
      isActive: configData.isActive !== undefined ? configData.isActive : true,
    });

    await config.save();

    return await PeriodicMaintenanceConfig.findById(config._id)
      .populate('productId', 'name productId hsnCode productType')
      .populate('createdBy', 'companyName email');
  } catch (error) {
    logger.error('Create config error:', error);
    throw error;
  }
};

/**
 * Update configuration
 */
const updateConfig = async (id, configData, tenantId) => {
  try {
    const config = await PeriodicMaintenanceConfig.findOne({ _id: id, tenantId });
    if (!config) {
      throw new Error('Configuration not found');
    }

    // If productId is being changed, check if new product already has a configuration
    if (configData.productId && configData.productId.toString() !== config.productId.toString()) {
      const existingConfig = await PeriodicMaintenanceConfig.findOne({
        tenantId,
        productId: configData.productId,
        _id: { $ne: id },
      });

      if (existingConfig) {
        throw new Error('The selected product already has a periodic maintenance configuration');
      }

      // Verify new product exists
      const product = await Product.findOne({ _id: configData.productId, tenantId });
      if (!product) {
        throw new Error('Product not found');
      }
    }

    // Update fields if provided
    if (configData.fields) {
      const processedFields = configData.fields.map((field, index) => {
        const processedField = {
          fieldType: field.fieldType,
          fieldName: field.fieldName,
          isRequired: field.isRequired || false,
          order: field.order || index,
        };

        if (field.fieldType === 'CHECKLIST') {
          processedField.checklistItems = (field.checklistItems || []).map(item => ({
            name: item.name,
          }));
        } else if (field.fieldType === 'NUMBER_INPUT') {
          processedField.numberInput = {
            minValue: field.numberInput?.minValue,
            maxValue: field.numberInput?.maxValue,
            rangeName: field.numberInput?.rangeName || '',
            noRange: field.numberInput?.noRange || false,
          };
        } else if (field.fieldType === 'RADIO_BUTTON') {
          processedField.radioButton = {
            groupName: field.radioButton?.groupName || field.fieldName,
            options: (field.radioButton?.options || []).map(option => ({
              label: option.label,
              value: option.value || option.label,
            })),
          };
        }

        return processedField;
      });

      config.fields = processedFields;
    }

    // Update other fields
    if (configData.name !== undefined) config.name = configData.name;
    if (configData.productId !== undefined) config.productId = configData.productId;
    if (configData.isActive !== undefined) config.isActive = configData.isActive;

    await config.save();

    return await PeriodicMaintenanceConfig.findById(config._id)
      .populate('productId', 'name productId hsnCode productType')
      .populate('createdBy', 'companyName email');
  } catch (error) {
    logger.error('Update config error:', error);
    throw error;
  }
};

/**
 * Delete configuration
 */
const deleteConfig = async (id, tenantId) => {
  try {
    const config = await PeriodicMaintenanceConfig.findOne({ _id: id, tenantId });
    if (!config) {
      throw new Error('Configuration not found');
    }

    await PeriodicMaintenanceConfig.deleteOne({ _id: id, tenantId });
    return { message: 'Configuration deleted successfully' };
  } catch (error) {
    logger.error('Delete config error:', error);
    throw error;
  }
};

/**
 * Get available products (products without periodic maintenance configuration)
 */
const getAvailableProducts = async (tenantId, query = {}) => {
  try {
    // Get all product IDs that already have configurations
    const configuredProductIds = await PeriodicMaintenanceConfig.find({ tenantId })
      .distinct('productId');

    // Build filter for products
    const filter = {
      tenantId,
      status: 'ACTIVE',
      _id: { $nin: configuredProductIds }, // Exclude products with existing configurations
    };

    // Add search filter
    if (query.search) {
      filter.$or = [
        { name: { $regex: query.search, $options: 'i' } },
        { productId: { $regex: query.search, $options: 'i' } },
        { hsnCode: { $regex: query.search, $options: 'i' } },
      ];
    }

    const products = await Product.find(filter)
      .select('name productId hsnCode productType description')
      .sort({ name: 1 })
      .limit(query.limit ? parseInt(query.limit) : 100);

    return products;
  } catch (error) {
    logger.error('Get available products error:', error);
    throw error;
  }
};

module.exports = {
  getConfigs,
  getConfigById,
  createConfig,
  updateConfig,
  deleteConfig,
  getAvailableProducts,
};

