/**
 * Periodic Maintenance Service
 */

const mongoose = require('mongoose');
const PeriodicMaintenanceSchedule = require('../models/PeriodicMaintenanceSchedule.model');
const Job = require('../models/Job.model');
const User = require('../models/User.model');
const Notification = require('../models/Notification.model');
const logger = require('../utils/logger');
const { calculateSLADeadline } = require('../utils/sla');

/**
 * Find agent with least schedules for a manager
 */
const findAgentWithLeastSchedules = async (managerId, tenantId) => {
  try {
    // Get manager and their assigned agents
    const manager = await User.findOne({ 
      _id: managerId, 
      tenantId, 
      role: 'MANAGER' 
    }).populate('assignedAgents', '_id name email');

    if (!manager || !manager.assignedAgents || manager.assignedAgents.length === 0) {
      logger.warn('No agents found for manager', { managerId });
      return null;
    }

    const agentIds = manager.assignedAgents.map(agent => agent._id);

    // Convert tenantId to ObjectId if it's a string
    const tenantObjectId = mongoose.Types.ObjectId.isValid(tenantId) 
      ? (typeof tenantId === 'string' ? new mongoose.Types.ObjectId(tenantId) : tenantId)
      : tenantId;

    // Count schedules for each agent
    const scheduleCounts = await PeriodicMaintenanceSchedule.aggregate([
      {
        $match: {
          tenantId: tenantObjectId,
          assignedAgentId: { $in: agentIds },
          isActive: true
        }
      },
      {
        $group: {
          _id: '$assignedAgentId',
          count: { $sum: 1 }
        }
      }
    ]);

    // Create a map of agentId to schedule count
    const countMap = new Map();
    scheduleCounts.forEach(item => {
      countMap.set(item._id.toString(), item.count);
    });

    // Find agent with minimum schedules
    let minCount = Infinity;
    let selectedAgent = null;
    let selectedAgentName = null;

    for (const agent of manager.assignedAgents) {
      const agentIdStr = agent._id.toString();
      const count = countMap.get(agentIdStr) || 0;
      
      if (count < minCount) {
        minCount = count;
        selectedAgent = agent._id;
        selectedAgentName = agent.name || agent.email;
      }
    }

    logger.info('Agent with least schedules selected', {
      managerId,
      selectedAgentId: selectedAgent,
      selectedAgentName,
      scheduleCount: minCount,
      totalAgents: manager.assignedAgents.length
    });

    return {
      agentId: selectedAgent,
      agentName: selectedAgentName
    };
  } catch (error) {
    logger.error('Error finding agent with least schedules:', error);
    throw error;
  }
};

/**
 * Get schedules
 */
const getSchedules = async (tenantId, userRole = null, userId = null, roleName = null) => {
  try {
    let query = { tenantId, isActive: true };
    
    // If manager, filter schedules by managerId
    const isManager = userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager');
    
    if (isManager && userId) {
      logger.info('🔵 [GET SCHEDULES] Manager detected, filtering schedules', { userId, userRole, roleName });
      
      // Try to find User account first (for User model managers)
      let manager = await User.findById(userId);
      
      // If not found or not a User model manager, try to find by email (for Employee managers)
      if (!manager || manager.role !== 'MANAGER') {
        logger.info('📱 [GET SCHEDULES] Manager not found as User, checking Employee collection...');
        const Employee = require('../models/Employee.model');
        const Role = require('../models/Role.model');
        const employee = await Employee.findById(userId).populate('roleId', 'name');
        
        if (employee) {
          logger.info('✅ [GET SCHEDULES] Employee found', { 
            employeeId: employee._id, 
            email: employee.email,
            roleId: employee.roleId?._id,
            roleName: employee.roleId?.name
          });
          
          // Validate that employee has Manager role from roles collection
          const roleNameFromCollection = employee.roleId?.name;
          if (!roleNameFromCollection || roleNameFromCollection.toLowerCase() !== 'manager') {
            logger.warn('⚠️ [GET SCHEDULES] Employee does not have Manager role', {
              employeeId: employee._id,
              roleId: employee.roleId?._id,
              roleName: roleNameFromCollection
            });
            return [];
          }
          
          // Find corresponding User account with MANAGER role by email (case-insensitive)
          // Try exact match first
          manager = await User.findOne({ 
            email: { $regex: new RegExp(`^${employee.email}$`, 'i') },
            tenantId: tenantId,
            role: 'MANAGER' 
          });
          
          if (!manager) {
            // Try case-insensitive match
            manager = await User.findOne({ 
              $expr: {
                $and: [
                  { $eq: [{ $toLower: '$email' }, employee.email.toLowerCase()] },
                  { $eq: ['$tenantId', mongoose.Types.ObjectId.isValid(tenantId) ? (typeof tenantId === 'string' ? new mongoose.Types.ObjectId(tenantId) : tenantId) : tenantId] },
                  { $eq: ['$role', 'MANAGER'] }
                ]
              }
            });
          }
          
          if (manager) {
            logger.info('✅ [GET SCHEDULES] Found corresponding User account for Employee manager', { 
              userId: manager._id,
              userEmail: manager.email,
              employeeEmail: employee.email
            });
          } else {
            logger.warn('⚠️ [GET SCHEDULES] Employee manager found but no corresponding User account', {
              employeeId: employee._id,
              employeeEmail: employee.email,
              tenantId: tenantId
            });
            // Check if there are any schedules with this employee ID as managerId (edge case)
            logger.info('🔍 [GET SCHEDULES] Checking if schedules exist with Employee ID as managerId...');
            const schedulesWithEmployeeId = await PeriodicMaintenanceSchedule.countDocuments({
              tenantId,
              isActive: true,
              managerId: employee._id
            });
            if (schedulesWithEmployeeId > 0) {
              logger.warn('⚠️ [GET SCHEDULES] Found schedules with Employee ID as managerId. These need to be migrated to User ID.');
            }
            return [];
          }
        } else {
          logger.warn('⚠️ [GET SCHEDULES] Employee not found', { userId });
          return [];
        }
      }
      
      if (manager && manager._id) {
        // Only show unassigned schedules (for assignment page)
        // Build query with managerId and unassigned condition
        // Also check for schedules that might have Employee ID as managerId (for backward compatibility)
        const managerObjectId = manager._id;
        
        // Check if there are schedules with this manager's User ID
        const schedulesCount = await PeriodicMaintenanceSchedule.countDocuments({
          tenantId,
          isActive: true,
          managerId: managerObjectId
        });
        
        logger.info('✅ [GET SCHEDULES] Filtering schedules by managerId (unassigned only)', { 
          managerId: managerObjectId,
          managerEmail: manager.email,
          totalSchedulesForManager: schedulesCount
        });
        
        query = {
          tenantId,
          isActive: true,
          managerId: managerObjectId,
          $or: [
            { assignedAgentId: { $exists: false } },
            { assignedAgentId: null }
          ]
        };
      } else {
        logger.warn('⚠️ [GET SCHEDULES] Manager not found, returning empty list');
        return [];
      }
    }
    
    // CRITICAL: Clean up empty strings FIRST before querying using raw MongoDB operations
    // This avoids Mongoose trying to cast empty strings to ObjectId
    try {
      const collection = PeriodicMaintenanceSchedule.collection;
      // Use aggregation to find documents with empty string assignedAgentId
      const emptyStringDocs = await collection.aggregate([
        {
          $match: {
            tenantId: mongoose.Types.ObjectId.isValid(tenantId) 
              ? (typeof tenantId === 'string' ? new mongoose.Types.ObjectId(tenantId) : tenantId)
              : tenantId
          }
        },
        {
          $project: {
            assignedAgentId: 1,
            isString: { $eq: [{ $type: '$assignedAgentId' }, 'string'] },
            isEmpty: { $eq: ['$assignedAgentId', ''] }
          }
        },
        {
          $match: {
            $and: [
              { isString: true },
              { isEmpty: true }
            ]
          }
        }
      ]).toArray();
      
      if (emptyStringDocs.length > 0) {
        const ids = emptyStringDocs.map(d => d._id);
        await collection.updateMany(
          { _id: { $in: ids } },
          { $set: { assignedAgentId: null } }
        );
        logger.info('✅ Cleaned up empty assignedAgentId values', { count: emptyStringDocs.length });
      }
    } catch (cleanupErr) {
      logger.warn('Error cleaning up empty assignedAgentId values:', cleanupErr);
    }
    
    logger.info('📋 [GET SCHEDULES] Final query:', JSON.stringify(query, null, 2));
    
    // Use aggregation pipeline to avoid ObjectId casting errors with empty strings
    // This is safer than using find() when we have empty strings in the database
    let schedules = [];
    
    try {
      const collection = PeriodicMaintenanceSchedule.collection;
      const tenantObjectId = mongoose.Types.ObjectId.isValid(tenantId) 
        ? (typeof tenantId === 'string' ? new mongoose.Types.ObjectId(tenantId) : tenantId)
        : tenantId;
      
      // Build aggregation pipeline
      const pipeline = [
        {
          $match: {
            tenantId: tenantObjectId,
            isActive: true
          }
        }
      ];
      
      // Add manager filter if present
      if (query.managerId) {
        const managerObjectId = mongoose.Types.ObjectId.isValid(query.managerId) 
          ? (typeof query.managerId === 'string' ? new mongoose.Types.ObjectId(query.managerId) : query.managerId)
          : query.managerId;
        
        logger.info('🔍 [GET SCHEDULES] Adding manager filter to aggregation', {
          managerId: managerObjectId.toString(),
          managerIdType: typeof managerObjectId
        });
        
        pipeline.push({
          $match: {
            managerId: managerObjectId
          }
        });
      }
      
      // Filter for unassigned schedules (null, undefined, or empty string)
      // Exclude schedules where assignedAgentId is a valid ObjectId (means it's assigned)
      if (query.$and && query.$and[0] && query.$and[0].$or) {
        pipeline.push({
          $match: {
            $and: [
              {
                $or: [
                  { assignedAgentId: { $exists: false } },
                  { assignedAgentId: null },
                  { 
                    $expr: {
                      $or: [
                        { $eq: [{ $type: '$assignedAgentId' }, 'missing'] },
                        { $eq: [{ $type: '$assignedAgentId' }, 'null'] },
                        { 
                          $and: [
                            { $eq: [{ $type: '$assignedAgentId' }, 'string'] },
                            { $eq: ['$assignedAgentId', ''] }
                          ]
                        }
                      ]
                    }
                  }
                ]
              },
              // Exclude schedules where assignedAgentId is an objectId (assigned)
              {
                $expr: {
                  $not: {
                    $eq: [{ $type: '$assignedAgentId' }, 'objectId']
                  }
                }
              }
            ]
          }
        });
      }
      
      // Sort by createdAt descending
      pipeline.push({
        $sort: { createdAt: -1 }
      });
      
      schedules = await collection.aggregate(pipeline).toArray();
      
      logger.info('📊 [GET SCHEDULES] Aggregation result', {
        schedulesFound: schedules.length,
        pipelineStages: pipeline.length,
        managerId: query.managerId ? query.managerId.toString() : 'none'
      });
      
      // If no schedules found and we have a manager, check if there are any schedules at all
      // This helps debug if the managerId doesn't match
      if (schedules.length === 0 && query.managerId && isManager) {
        const totalSchedulesCount = await collection.countDocuments({
          tenantId: tenantObjectId,
          isActive: true
        });
        
        logger.info('🔍 [GET SCHEDULES] Debugging: No schedules found for manager', {
          managerId: query.managerId.toString(),
          totalSchedulesInTenant: totalSchedulesCount,
          checkingForUnassigned: query.$and && query.$and[0] && query.$and[0].$or ? true : false
        });
        
        // Check if there are schedules with different managerIds (for debugging)
        const managerIdsInSchedules = await collection.distinct('managerId', {
          tenantId: tenantObjectId,
          isActive: true
        });
        
        logger.info('🔍 [GET SCHEDULES] All managerIds in schedules', {
          managerIds: managerIdsInSchedules.map(id => id.toString()),
          lookingFor: query.managerId.toString()
        });
      }
      
      // Keep _id as ObjectId for consistency with Mongoose
      // The _id will be converted to string when needed
      
    } catch (aggError) {
      // Fallback to regular find if aggregation fails
      logger.warn('Aggregation failed, falling back to find:', aggError);
      try {
        // Build a simpler query that won't cause casting errors
        const simpleQuery = {
          tenantId,
          isActive: true
        };
        
        if (query.managerId) {
          simpleQuery.managerId = query.managerId;
        }
        
        // For unassigned, we'll filter after fetching
        let schedulesRaw = await PeriodicMaintenanceSchedule.find(simpleQuery)
          .sort({ createdAt: -1 })
          .lean();
        
        // Filter out empty strings manually
        schedules = schedulesRaw.filter(s => {
          const agentId = s.assignedAgentId;
          return !agentId || agentId === null || 
                 (typeof agentId === 'string' && agentId.trim() !== '' && mongoose.Types.ObjectId.isValid(agentId));
        });
        
        // Further filter for unassigned if needed
        if (query.$and && query.$and[0] && query.$and[0].$or) {
          schedules = schedules.filter(s => 
            !s.assignedAgentId || 
            s.assignedAgentId === null || 
            (typeof s.assignedAgentId === 'string' && s.assignedAgentId.trim() === '')
          );
        }
      } catch (findError) {
        logger.error('Both aggregation and find failed:', findError);
        throw findError;
      }
    }
    
    // Now manually populate all fields to avoid any casting errors
    const customerIds = [...new Set(schedules.map(s => s.customerId).filter(Boolean))];
    const productIds = [...new Set(schedules.map(s => s.productId).filter(Boolean))];
    const managerIds = [...new Set(schedules.map(s => s.managerId).filter(Boolean))];
    const workTypeIds = [...new Set(schedules.map(s => s.workTypeId).filter(Boolean))];
    const agentIds = [...new Set(schedules.map(s => s.assignedAgentId).filter(id => id && mongoose.Types.ObjectId.isValid(id)))];
    
    // Fetch all related data in parallel
    const Customer = require('../models/Customer.model');
    const Product = require('../models/Product.model');
    const PeriodicMaintenanceConfig = require('../models/PeriodicMaintenanceConfig.model');
    
    const [customers, products, managers, workTypes, agents] = await Promise.all([
      customerIds.length > 0 ? Customer.find({ _id: { $in: customerIds } }).select('name email contactNumber').lean() : [],
      productIds.length > 0 ? Product.find({ _id: { $in: productIds } }).select('name productId productType hsnCode').lean() : [],
      managerIds.length > 0 ? User.find({ _id: { $in: managerIds } }).select('name email phone').lean() : [],
      workTypeIds.length > 0 ? PeriodicMaintenanceConfig.find({ _id: { $in: workTypeIds } }).select('name productId').lean() : [],
      agentIds.length > 0 ? User.find({ _id: { $in: agentIds } }).select('name email phone').lean() : []
    ]);
    
    // Create lookup maps
    const customerMap = new Map(customers.map(c => [c._id.toString(), c]));
    const productMap = new Map(products.map(p => [p._id.toString(), p]));
    const managerMap = new Map(managers.map(m => [m._id.toString(), m]));
    const workTypeMap = new Map(workTypes.map(w => [w._id.toString(), w]));
    const agentMap = new Map(agents.map(a => [a._id.toString(), a]));
    
    // Populate schedules with lookup data
    schedules = schedules.map(schedule => {
      // Clean up empty strings - only set to null if it's actually empty
      if (schedule.assignedAgentId === '' || (typeof schedule.assignedAgentId === 'string' && schedule.assignedAgentId.trim() === '')) {
        schedule.assignedAgentId = null;
      } else if (schedule.assignedAgentId && mongoose.Types.ObjectId.isValid(schedule.assignedAgentId)) {
        // If it's a valid ObjectId, try to populate it
        // If agent not found in map, keep the ObjectId (shouldn't happen for unassigned, but be safe)
        const agentData = agentMap.get(schedule.assignedAgentId.toString());
        if (agentData) {
          schedule.assignedAgentId = agentData;
        }
        // If agentData is not found, keep the ObjectId as-is (this means it's assigned)
        // But since we're filtering for unassigned, this shouldn't happen
      } else if (schedule.assignedAgentId === null || schedule.assignedAgentId === undefined) {
        schedule.assignedAgentId = null;
      }
      // If it's already an object (populated), leave it as-is
      
      // Populate other fields
      if (schedule.customerId) {
        schedule.customerId = customerMap.get(schedule.customerId.toString()) || schedule.customerId;
      }
      if (schedule.productId) {
        schedule.productId = productMap.get(schedule.productId.toString()) || schedule.productId;
      }
      if (schedule.managerId) {
        schedule.managerId = managerMap.get(schedule.managerId.toString()) || schedule.managerId;
      }
      if (schedule.workTypeId) {
        schedule.workTypeId = workTypeMap.get(schedule.workTypeId.toString()) || schedule.workTypeId;
      }
      
      return schedule;
    });

    logger.info('✅ [GET SCHEDULES] Found schedules', { count: schedules.length, isManager, userId });
    return schedules;
  } catch (error) {
    logger.error('Get schedules error:', error);
    throw error;
  }
};

/**
 * Create schedule
 */
const createSchedule = async (scheduleData, tenantId) => {
  try {
    // Don't auto-assign agent - let manager assign manually through assignScheduleToAgent endpoint
    // Remove any auto-assigned agent if not explicitly provided
    if (!scheduleData.assignedAgentId) {
      scheduleData.assignedAgentId = null;
      scheduleData.assignedAgentName = null;
    }

    // Normalize periodFrom and periodTo to start of day to avoid timezone issues
    if (scheduleData.periodFrom) {
      const fromDate = new Date(scheduleData.periodFrom);
      const fromYear = fromDate.getFullYear();
      const fromMonth = fromDate.getMonth();
      const fromDay = fromDate.getDate();
      scheduleData.periodFrom = new Date(fromYear, fromMonth, fromDay, 0, 0, 0, 0);
    }
    
    if (scheduleData.periodTo) {
      const toDate = new Date(scheduleData.periodTo);
      const toYear = toDate.getFullYear();
      const toMonth = toDate.getMonth();
      const toDay = toDate.getDate();
      scheduleData.periodTo = new Date(toYear, toMonth, toDay, 23, 59, 59, 999);
    }

    const schedule = new PeriodicMaintenanceSchedule({
      ...scheduleData,
      tenantId,
    });

    // Calculate next generation date if periodFrom is provided
    if (schedule.periodFrom && schedule.frequency) {
      schedule.nextGenerationDate = calculateNextGenerationDate(
        schedule.periodFrom,
        schedule.frequency
      );
    }

    await schedule.save();
    
    // Populate related fields
    await schedule.populate([
      { path: 'customerId', select: 'name email contactNumber' },
      { path: 'productId', select: 'name productId productType hsnCode' },
      { path: 'managerId', select: 'name email phone' },
      { path: 'assignedAgentId', select: 'name email phone' },
      { path: 'workTypeId', select: 'name productId' }
    ]);
    
    return schedule;
  } catch (error) {
    logger.error('Create schedule error:', error);
    throw error;
  }
};

/**
 * Generate jobs from schedule
 */
const generateJobsFromSchedule = async (scheduleId, tenantId) => {
  try {
    const schedule = await PeriodicMaintenanceSchedule.findOne({
      _id: scheduleId,
      tenantId,
      isActive: true,
    });

    if (!schedule) {
      throw new Error('Schedule not found or inactive');
    }

    const jobs = [];
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    // Check if jobs need to be generated
    if (schedule.nextGenerationDate && new Date(schedule.nextGenerationDate) <= today) {
      // Generate jobs for each customer
      for (const customerId of schedule.customerIds) {
        // Get customer details
        const customer = await User.findById(customerId);
        if (!customer || customer.role !== 'CUSTOMER') continue;

        // Assign to primary agent (first in list) or rotate
        const agentIndex = schedule.assignedAgentIds.length > 0 
          ? Math.floor(Math.random() * schedule.assignedAgentIds.length)
          : 0;
        const assignedAgentId = schedule.assignedAgentIds[agentIndex];

        // Create job
        const job = new Job({
          tenantId,
          ticketType: 'PERIODIC',
          customerId,
          customerName: customer.name,
          customerPhone: customer.phone,
          customerEmail: customer.email,
          assignedAgentId,
          managerId: customer.managerId || null,
          status: 'NEW',
          jobLocationLat: customer.address?.coordinates?.lat || 0,
          jobLocationLng: customer.address?.coordinates?.lng || 0,
          jobLocationAddress: `${customer.address?.street || ''}, ${customer.address?.city || ''}`.trim(),
          scheduledDate: schedule.nextGenerationDate,
          scheduleId: schedule._id,
          slaDeadline: calculateSLADeadline('PERIODIC', null, schedule.nextGenerationDate),
        });

        await job.save();

        // Create notification for agent
        const notification = new Notification({
          tenantId,
          userId: assignedAgentId,
          jobId: job._id,
          title: 'New Periodic Maintenance Job',
          message: `New ${schedule.serviceType} job assigned for ${customer.name}`,
          notificationType: 'JOB_ASSIGNED',
        });
        await notification.save();

        jobs.push(job);
      }

      // Update schedule
      schedule.lastGeneratedAt = new Date();
      schedule.nextGenerationDate = calculateNextGenerationDate(
        schedule.nextGenerationDate,
        schedule.recurrenceInterval
      );
      await schedule.save();
    }

    return jobs;
  } catch (error) {
    logger.error('Generate jobs error:', error);
    throw error;
  }
};

/**
 * Calculate next generation date based on recurrence
 */
const calculateNextGenerationDate = (startDate, interval) => {
  const date = new Date(startDate);
  
  switch (interval) {
    case 'DAILY':
      date.setDate(date.getDate() + 1);
      break;
    case 'WEEKLY':
      date.setDate(date.getDate() + 7);
      break;
    case 'MONTHLY':
      date.setMonth(date.getMonth() + 1);
      break;
    case 'QUARTERLY':
      date.setMonth(date.getMonth() + 3);
      break;
    case 'SEMI_ANNUAL':
      date.setMonth(date.getMonth() + 6);
      break;
    case 'YEARLY':
      date.setFullYear(date.getFullYear() + 1);
      break;
    default:
      date.setDate(date.getDate() + 1);
  }

  return date;
};

/**
 * Update schedule
 */
const updateSchedule = async (scheduleId, scheduleData, tenantId) => {
  try {
    const schedule = await PeriodicMaintenanceSchedule.findOneAndUpdate(
      { _id: scheduleId, tenantId },
      { $set: scheduleData },
      { new: true, runValidators: true }
    );

    if (!schedule) {
      throw new Error('Schedule not found');
    }

    // Populate related fields
    await schedule.populate([
      { path: 'customerId', select: 'name email contactNumber' },
      { path: 'productId', select: 'name productId productType hsnCode' },
      { path: 'managerId', select: 'name email phone' },
      { path: 'assignedAgentId', select: 'name email phone' },
      { path: 'workTypeId', select: 'name productId' },
      { path: 'assignedAgentIds', select: 'name email phone' },
      { path: 'customerIds', select: 'name email phone' }
    ]);

    return schedule;
  } catch (error) {
    logger.error('Update schedule error:', error);
    throw error;
  }
};

/**
 * Get schedules for a customer
 */
const getCustomerSchedules = async (tenantId, customerId) => {
  try {
    logger.info('🔵 [GET CUSTOMER SCHEDULES] Starting', { tenantId, customerId });
    
    // Find schedules assigned to this customer
    const schedules = await PeriodicMaintenanceSchedule.find({
      tenantId,
      customerId,
    })
      .populate('customerId', 'name email contactNumber address')
      .populate('productId', 'name productId productType hsnCode')
      .populate('managerId', 'name email phone')
      .populate('assignedAgentId', 'name email phone')
      .populate('workTypeId', 'name productId fields')
      .sort({ createdAt: -1 });
    
    // Get all jobs for these schedules
    const Job = require('../models/Job.model');
    const scheduleIds = schedules.map(s => s._id);
    const jobs = await Job.find({
      tenantId,
      scheduleId: { $in: scheduleIds },
      ticketType: 'PERIODIC',
    })
      .select('_id ticketNumber status scheduledDate createdAt scheduleId assignedAgentId')
      .populate('assignedAgentId', 'name email phone')
      .sort({ scheduledDate: 1 });
    
    // Add jobs and scheduled dates to each schedule
    const schedulesWithJobs = schedules.map(schedule => {
      const scheduleObj = schedule.toObject();
      const scheduleJobs = jobs.filter(job => 
        job.scheduleId && job.scheduleId.toString() === schedule._id.toString()
      );
      
      scheduleObj.periodicJobs = scheduleJobs.map(job => ({
        id: job._id,
        ticketNumber: job.ticketNumber,
        status: job.status,
        scheduledDate: job.scheduledDate,
        createdAt: job.createdAt,
        assignedAgentId: job.assignedAgentId,
      }));
      
      // Calculate all scheduled dates
      if (schedule.periodFrom && schedule.periodTo && schedule.frequency) {
        const allDates = calculateScheduledDates(
          schedule.periodFrom,
          schedule.periodTo,
          schedule.frequency
        );
        
        scheduleObj.allScheduledDates = allDates.map(dateStr => {
          const date = new Date(dateStr);
          date.setHours(0, 0, 0, 0);
          
          const jobForDate = scheduleJobs.find(job => {
            const jobDate = new Date(job.scheduledDate);
            jobDate.setHours(0, 0, 0, 0);
            return jobDate.getTime() === date.getTime();
          });
          
          return {
            date: dateStr,
            hasJob: !!jobForDate,
            jobId: jobForDate?._id,
            ticketNumber: jobForDate?.ticketNumber,
            status: jobForDate?.status || 'PENDING',
            assignedAgentId: jobForDate?.assignedAgentId || schedule.assignedAgentId,
          };
        });
      }
      
      return scheduleObj;
    });
    
    logger.info('✅ [GET CUSTOMER SCHEDULES] Found schedules', { count: schedulesWithJobs.length });
    return schedulesWithJobs;
  } catch (error) {
    logger.error('Get customer schedules error:', error);
    throw error;
  }
};

/**
 * Get schedules assigned to an agent
 */
const getAgentSchedules = async (tenantId, userId, userRole, roleName = null) => {
  try {
    logger.info('🔵 [GET AGENT SCHEDULES] Starting', { userId, userRole, roleName });
    
    // Build list of agent IDs (both User and Employee IDs)
    const agentIds = [userId.toString()];
    
    // If agent is an Employee, also check for corresponding User account
    if (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent') {
      const Employee = require('../models/Employee.model');
      const employee = await Employee.findById(userId);
      if (employee) {
        const agentUser = await User.findOne({
          email: employee.email,
          tenantId,
          role: 'SERVICE_AGENT',
        });
        if (agentUser) {
          agentIds.push(agentUser._id.toString());
          logger.info('✅ [GET AGENT SCHEDULES] Found User account for Employee agent', { userId: agentUser._id });
        }
      }
    } else if (userRole === 'SERVICE_AGENT') {
      // If agent is a User, also check for corresponding Employee account
      const agentUser = await User.findById(userId);
      if (agentUser) {
        const Employee = require('../models/Employee.model');
        const employee = await Employee.findOne({
          email: agentUser.email,
          tenantId,
        }).populate('roleId', 'name');
        if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
          agentIds.push(employee._id.toString());
          logger.info('✅ [GET AGENT SCHEDULES] Found Employee account for User agent', { employeeId: employee._id });
        }
      }
    }
    
    // Find schedules assigned to this agent (check both User and Employee IDs)
    // Show both active and inactive schedules so agent can see all their assignments
    const schedules = await PeriodicMaintenanceSchedule.find({
      tenantId,
      assignedAgentId: { $in: agentIds },
      // Don't filter by isActive - show all assigned schedules
    })
      .populate('customerId', 'name email contactNumber address')
      .populate('productId', 'name productId productType hsnCode')
      .populate('managerId', 'name email phone')
      .populate('assignedAgentId', 'name email phone')
      .populate('workTypeId', 'name productId fields')
      .sort({ nextGenerationDate: 1 }); // Sort by next generation date
    
    logger.info('✅ [GET AGENT SCHEDULES] Found schedules', { count: schedules.length, agentIds });
    return schedules;
  } catch (error) {
    logger.error('Get agent schedules error:', error);
    throw error;
  }
};

/**
 * Get schedule by ID with associated jobs
 */
const getScheduleById = async (scheduleId, tenantId) => {
  try {
    const schedule = await PeriodicMaintenanceSchedule.findOne({
      _id: scheduleId,
      tenantId,
    })
      .populate('customerId', 'name email contactNumber address')
      .populate('productId', 'name productId productType hsnCode')
      .populate('managerId', 'name email phone')
      .populate('assignedAgentId', 'name email phone')
      .populate('workTypeId', 'name productId fields');

    if (!schedule) {
      throw new Error('Schedule not found');
    }

    // Get all jobs for this schedule
    const Job = require('../models/Job.model');
    const jobs = await Job.find({
      tenantId,
      scheduleId: schedule._id,
      ticketType: 'PERIODIC',
    })
      .select('_id ticketNumber status scheduledDate createdAt assignedAgentId')
      .populate('assignedAgentId', 'name email phone')
      .sort({ scheduledDate: 1 });

    // Convert schedule to object and add jobs
    const scheduleObj = schedule.toObject();
    scheduleObj.periodicJobs = jobs.map(job => ({
      id: job._id,
      ticketNumber: job.ticketNumber,
      status: job.status,
      scheduledDate: job.scheduledDate,
      createdAt: job.createdAt,
      assignedAgentId: job.assignedAgentId,
    }));

    // Calculate all scheduled dates based on frequency
    if (schedule.periodFrom && schedule.periodTo && schedule.frequency) {
      const allDates = calculateScheduledDates(
        schedule.periodFrom,
        schedule.periodTo,
        schedule.frequency
      );
      
      // Map each scheduled date with its job info (if exists)
      scheduleObj.allScheduledDates = allDates.map(dateStr => {
        const date = new Date(dateStr);
        date.setHours(0, 0, 0, 0);
        
        // Find job for this date
        const jobForDate = jobs.find(job => {
          const jobDate = new Date(job.scheduledDate);
          jobDate.setHours(0, 0, 0, 0);
          return jobDate.getTime() === date.getTime();
        });
        
        return {
          date: dateStr,
          hasJob: !!jobForDate,
          jobId: jobForDate?._id,
          ticketNumber: jobForDate?.ticketNumber,
          status: jobForDate?.status,
          assignedAgentId: jobForDate?.assignedAgentId || schedule.assignedAgentId,
        };
      });
    }

    return scheduleObj;
  } catch (error) {
    logger.error('Get schedule by ID error:', error);
    throw error;
  }
};

/**
 * Delete schedule
 */
const deleteSchedule = async (scheduleId, tenantId) => {
  try {
    const schedule = await PeriodicMaintenanceSchedule.findOneAndUpdate(
      { _id: scheduleId, tenantId },
      { isActive: false },
      { new: true }
    );

    if (!schedule) {
      throw new Error('Schedule not found');
    }

    return { message: 'Schedule deleted successfully' };
  } catch (error) {
    logger.error('Delete schedule error:', error);
    throw error;
  }
};

/**
 * Calculate scheduled dates based on frequency and date range
 */
const calculateScheduledDates = (periodFrom, periodTo, frequency) => {
  const dates = [];
  // Create new Date objects to avoid mutating the original dates
  const fromDate = new Date(periodFrom);
  const toDate = new Date(periodTo);
  
  // Normalize to start of day in local timezone to preserve the exact date
  // Extract year, month, day to avoid timezone conversion issues
  const fromYear = fromDate.getFullYear();
  const fromMonth = fromDate.getMonth();
  const fromDay = fromDate.getDate();
  const normalizedFromDate = new Date(fromYear, fromMonth, fromDay, 0, 0, 0, 0);
  
  const toYear = toDate.getFullYear();
  const toMonth = toDate.getMonth();
  const toDay = toDate.getDate();
  const normalizedToDate = new Date(toYear, toMonth, toDay, 23, 59, 59, 999);

  // Start from the exact periodFrom date (not one day before)
  let currentDate = new Date(normalizedFromDate);

  switch (frequency) {
    case 'DAILY':
      while (currentDate <= normalizedToDate) {
        dates.push(new Date(currentDate).toISOString());
        currentDate.setDate(currentDate.getDate() + 1);
      }
      break;

    case 'WEEKLY':
      while (currentDate <= normalizedToDate) {
        dates.push(new Date(currentDate).toISOString());
        currentDate.setDate(currentDate.getDate() + 7);
      }
      break;

    case 'MONTHLY':
      while (currentDate <= normalizedToDate) {
        dates.push(new Date(currentDate).toISOString());
        currentDate.setMonth(currentDate.getMonth() + 1);
      }
      break;

    case 'QUARTERLY':
      while (currentDate <= normalizedToDate) {
        dates.push(new Date(currentDate).toISOString());
        currentDate.setMonth(currentDate.getMonth() + 3);
      }
      break;

    case 'YEARLY':
      while (currentDate <= normalizedToDate) {
        dates.push(new Date(currentDate).toISOString());
        currentDate.setFullYear(currentDate.getFullYear() + 1);
      }
      break;

    default:
      // Default to daily
      while (currentDate <= normalizedToDate) {
        dates.push(new Date(currentDate).toISOString());
        currentDate.setDate(currentDate.getDate() + 1);
      }
  }

  return dates;
};

/**
 * Generate periodic jobs for a schedule based on frequency and date range
 */
const generatePeriodicJobsForSchedule = async (schedule, tenantId) => {
  try {
    const Job = require('../models/Job.model');
    const Customer = require('../models/Customer.model');
    const Notification = require('../models/Notification.model');
    // calculateSLADeadline is already imported at the top of the file
    
    if (!schedule.assignedAgentId || !schedule.customerId || !schedule.periodFrom || !schedule.periodTo) {
      logger.warn('⚠️ [GENERATE JOBS] Schedule missing required fields', {
        scheduleId: schedule._id,
        hasAgent: !!schedule.assignedAgentId,
        hasCustomer: !!schedule.customerId,
        hasPeriodFrom: !!schedule.periodFrom,
        hasPeriodTo: !!schedule.periodTo,
      });
      return;
    }

    // Get customer details - schedule uses Customer model
    let customer = schedule.customerId;
    // If customerId is already populated (object), use it; otherwise fetch it
    if (!customer || (typeof customer === 'object' && !customer._id)) {
      customer = await Customer.findById(schedule.customerId);
    } else if (typeof customer === 'string' || (customer && customer._id)) {
      // If it's an ID string or has _id, fetch the customer
      const customerId = typeof customer === 'string' ? customer : customer._id;
      customer = await Customer.findById(customerId);
    }
    
    if (!customer) {
      logger.warn('⚠️ [GENERATE JOBS] Customer not found', { customerId: schedule.customerId });
      return;
    }

    // Get address and coordinates
    let jobLocationLat = 0;
    let jobLocationLng = 0;
    let jobLocationAddress = '';
    
    if (schedule.address) {
      if (schedule.address.native) {
        jobLocationAddress = schedule.address.native;
      } else {
        const parts = [];
        if (schedule.address.doorNumber) parts.push(`Door: ${schedule.address.doorNumber}`);
        if (schedule.address.floor) parts.push(`Floor: ${schedule.address.floor}`);
        if (schedule.address.apartmentBlockName) parts.push(schedule.address.apartmentBlockName);
        if (schedule.address.district) parts.push(schedule.address.district);
        if (schedule.address.state) parts.push(schedule.address.state);
        if (schedule.address.pincode) parts.push(`PIN: ${schedule.address.pincode}`);
        jobLocationAddress = parts.join(', ');
      }
    }
    
    if (customer.address && customer.address.coordinates) {
      jobLocationLat = customer.address.coordinates.lat || 0;
      jobLocationLng = customer.address.coordinates.lng || 0;
    }
    
    if (!jobLocationAddress && customer.address && customer.address.fullAddress) {
      jobLocationAddress = customer.address.fullAddress;
    }

    // Calculate scheduled dates based on frequency
    const scheduledDates = calculateScheduledDates(
      schedule.periodFrom,
      schedule.periodTo,
      schedule.frequency
    );

    logger.info('📅 [GENERATE JOBS] Generating jobs for schedule', {
      scheduleId: schedule._id,
      frequency: schedule.frequency,
      periodFrom: schedule.periodFrom,
      periodTo: schedule.periodTo,
      datesCount: scheduledDates.length,
    });

    const jobs = [];
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    for (const scheduledDateStr of scheduledDates) {
      // scheduledDates are ISO strings, convert to Date
      const scheduleDate = new Date(scheduledDateStr);
      scheduleDate.setHours(0, 0, 0, 0);
      
      // Only create jobs for dates that are today or in the future
      if (scheduleDate < today) {
        continue; // Skip past dates
      }

      // Check if job already exists for this schedule and date
      // Use date range query to handle timezone issues
      const startOfDay = new Date(scheduleDate);
      startOfDay.setHours(0, 0, 0, 0);
      const endOfDay = new Date(scheduleDate);
      endOfDay.setHours(23, 59, 59, 999);

      const existingJob = await Job.findOne({
        tenantId,
        scheduleId: schedule._id,
        scheduledDate: {
          $gte: startOfDay,
          $lte: endOfDay,
        },
        ticketType: 'PERIODIC',
      });

      if (existingJob) {
        logger.info('⏭️ [GENERATE JOBS] Job already exists', {
          scheduleId: schedule._id,
          scheduledDate: scheduleDate,
        });
        continue;
      }

      // Create job
      const job = new Job({
        tenantId,
        ticketType: 'PERIODIC',
        customerId: customer._id,
        customerName: schedule.customerName || customer.name,
        customerPhone: customer.contactNumber || customer.phone,
        customerEmail: customer.email,
        assignedAgentId: schedule.assignedAgentId,
        managerId: schedule.managerId,
        status: 'NEW',
        jobLocationLat: jobLocationLat,
        jobLocationLng: jobLocationLng,
        jobLocationAddress: jobLocationAddress || 'Address not available',
        scheduledDate: scheduleDate,
        scheduleId: schedule._id,
        slaDeadline: calculateSLADeadline('PERIODIC', null, scheduleDate),
      });

      await job.save();
      jobs.push(job);

      // Create notification for agent
      try {
        const notification = new Notification({
          tenantId,
          userId: schedule.assignedAgentId,
          jobId: job._id,
          title: 'New Periodic Maintenance Job',
          message: `New ${schedule.workType || 'periodic'} job scheduled for ${schedule.customerName}`,
          notificationType: 'JOB_ASSIGNED',
        });
        await notification.save();
      } catch (notifError) {
        logger.warn('⚠️ [GENERATE JOBS] Failed to create notification', { error: notifError });
      }
    }

    logger.info('✅ [GENERATE JOBS] Generated jobs for schedule', {
      scheduleId: schedule._id,
      jobsCreated: jobs.length,
      totalDates: scheduledDates.length,
    });

    return jobs;
  } catch (error) {
    logger.error('❌ [GENERATE JOBS] Error generating jobs for schedule:', error);
    throw error;
  }
};

/**
 * Assign schedule to agent (by manager)
 * NOTE: This function only updates the specific schedule identified by scheduleId.
 * It does NOT affect other schedules. Each schedule is updated independently.
 */
const assignScheduleToAgent = async (scheduleId, agentId, agentName, tenantId) => {
  try {
    // Find and update ONLY the specific schedule - does not affect other schedules
    const schedule = await PeriodicMaintenanceSchedule.findOne({
      _id: scheduleId,
      tenantId,
    });

    if (!schedule) {
      throw new Error('Schedule not found');
    }

    // Handle both Employee ID and User ID
    // If agentId is an Employee ID, find the corresponding User account
    let finalAgentId = agentId;
    let finalAgentName = agentName;

    // First, try to find as User
    let userAgent = await User.findOne({
      _id: agentId,
      tenantId,
      role: 'SERVICE_AGENT',
    });

    // If not found, try as Employee and find corresponding User
    if (!userAgent) {
      const Employee = require('../models/Employee.model');
      const employee = await Employee.findById(agentId);
      
      if (employee) {
        logger.info('📱 [ASSIGN SCHEDULE] Agent ID is Employee ID, finding corresponding User account...');
        // Find corresponding User account by email
        userAgent = await User.findOne({
          email: employee.email,
          tenantId,
          role: 'SERVICE_AGENT',
        });
        
        if (userAgent) {
          finalAgentId = userAgent._id;
          finalAgentName = userAgent.name || `${employee.firstName} ${employee.lastName}`;
          logger.info('✅ [ASSIGN SCHEDULE] Found corresponding User account', {
            employeeId: agentId,
            userId: finalAgentId,
            name: finalAgentName,
          });
        } else {
          logger.warn('⚠️ [ASSIGN SCHEDULE] Employee found but no corresponding User account');
          // Still use Employee ID, but this might cause issues with population
          finalAgentId = agentId;
          finalAgentName = finalAgentName || `${employee.firstName} ${employee.lastName}`;
        }
      }
    } else {
      finalAgentName = userAgent.name || finalAgentName;
    }

    // Update assigned agent
    schedule.assignedAgentId = finalAgentId;
    schedule.assignedAgentName = finalAgentName;

    await schedule.save();

    // Generate periodic jobs based on frequency and date range
    await generatePeriodicJobsForSchedule(schedule, tenantId);

    // Populate related fields
    await schedule.populate([
      { path: 'customerId', select: 'name email contactNumber' },
      { path: 'productId', select: 'name productId productType hsnCode' },
      { path: 'managerId', select: 'name email phone' },
      { path: 'assignedAgentId', select: 'name email phone' },
      { path: 'workTypeId', select: 'name productId' },
    ]);

    logger.info('✅ [ASSIGN SCHEDULE] Schedule assigned to agent', {
      scheduleId,
      originalAgentId: agentId,
      finalAgentId,
      agentName: finalAgentName,
    });

    return schedule;
  } catch (error) {
    logger.error('❌ [ASSIGN SCHEDULE] Assign schedule to agent error:', error);
    throw error;
  }
};

/**
 * Assign agent to a specific date in schedule
 */
const assignAgentToScheduleDate = async (scheduleId, date, agentId, agentName, tenantId) => {
  try {
    const schedule = await PeriodicMaintenanceSchedule.findOne({
      _id: scheduleId,
      tenantId,
    });

    if (!schedule) {
      throw new Error('Schedule not found');
    }

    // Handle both Employee ID and User ID
    let finalAgentId = agentId;
    let finalAgentName = agentName;

    // First, try to find as User
    let userAgent = await User.findOne({
      _id: agentId,
      tenantId,
      role: 'SERVICE_AGENT',
    });

    // If not found, try as Employee and find corresponding User
    if (!userAgent) {
      const Employee = require('../models/Employee.model');
      const employee = await Employee.findById(agentId);
      
      if (employee) {
        logger.info('📱 [ASSIGN DATE] Agent ID is Employee ID, finding corresponding User account...');
        userAgent = await User.findOne({
          email: employee.email,
          tenantId,
          role: 'SERVICE_AGENT',
        });
        
        if (userAgent) {
          finalAgentId = userAgent._id;
          finalAgentName = userAgent.name || `${employee.firstName} ${employee.lastName}`;
        } else {
          finalAgentId = agentId;
          finalAgentName = finalAgentName || `${employee.firstName} ${employee.lastName}`;
        }
      }
    } else {
      finalAgentName = userAgent.name || finalAgentName;
    }

    // Normalize date to start of day for comparison
    // Handle both ISO string format (YYYY-MM-DD) and Date objects
    let targetDate;
    if (typeof date === 'string') {
      // If it's a string in YYYY-MM-DD format, parse it correctly
      // Use local timezone to avoid timezone issues
      const [year, month, day] = date.split('-').map(Number);
      targetDate = new Date(year, month - 1, day);
    } else {
      targetDate = new Date(date);
    }
    targetDate.setHours(0, 0, 0, 0);
    
    logger.info('📅 [ASSIGN DATE] Parsing date', {
      inputDate: date,
      parsedDate: targetDate.toISOString(),
      dateString: targetDate.toDateString()
    });

    // Initialize dateAgentAssignments if it doesn't exist
    if (!schedule.dateAgentAssignments) {
      schedule.dateAgentAssignments = [];
    }

    // Find existing assignment for this date
    const existingIndex = schedule.dateAgentAssignments.findIndex(assignment => {
      if (!assignment || !assignment.date) return false;
      const assignmentDate = new Date(assignment.date);
      assignmentDate.setHours(0, 0, 0, 0);
      
      // Compare dates using multiple methods for reliability
      const timeMatch = assignmentDate.getTime() === targetDate.getTime();
      const dateStringMatch = assignmentDate.toDateString() === targetDate.toDateString();
      const isoDateMatch = assignmentDate.toISOString().split('T')[0] === targetDate.toISOString().split('T')[0];
      
      const match = timeMatch || dateStringMatch || isoDateMatch;
      
      if (match) {
        logger.info('✅ [ASSIGN DATE] Found existing assignment', {
          assignmentDate: assignmentDate.toISOString(),
          targetDate: targetDate.toISOString(),
          existingAgent: assignment.agentName
        });
      }
      
      return match;
    });

    if (existingIndex >= 0) {
      // Update existing assignment
      // Ensure date is normalized to UTC midnight
      const normalizedDate = new Date(targetDate);
      normalizedDate.setUTCHours(0, 0, 0, 0);
      
      logger.info('🔄 [ASSIGN DATE] Updating existing assignment', {
        index: existingIndex,
        oldAgent: schedule.dateAgentAssignments[existingIndex].agentName,
        newAgent: finalAgentName,
        date: normalizedDate.toISOString()
      });
      schedule.dateAgentAssignments[existingIndex].agentId = finalAgentId;
      schedule.dateAgentAssignments[existingIndex].agentName = finalAgentName;
      schedule.dateAgentAssignments[existingIndex].date = normalizedDate; // Update date to ensure consistency
    } else {
      // Add new assignment
      // Ensure date is stored as a Date object at midnight UTC to avoid timezone issues
      const normalizedDate = new Date(targetDate);
      normalizedDate.setUTCHours(0, 0, 0, 0);
      
      logger.info('➕ [ASSIGN DATE] Adding new assignment', {
        date: normalizedDate.toISOString(),
        dateString: normalizedDate.toDateString(),
        agentId: finalAgentId,
        agentName: finalAgentName,
        currentAssignmentsCount: schedule.dateAgentAssignments.length
      });
      schedule.dateAgentAssignments.push({
        date: normalizedDate,
        agentId: finalAgentId,
        agentName: finalAgentName,
      });
    }

    // Mark the array as modified to ensure Mongoose saves it
    schedule.markModified('dateAgentAssignments');
    
    // Save and verify
    const savedSchedule = await schedule.save();
    
    logger.info('💾 [ASSIGN DATE] Schedule saved', {
      scheduleId: savedSchedule._id,
      dateAgentAssignmentsCount: savedSchedule.dateAgentAssignments?.length || 0,
      lastAssignment: savedSchedule.dateAgentAssignments && savedSchedule.dateAgentAssignments.length > 0 
        ? savedSchedule.dateAgentAssignments[savedSchedule.dateAgentAssignments.length - 1]
        : null
    });

    // Reload schedule to ensure we have the latest data including dateAgentAssignments
    const updatedSchedule = await PeriodicMaintenanceSchedule.findById(scheduleId)
      .populate([
        { path: 'customerId', select: 'name email contactNumber' },
        { path: 'productId', select: 'name productId productType hsnCode' },
        { path: 'managerId', select: 'name email phone' },
        { path: 'assignedAgentId', select: 'name email phone' },
        { path: 'workTypeId', select: 'name productId' },
      ]);

    // Convert to plain object to ensure dateAgentAssignments is included in response
    const scheduleObj = updatedSchedule ? updatedSchedule.toObject() : savedSchedule.toObject();

    logger.info('✅ [ASSIGN DATE] Agent assigned to specific date', {
      scheduleId,
      date: targetDate.toISOString(),
      agentId: finalAgentId,
      agentName: finalAgentName,
      dateAgentAssignmentsCount: scheduleObj.dateAgentAssignments?.length || 0,
      dateAgentAssignments: scheduleObj.dateAgentAssignments
    });

    return scheduleObj;
  } catch (error) {
    logger.error('❌ [ASSIGN DATE] Assign agent to date error:', error);
    throw error;
  }
};

module.exports = {
  findAgentWithLeastSchedules,
  getSchedules,
  getAgentSchedules,
  getCustomerSchedules,
  getScheduleById,
  createSchedule,
  generateJobsFromSchedule,
  generatePeriodicJobsForSchedule,
  calculateScheduledDates,
  updateSchedule,
  deleteSchedule,
  assignScheduleToAgent,
  assignAgentToScheduleDate,
  calculateNextGenerationDate,
};

