/**
 * Notification Service
 */

const Notification = require('../models/Notification.model');
const logger = require('../utils/logger');

/**
 * Get notifications for user
 */
const getNotifications = async (userId, tenantId, filters = {}) => {
  try {
    const query = {
      tenantId,
      userId,
    };

    if (filters.isRead !== undefined) {
      query.isRead = filters.isRead;
    }

    if (filters.notificationType) {
      query.notificationType = filters.notificationType;
    }

    const notifications = await Notification.find(query)
      .populate('jobId', 'ticketNumber status')
      .sort({ createdAt: -1 })
      .limit(filters.limit || 50);

    return notifications;
  } catch (error) {
    logger.error('Get notifications error:', error);
    throw error;
  }
};

/**
 * Mark notification as read
 */
const markNotificationRead = async (notificationId, userId, tenantId) => {
  try {
    const notification = await Notification.findOneAndUpdate(
      {
        _id: notificationId,
        userId,
        tenantId,
      },
      {
        isRead: true,
        readAt: new Date(),
      },
      { new: true }
    );

    if (!notification) {
      throw new Error('Notification not found');
    }

    return notification;
  } catch (error) {
    logger.error('Mark notification read error:', error);
    throw error;
  }
};

/**
 * Mark all notifications as read
 */
const markAllNotificationsRead = async (userId, tenantId) => {
  try {
    const result = await Notification.updateMany(
      {
        userId,
        tenantId,
        isRead: false,
      },
      {
        isRead: true,
        readAt: new Date(),
      }
    );

    return { count: result.modifiedCount };
  } catch (error) {
    logger.error('Mark all notifications read error:', error);
    throw error;
  }
};

/**
 * Get unread count
 */
const getUnreadCount = async (userId, tenantId) => {
  try {
    const count = await Notification.countDocuments({
      userId,
      tenantId,
      isRead: false,
    });

    return { count };
  } catch (error) {
    logger.error('Get unread count error:', error);
    throw error;
  }
};

/**
 * Create notification
 */
const createNotification = async (notificationData) => {
  try {
    const notification = new Notification(notificationData);
    await notification.save();
    return notification;
  } catch (error) {
    logger.error('Create notification error:', error);
    throw error;
  }
};

module.exports = {
  getNotifications,
  markNotificationRead,
  markAllNotificationsRead,
  getUnreadCount,
  createNotification,
};

