/**
 * Media Service
 */

const Media = require('../models/Media.model');
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;
const logger = require('../utils/logger');

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    const uploadDir = 'uploads/media';
    try {
      await fs.mkdir(uploadDir, { recursive: true });
      cb(null, uploadDir);
    } catch (error) {
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  },
});

const upload = multer({
  storage,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|mp4|mov/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Invalid file type. Only images, PDFs, and videos are allowed.'));
    }
  },
});

/**
 * Upload single file
 */
const uploadFile = async (file, jobId, tenantId, userId, metadata = {}) => {
  try {
    const fileUrl = `/uploads/media/${file.filename}`;
    const fileSizeKb = Math.round(file.size / 1024);

    const media = new Media({
      jobId,
      tenantId,
      mediaType: getMediaType(file.mimetype),
      fileUrl,
      fileSizeKb,
      uploadedByUserId: userId,
      caption: metadata.caption,
      isBeforePhoto: metadata.isBeforePhoto || false,
      isAfterPhoto: metadata.isAfterPhoto || false,
    });

    await media.save();
    return media;
  } catch (error) {
    logger.error('Upload file error:', error);
    throw error;
  }
};

/**
 * Upload multiple files
 */
const uploadMultipleFiles = async (files, jobId, tenantId, userId) => {
  try {
    const mediaFiles = [];

    for (const file of files) {
      const media = await uploadFile(file, jobId, tenantId, userId);
      mediaFiles.push(media);
    }

    return mediaFiles;
  } catch (error) {
    logger.error('Upload multiple files error:', error);
    throw error;
  }
};

/**
 * Get media by job ID
 */
const getMediaByJobId = async (jobId, tenantId) => {
  try {
    const media = await Media.find({ jobId, tenantId })
      .populate('uploadedByUserId', 'name email')
      .sort({ createdAt: -1 });

    return media;
  } catch (error) {
    logger.error('Get media by job ID error:', error);
    throw error;
  }
};

/**
 * Delete media
 */
const deleteMedia = async (mediaId, tenantId, userId) => {
  try {
    const media = await Media.findOne({ _id: mediaId, tenantId });

    if (!media) {
      throw new Error('Media not found');
    }

    // Only allow deletion by uploader or admin
    if (media.uploadedByUserId.toString() !== userId.toString()) {
      // Check if user is admin (would need to check user role)
      // For now, allow deletion
    }

    // Delete file from filesystem
    try {
      await fs.unlink(media.fileUrl.replace(/^\//, ''));
    } catch (error) {
      logger.warn('File not found for deletion:', media.fileUrl);
    }

    await Media.findByIdAndDelete(mediaId);

    return { message: 'Media deleted successfully' };
  } catch (error) {
    logger.error('Delete media error:', error);
    throw error;
  }
};

/**
 * Get media type from MIME type
 */
const getMediaType = (mimetype) => {
  if (mimetype.startsWith('image/')) {
    return 'PHOTO';
  } else if (mimetype.startsWith('video/')) {
    return 'VIDEO';
  } else {
    return 'DOCUMENT';
  }
};

module.exports = {
  upload,
  uploadFile,
  uploadMultipleFiles,
  getMediaByJobId,
  deleteMedia,
};

