/**
 * KPI Service
 */

const KPIScore = require('../models/KPIScore.model');
const Job = require('../models/Job.model');
const JobApproval = require('../models/JobApproval.model');
const User = require('../models/User.model');
const { calculateAgentKPI: calculateAgentKPIScore, calculateManagerKPI } = require('../utils/kpi');
const logger = require('../utils/logger');

/**
 * Calculate and update agent KPI
 */
const calculateAndUpdateAgentKPI = async (agentId, tenantId, startDate, endDate) => {
  try {
    const agent = await User.findById(agentId);
    if (!agent || agent.role !== 'SERVICE_AGENT') {
      throw new Error('Agent not found');
    }

    // Get jobs in date range
    const jobs = await Job.find({
      tenantId,
      assignedAgentId: agentId,
      createdAt: { $gte: startDate, $lte: endDate },
    });

    if (jobs.length === 0) {
      return null;
    }

    // Calculate metrics
    const completedJobs = jobs.filter(j => j.status === 'CLOSED');
    const acceptedJobs = jobs.filter(j => j.status !== 'NEW');
    
    // Response time (average time to accept)
    const responseTimes = [];
    acceptedJobs.forEach(job => {
      if (job.acceptedAt && job.createdAt) {
        const diff = (new Date(job.acceptedAt) - new Date(job.createdAt)) / 60000; // minutes
        responseTimes.push(diff);
      }
    });
    const avgResponseTime = responseTimes.length > 0
      ? responseTimes.reduce((a, b) => a + b, 0) / responseTimes.length
      : 0;

    // Approval rate
    const approvals = await JobApproval.find({
      tenantId,
      jobId: { $in: jobs.map(j => j._id) },
    });
    const approvedCount = approvals.filter(a => a.approvalStatus === 'APPROVED').length;
    const approvalRate = approvals.length > 0
      ? (approvedCount / approvals.length) * 100
      : 0;

    // On-time completion rate
    const onTimeJobs = completedJobs.filter(job => {
      if (!job.slaDeadline || !job.completedAt) return false;
      return new Date(job.completedAt) <= new Date(job.slaDeadline);
    });
    const onTimeCompletionRate = completedJobs.length > 0
      ? (onTimeJobs.length / completedJobs.length) * 100
      : 0;

    // Quality score (based on rejection rate - lower rejection = higher quality)
    const rejectedCount = approvals.filter(a => a.approvalStatus === 'REJECTED').length;
    const qualityScore = approvals.length > 0
      ? 100 - ((rejectedCount / approvals.length) * 100)
      : 100;

    // Customer rating (from user model)
    const avgCustomerRating = agent.rating || 0;

    // Calculate KPI
    const kpiData = {
      avgResponseTime,
      approvalRate,
      onTimeCompletionRate,
      qualityScore,
      avgCustomerRating,
    };

    const totalScore = calculateAgentKPIScore(kpiData);

    // Save or update KPI score
    const scoreDate = new Date();
    scoreDate.setHours(0, 0, 0, 0);

    let kpiScore = await KPIScore.findOne({
      tenantId,
      userId: agentId,
      role: 'SERVICE_AGENT',
      scoreDate,
    });

    if (kpiScore) {
      kpiScore.responseTimeScore = Math.min(20, (avgResponseTime <= 15 ? 20 : avgResponseTime <= 30 ? 15 : avgResponseTime <= 60 ? 10 : 5));
      kpiScore.approvalRateScore = Math.min(30, (approvalRate >= 95 ? 30 : approvalRate >= 85 ? 25 : approvalRate >= 75 ? 20 : approvalRate >= 60 ? 15 : 10));
      kpiScore.onTimeScore = Math.min(25, (onTimeCompletionRate >= 95 ? 25 : onTimeCompletionRate >= 85 ? 20 : onTimeCompletionRate >= 75 ? 15 : onTimeCompletionRate >= 60 ? 10 : 5));
      kpiScore.qualityScore = Math.min(15, (qualityScore >= 90 ? 15 : qualityScore >= 80 ? 12 : qualityScore >= 70 ? 9 : qualityScore >= 60 ? 6 : 3));
      kpiScore.customerRatingScore = Math.min(10, (avgCustomerRating >= 4.5 ? 10 : avgCustomerRating >= 4.0 ? 8 : avgCustomerRating >= 3.5 ? 6 : avgCustomerRating >= 3.0 ? 4 : 2));
      kpiScore.totalScore = totalScore;
      kpiScore.totalJobs = jobs.length;
      kpiScore.completedJobs = completedJobs.length;
      kpiScore.approvedJobs = approvedCount;
      kpiScore.rejectedJobs = rejectedCount;
    } else {
      kpiScore = new KPIScore({
        tenantId,
        userId: agentId,
        role: 'SERVICE_AGENT',
        scoreDate,
        responseTimeScore: Math.min(20, (avgResponseTime <= 15 ? 20 : avgResponseTime <= 30 ? 15 : avgResponseTime <= 60 ? 10 : 5)),
        approvalRateScore: Math.min(30, (approvalRate >= 95 ? 30 : approvalRate >= 85 ? 25 : approvalRate >= 75 ? 20 : approvalRate >= 60 ? 15 : 10)),
        onTimeScore: Math.min(25, (onTimeCompletionRate >= 95 ? 25 : onTimeCompletionRate >= 85 ? 20 : onTimeCompletionRate >= 75 ? 15 : onTimeCompletionRate >= 60 ? 10 : 5)),
        qualityScore: Math.min(15, (qualityScore >= 90 ? 15 : qualityScore >= 80 ? 12 : qualityScore >= 70 ? 9 : qualityScore >= 60 ? 6 : 3)),
        customerRatingScore: Math.min(10, (avgCustomerRating >= 4.5 ? 10 : avgCustomerRating >= 4.0 ? 8 : avgCustomerRating >= 3.5 ? 6 : avgCustomerRating >= 3.0 ? 4 : 2)),
        totalScore,
        totalJobs: jobs.length,
        completedJobs: completedJobs.length,
        approvedJobs: approvedCount,
        rejectedJobs: rejectedCount,
      });
    }

    await kpiScore.save();

    // Update user's KPI score
    agent.kpiScore = totalScore;
    agent.lastKpiUpdate = new Date();
    await agent.save();

    return kpiScore;
  } catch (error) {
    logger.error('Calculate agent KPI error:', error);
    throw error;
  }
};

/**
 * Get user KPI
 */
const getUserKPI = async (userId, tenantId) => {
  try {
    const user = await User.findById(userId);
    if (!user) {
      throw new Error('User not found');
    }

    // Get latest KPI score
    const kpiScore = await KPIScore.findOne({
      tenantId,
      userId,
      role: user.role,
    })
      .sort({ scoreDate: -1 });

    // Get KPI history (last 30 days)
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

    const kpiHistory = await KPIScore.find({
      tenantId,
      userId,
      role: user.role,
      scoreDate: { $gte: thirtyDaysAgo },
    }).sort({ scoreDate: 1 });

    return {
      current: kpiScore,
      history: kpiHistory,
      user: {
        name: user.name,
        role: user.role,
        rating: user.rating,
      },
    };
  } catch (error) {
    logger.error('Get user KPI error:', error);
    throw error;
  }
};

/**
 * Get leaderboard
 */
const getLeaderboard = async (tenantId, role, limit = 10) => {
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const leaderboard = await KPIScore.find({
      tenantId,
      role,
      scoreDate: today,
    })
      .populate('userId', 'name email profilePhotoUrl')
      .sort({ totalScore: -1 })
      .limit(limit);

    return leaderboard;
  } catch (error) {
    logger.error('Get leaderboard error:', error);
    throw error;
  }
};

/**
 * Get organization KPI (aggregated metrics across all agents)
 */
const getOrganizationKPI = async (tenantId) => {
  try {
    // Get all KPI scores for today (or most recent)
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    // Get all agent KPI scores for today
    const kpiScores = await KPIScore.find({
      tenantId,
      role: 'SERVICE_AGENT',
      scoreDate: today,
    });

    // If no scores for today, get the most recent scores
    let scoresToUse = kpiScores;
    if (scoresToUse.length === 0) {
      // Get most recent scores for each agent
      const agents = await User.find({ tenantId, role: 'SERVICE_AGENT' });
      const recentScores = await Promise.all(
        agents.map(async (agent) => {
          const score = await KPIScore.findOne({
            tenantId,
            userId: agent._id,
            role: 'SERVICE_AGENT',
          })
            .sort({ scoreDate: -1 })
            .limit(1);
          return score;
        })
      );
      scoresToUse = recentScores.filter(score => score !== null);
    }

    if (scoresToUse.length === 0) {
      // Return default values if no KPI data exists
      return {
        avgResponseTime: 0,
        approvalRate: 0,
        onTimeRate: 0,
        avgKPIScore: 0,
        totalAgents: 0,
        totalJobs: 0,
        completedJobs: 0,
        approvedJobs: 0,
        rejectedJobs: 0,
      };
    }

    // Calculate averages from actual job data for more accuracy
    const allJobs = await Job.find({
      tenantId,
      assignedAgentId: { $exists: true, $ne: null },
      createdAt: { $gte: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000) }, // Last 30 days
    });

    // Calculate average response time (time to accept job)
    const responseTimes = [];
    allJobs.forEach(job => {
      if (job.acceptedAt && job.createdAt) {
        const diff = (new Date(job.acceptedAt) - new Date(job.createdAt)) / 60000; // minutes
        responseTimes.push(diff);
      }
    });
    const avgResponseTime = responseTimes.length > 0
      ? responseTimes.reduce((a, b) => a + b, 0) / responseTimes.length
      : 0;

    // Calculate approval rate
    const approvals = await JobApproval.find({
      tenantId,
      jobId: { $in: allJobs.map(j => j._id) },
    });
    const approvedCount = approvals.filter(a => a.approvalStatus === 'APPROVED').length;
    const approvalRate = approvals.length > 0
      ? approvedCount / approvals.length
      : 0;

    // Calculate on-time completion rate
    const completedJobs = allJobs.filter(j => j.status === 'CLOSED');
    const onTimeJobs = completedJobs.filter(job => {
      if (!job.slaDeadline || !job.completedAt) return false;
      return new Date(job.completedAt) <= new Date(job.slaDeadline);
    });
    const onTimeRate = completedJobs.length > 0
      ? onTimeJobs.length / completedJobs.length
      : 0;

    // Calculate average KPI score from scores
    const totalScoreSum = scoresToUse.reduce((sum, score) => sum + (score.totalScore || 0), 0);
    const avgKPIScore = scoresToUse.length > 0
      ? totalScoreSum / scoresToUse.length
      : 0;

    // Aggregate job statistics
    const totalJobs = allJobs.length;
    const completedJobsCount = completedJobs.length;
    const approvedJobsCount = approvedCount;
    const rejectedJobsCount = approvals.filter(a => a.approvalStatus === 'REJECTED').length;

    return {
      avgResponseTime: Math.round(avgResponseTime * 10) / 10, // Round to 1 decimal
      approvalRate: Math.round(approvalRate * 1000) / 1000, // Round to 3 decimals (for percentage)
      onTimeRate: Math.round(onTimeRate * 1000) / 1000, // Round to 3 decimals (for percentage)
      avgKPIScore: Math.round(avgKPIScore * 10) / 10, // Round to 1 decimal
      totalAgents: scoresToUse.length,
      totalJobs,
      completedJobs: completedJobsCount,
      approvedJobs: approvedJobsCount,
      rejectedJobs: rejectedJobsCount,
    };
  } catch (error) {
    logger.error('Get organization KPI error:', error);
    throw error;
  }
};

module.exports = {
  calculateAndUpdateAgentKPI,
  getUserKPI,
  getLeaderboard,
  getOrganizationKPI,
};

