/**
 * Job Service
 */

const Job = require('../models/Job.model');
const JobWorkDetails = require('../models/JobWorkDetails.model');
const JobActivity = require('../models/JobActivity.model');
const User = require('../models/User.model');
const Employee = require('../models/Employee.model');
const { calculateSLADeadline } = require('../utils/sla');
const logger = require('../utils/logger');

/**
 * Helper function to populate assignedAgentId (can be User or Employee)
 */
const populateAssignedAgent = async (job) => {
  if (!job.assignedAgentId) {
    return job;
  }
  
  try {
    // Try User first
    const user = await User.findById(job.assignedAgentId).select('name email phone');
    if (user) {
      job.assignedAgentId = user;
      job.assignedAgentModel = 'User';
      return job;
    }
    
    // Try Employee if User not found
    const employee = await Employee.findById(job.assignedAgentId).select('firstName lastName email contactNumber');
    if (employee) {
      job.assignedAgentId = employee;
      job.assignedAgentModel = 'Employee';
      return job;
    }
  } catch (e) {
    logger.warn('Error populating assignedAgentId:', e);
  }
  
  return job;
};

/**
 * Create job
 */
const createJob = async (jobData, tenantId, userId) => {
  try {
    const job = new Job({
      ...jobData,
      tenantId,
      createdBy: userId,
    });

    // Calculate SLA deadline
    if (job.ticketType === 'COMPLIANCE') {
      job.slaDeadline = calculateSLADeadline(
        job.ticketType,
        job.priority,
        new Date()
      );
    }

    await job.save();

    // Create job created activity
    try {
      const createdBy = await User.findById(userId);
      if (createdBy) {
        await createJobActivity({
          jobId: job._id,
          tenantId,
          activityType: 'JOB_CREATED',
          performedBy: userId,
          performedByModel: 'User',
          performedByName: createdBy.name || createdBy.email,
          description: 'Job created',
        });
      }
    } catch (activityError) {
      logger.warn('Failed to create job activity:', activityError);
      // Don't fail job creation if activity logging fails
    }

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Create job error:', error);
    throw error;
  }
};

/**
 * Get job by ID with proper isolation
 */
const getJobById = async (jobId, tenantId, userRole, userId, roleName = null) => {
  try {
    logger.info('🔵 [GET JOB BY ID] Starting', { jobId, userId, userRole, roleName });
    
    // First, get the job directly
    const job = await Job.findOne({ _id: jobId, tenantId })
      .populate('customerId', 'name email phone')
      .populate('managerId', 'name email')
      .populate('complaintCategories', 'name priority');
    
    if (!job) {
      logger.warn('⚠️ [GET JOB BY ID] Job not found', { jobId, tenantId });
      throw new Error('Job not found');
    }
    
    // Determine if user is a manager
    const isManager = userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager');
    const isAgent = userRole === 'SERVICE_AGENT' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent');
    
    // Check access based on role
    if (isManager) {
      logger.info('👔 [GET JOB BY ID] Manager access check', { userId, userRole, roleName });
      
      // Try to find User account first
      let manager = await User.findById(userId);
      
      // If not found or not a User model manager, check Employee
      if (!manager || manager.role !== 'MANAGER') {
        const employee = await Employee.findById(userId).populate('roleId', 'name');
        if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
          manager = await User.findOne({ 
            email: employee.email, 
            tenantId: tenantId,
            role: 'MANAGER' 
          });
        }
      }
      
      if (manager && manager.assignedAgents && manager.assignedAgents.length > 0) {
        // Get assigned User agents
        const assignedUserAgents = await User.find({
          _id: { $in: manager.assignedAgents },
          tenantId,
          role: 'SERVICE_AGENT',
        }).select('email');
        
        // Get emails and find corresponding Employee records
        const assignedAgentEmails = assignedUserAgents.map(agent => agent.email);
        const assignedEmployees = await Employee.find({
          email: { $in: assignedAgentEmails },
          tenantId,
        }).select('_id email');
        
        // Build combined list of agent IDs
        const agentUserIds = manager.assignedAgents.map(id => id.toString());
        const agentEmployeeIds = assignedEmployees.map(emp => emp._id.toString());
        const allAgentIds = [...agentUserIds, ...agentEmployeeIds];
        
        const jobAgentIdStr = job.assignedAgentId?.toString();
        const hasAccess = allAgentIds.includes(jobAgentIdStr);
        
        logger.info('🔍 [GET JOB BY ID] Manager access check result', { hasAccess, jobAgentId: jobAgentIdStr });
        
        if (!hasAccess) {
          logger.warn('⚠️ [GET JOB BY ID] Manager does not have access to this job');
          throw new Error('Job not found');
        }
      } else {
        logger.warn('⚠️ [GET JOB BY ID] Manager has no assigned agents');
        throw new Error('Job not found');
      }
    } else if (isAgent || (userRole === 'EMPLOYEE' && !isManager)) {
      // Agent/Employee can only see own jobs
      // Build list of agent IDs (both User and Employee IDs)
      const agentIds = [userId.toString()];
      
      // If agent is an Employee, also check for corresponding User account
      if (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent') {
        const employee = await Employee.findById(userId);
        if (employee) {
          const agentUser = await User.findOne({
            email: employee.email,
            tenantId,
            role: 'SERVICE_AGENT',
          });
          if (agentUser) {
            agentIds.push(agentUser._id.toString());
            logger.info('✅ [GET JOB BY ID] Found User account for Employee agent', { userId: agentUser._id });
          }
        }
      } else if (userRole === 'SERVICE_AGENT') {
        // If agent is a User, also check for corresponding Employee account
        const agentUser = await User.findById(userId);
        if (agentUser) {
          const employee = await Employee.findOne({
            email: agentUser.email,
            tenantId,
          }).populate('roleId', 'name');
          if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
            agentIds.push(employee._id.toString());
            logger.info('✅ [GET JOB BY ID] Found Employee account for User agent', { employeeId: employee._id });
          }
        }
      }
      
      const jobAgentIdStr = job.assignedAgentId?.toString();
      const hasAccess = agentIds.includes(jobAgentIdStr);
      
      logger.info('🔍 [GET JOB BY ID] Agent access check', { jobAgentId: jobAgentIdStr, agentIds, hasAccess });
      
      if (!hasAccess) {
        logger.warn('⚠️ [GET JOB BY ID] Agent does not have access to this job', { jobAgentId: jobAgentIdStr, agentIds });
        throw new Error('Job not found');
      }
    } else if (userRole === 'CUSTOMER') {
      // Customer can only see own jobs
      const jobCustomerIdStr = job.customerId?.toString();
      const userIdStr = userId.toString();
      
      if (jobCustomerIdStr !== userIdStr) {
        logger.warn('⚠️ [GET JOB BY ID] Customer does not have access to this job');
        throw new Error('Job not found');
      }
    }
    
    // Populate assigned agent
    await populateAssignedAgent(job);
    
    logger.info('✅ [GET JOB BY ID] Job retrieved successfully', { jobId: job._id });
    return job;
  } catch (error) {
    logger.error('❌ [GET JOB BY ID] Error:', error);
    throw error;
  }
};

/**
 * Get jobs with proper isolation
 */
const getJobs = async (filters, tenantId, userRole, userId, roleName = null) => {
  try {
    const query = { tenantId };

    // Determine if user is a manager (by roleName or userRole)
    const isManager = userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager');
    const isAgent = userRole === 'SERVICE_AGENT' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent');

    // Store manager's assigned agent IDs for later use in filtering
    let managerAgentIds = null;

    // Manager isolation - only see assigned agents' jobs
    if (isManager) {
      logger.info('🔵 [GET JOBS] Manager detected', { userId, userRole, roleName });
      
      // Try to find User account first (for User model managers)
      let manager = await User.findById(userId);
      
      // If not found or not a User model manager, try to find by email (for Employee managers)
      if (!manager || manager.role !== 'MANAGER') {
        logger.info('📱 [GET JOBS] Manager not found as User, checking Employee collection...');
        const employee = await Employee.findById(userId);
        if (employee) {
          logger.info('✅ [GET JOBS] Employee found', { employeeId: employee._id, email: employee.email });
          // Find corresponding User account with MANAGER role
          manager = await User.findOne({ 
            email: employee.email, 
            tenantId: tenantId,
            role: 'MANAGER' 
          });
          if (manager) {
            logger.info('✅ [GET JOBS] Found corresponding User account for Employee manager', { userId: manager._id });
          } else {
            logger.warn('⚠️ [GET JOBS] Employee manager found but no corresponding User account');
          }
        }
      }
      
      if (manager && manager.assignedAgents && manager.assignedAgents.length > 0) {
        logger.info('👥 [GET JOBS] Manager has assigned agents', { count: manager.assignedAgents.length, agentIds: manager.assignedAgents });
        
        // Get the User agents (these are the assigned agents)
        const assignedUserAgents = await User.find({
          _id: { $in: manager.assignedAgents },
          tenantId,
          role: 'SERVICE_AGENT',
        }).select('email');
        
        logger.info('📧 [GET JOBS] Found assigned User agents', { count: assignedUserAgents.length });
        
        // Get emails of assigned User agents
        const assignedAgentEmails = assignedUserAgents.map(agent => agent.email);
        logger.info('📧 [GET JOBS] Assigned agent emails', { emails: assignedAgentEmails });
        
        // Find corresponding Employee records for these agents (by email)
        const assignedEmployees = await Employee.find({
          email: { $in: assignedAgentEmails },
          tenantId,
        }).select('_id email');
        
        logger.info('👤 [GET JOBS] Found corresponding Employee records', { count: assignedEmployees.length });
        
        // Build combined list of agent IDs (both User IDs and Employee IDs)
        const agentUserIds = manager.assignedAgents.map(id => id.toString());
        const agentEmployeeIds = assignedEmployees.map(emp => emp._id.toString());
        const allAgentIds = [...agentUserIds, ...agentEmployeeIds];
        
        // Store for later use in filtering
        managerAgentIds = allAgentIds;
        
        logger.info('🔍 [GET JOBS] Combined agent IDs for query', { 
          userIds: agentUserIds, 
          employeeIds: agentEmployeeIds,
          total: allAgentIds.length 
        });
        
        // Manager can see ONLY jobs assigned to their agents (not unassigned jobs)
        query.assignedAgentId = { $in: allAgentIds };
      } else {
        logger.warn('⚠️ [GET JOBS] Manager has no assigned agents - returning empty list');
        // Manager with no assigned agents sees nothing
        return {
          jobs: [],
          pagination: {
            page: parseInt(filters.page) || 1,
            limit: parseInt(filters.limit) || 10,
            total: 0,
            totalPages: 0,
          },
        };
      }
    } else if (isAgent) {
      // Agent/Employee isolation - see own jobs AND unassigned jobs from their manager
      logger.info('🔵 [GET JOBS] Agent detected - showing own jobs and unassigned jobs from manager', { userId, userRole, roleName });
      
      // Find agent's manager
      let agent = await User.findById(userId);
      let managerId = null;
      
      if (agent && agent.role === 'SERVICE_AGENT' && agent.managerId) {
        managerId = agent.managerId;
        logger.info('✅ [GET JOBS] Found manager from User agent', { agentId: userId, managerId });
      } else {
        // Try Employee model
        const employee = await Employee.findById(userId).populate('roleId', 'name');
        if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
          logger.info('✅ [GET JOBS] Found Employee agent', { employeeId: userId, employeeManagerId: employee.managerId });
          
          // First check Employee's managerId
          if (employee.managerId) {
            // Find the corresponding User account for this manager
            const managerEmployee = await Employee.findById(employee.managerId).populate('roleId', 'name');
            if (managerEmployee && managerEmployee.roleId && managerEmployee.roleId.name && managerEmployee.roleId.name.toLowerCase() === 'manager') {
              // Find User account for this manager
              const managerUser = await User.findOne({
                email: managerEmployee.email,
                tenantId,
                role: 'MANAGER',
              });
              if (managerUser) {
                managerId = managerUser._id;
                logger.info('✅ [GET JOBS] Found manager User account from Employee manager', { managerId });
              } else {
                // Use Employee ID as fallback
                managerId = employee.managerId;
                logger.info('⚠️ [GET JOBS] Using Employee manager ID as fallback', { managerId });
              }
            } else {
              managerId = employee.managerId;
            }
          } else {
            // Try to find User account for this agent and get managerId from there
            const agentUser = await User.findOne({
              email: employee.email,
              tenantId,
              role: 'SERVICE_AGENT',
            });
            if (agentUser && agentUser.managerId) {
              managerId = agentUser.managerId;
              logger.info('✅ [GET JOBS] Found manager from User account of Employee agent', { managerId });
            }
          }
        }
      }
      
      if (managerId) {
        // Build list of agent IDs (both User and Employee IDs)
        const agentIds = [userId.toString()];
        
        // If agent is an Employee, also check for corresponding User account
        if (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent') {
          const employee = await Employee.findById(userId);
          if (employee) {
            const agentUser = await User.findOne({
              email: employee.email,
              tenantId,
              role: 'SERVICE_AGENT',
            });
            if (agentUser) {
              agentIds.push(agentUser._id.toString());
              logger.info('✅ [GET JOBS] Found User account for Employee agent', { userId: agentUser._id });
            }
          }
        } else if (userRole === 'SERVICE_AGENT') {
          // If agent is a User, also check for corresponding Employee account
          const agentUser = await User.findById(userId);
          if (agentUser) {
            const employee = await Employee.findOne({
              email: agentUser.email,
              tenantId,
            }).populate('roleId', 'name');
            if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
              agentIds.push(employee._id.toString());
              logger.info('✅ [GET JOBS] Found Employee account for User agent', { employeeId: employee._id });
            }
          }
        }
        
        // Show jobs assigned to this agent (by any of their IDs) OR unassigned jobs from their manager
        query.$or = [
          { assignedAgentId: { $in: agentIds } },
          {
            managerId: managerId,
            $or: [
              { assignedAgentId: { $exists: false } },
              { assignedAgentId: null }
            ],
          },
        ];
        logger.info('✅ [GET JOBS] Agent query built', { agentIds, managerId, hasUnassigned: true });
      } else {
        // No manager - only show own jobs (check both User and Employee IDs)
        const agentIds = [userId.toString()];
        
        // If agent is an Employee, also check for corresponding User account
        if (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent') {
          const employee = await Employee.findById(userId);
          if (employee) {
            const agentUser = await User.findOne({
              email: employee.email,
              tenantId,
              role: 'SERVICE_AGENT',
            });
            if (agentUser) {
              agentIds.push(agentUser._id.toString());
            }
          }
        } else if (userRole === 'SERVICE_AGENT') {
          // If agent is a User, also check for corresponding Employee account
          const agentUser = await User.findById(userId);
          if (agentUser) {
            const employee = await Employee.findOne({
              email: agentUser.email,
              tenantId,
            }).populate('roleId', 'name');
            if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
              agentIds.push(employee._id.toString());
            }
          }
        }
        
        query.assignedAgentId = { $in: agentIds };
        logger.info('✅ [GET JOBS] Agent query built (no manager)', { agentIds });
      }
    } else if (userRole === 'EMPLOYEE' && !isManager) {
      // Generic Employee (not manager, not agent) - show own jobs
      logger.info('🔵 [GET JOBS] Generic Employee detected - showing own jobs', { userId, userRole, roleName });
      query.assignedAgentId = userId;
    }

    // Customer isolation - only see own jobs
    if (userRole === 'CUSTOMER') {
      query.customerId = userId;
    }

    // Apply filters
    if (filters.status) {
      // Support comma-separated statuses
      if (filters.status.includes(',')) {
        query.status = { $in: filters.status.split(',').map(s => s.trim()) };
      } else {
        query.status = filters.status;
      }
    }
    if (filters.ticketType) {
      query.ticketType = filters.ticketType;
    }
    
    // Handle assignedAgentId filter
    // For managers, this should be combined with their assigned agents list
    if (filters.assignedAgentId) {
      if (isManager && managerAgentIds) {
        // Manager filtering by specific agent - ensure it's one of their assigned agents
        const filteredAgentId = filters.assignedAgentId.toString();
        logger.info('🔍 [GET JOBS] Manager filtering by agent', { filteredAgentId, managerAgentIds });
        
        // Check if the filtered agent ID is in the manager's assigned agents
        if (managerAgentIds.includes(filteredAgentId)) {
          // Find both User and Employee IDs for this agent (if they exist)
          // The filteredAgentId could be either a User ID or Employee ID
          const agentUser = await User.findById(filteredAgentId);
          let agentIds = [filteredAgentId];
          
          if (agentUser) {
            // If it's a User, find corresponding Employee
            const agentEmployee = await Employee.findOne({
              email: agentUser.email,
              tenantId,
            });
            if (agentEmployee) {
              agentIds.push(agentEmployee._id.toString());
            }
          } else {
            // If it's an Employee, find corresponding User
            const agentEmployee = await Employee.findById(filteredAgentId);
            if (agentEmployee) {
              const agentUser = await User.findOne({
                email: agentEmployee.email,
                tenantId,
                role: 'SERVICE_AGENT',
              });
              if (agentUser) {
                agentIds.push(agentUser._id.toString());
              }
            }
          }
          
          // Filter to only this specific agent (check both User and Employee IDs)
          query.assignedAgentId = { $in: agentIds };
          logger.info('✅ [GET JOBS] Filtered agent is in manager\'s assigned agents - applying filter', { agentIds });
        } else {
          // Agent not in manager's list - return empty
          logger.warn('⚠️ [GET JOBS] Filtered agent not in manager\'s assigned agents', { 
            filteredAgentId, 
            managerAgentIds 
          });
          return {
            jobs: [],
            pagination: {
              page: parseInt(filters.page) || 1,
              limit: parseInt(filters.limit) || 10,
              total: 0,
              totalPages: 0,
            },
          };
        }
      } else {
        // For non-managers or managers without agent list, check both User and Employee IDs
        const filteredAgentId = filters.assignedAgentId.toString();
        const agentUser = await User.findById(filteredAgentId);
        let agentIds = [filteredAgentId];
        
        if (agentUser) {
          // If it's a User, find corresponding Employee
          const agentEmployee = await Employee.findOne({
            email: agentUser.email,
            tenantId,
          });
          if (agentEmployee) {
            agentIds.push(agentEmployee._id.toString());
          }
        } else {
          // If it's an Employee, find corresponding User
          const agentEmployee = await Employee.findById(filteredAgentId);
          if (agentEmployee) {
            const agentUser = await User.findOne({
              email: agentEmployee.email,
              tenantId,
              role: 'SERVICE_AGENT',
            });
            if (agentUser) {
              agentIds.push(agentUser._id.toString());
            }
          }
        }
        
        query.assignedAgentId = { $in: agentIds };
      }
    }
    if (filters.ticketNumber) {
      query.ticketNumber = { $regex: filters.ticketNumber, $options: 'i' };
    }
    if (filters.fromDate || filters.toDate) {
      const fromDate = filters.fromDate ? new Date(filters.fromDate) : null;
      const toDate = filters.toDate ? new Date(filters.toDate) : null;
      
      if (fromDate) {
        fromDate.setHours(0, 0, 0, 0);
      }
      if (toDate) {
        toDate.setHours(23, 59, 59, 999);
      }
      
      // For PERIODIC jobs, use scheduledDate (when job is scheduled to be done)
      // For PENDING_APPROVAL jobs, use updatedAt (when job was submitted/updated for approval)
      // For other jobs, use createdAt
      const isPeriodic = filters.ticketType === 'PERIODIC';
      const isPendingApproval = filters.status === 'PENDING_APPROVAL' || 
                                (filters.status && filters.status.includes('PENDING_APPROVAL'));
      
      let dateField;
      if (isPeriodic) {
        dateField = 'scheduledDate';
      } else if (isPendingApproval) {
        dateField = 'updatedAt';
      } else {
        dateField = 'createdAt';
      }
      
      if (!query[dateField]) {
        query[dateField] = {};
      }
      
      if (fromDate) {
        query[dateField].$gte = fromDate;
      }
      if (toDate) {
        query[dateField].$lte = toDate;
      }
    }

    // Pagination
    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 10;
    const skip = (page - 1) * limit;

    // Get total count for pagination (with timeout protection)
    let total;
    try {
      total = await Job.countDocuments(query).maxTimeMS(5000);
    } catch (countError) {
      logger.error('❌ [GET JOBS] Count query timeout or error:', countError);
      total = 0;
    }

    // Get jobs (with timeout protection)
    let jobs;
    try {
      jobs = await Job.find(query)
        .skip(skip)
        .limit(limit)
        .populate('customerId', 'name email phone')
        .populate('managerId', 'name email')
        .populate('productId', 'name productName')
        .populate('complaintCategories', 'name priority')
        .sort({ createdAt: -1 })
        .maxTimeMS(10000); // 10 second timeout
    } catch (queryError) {
      logger.error('❌ [GET JOBS] Query timeout or error:', queryError);
      return {
        jobs: [],
        pagination: {
          page,
          limit,
          total: 0,
          totalPages: 0,
        },
      };
    }
    
    // Manually populate assignedAgentId for each job (can be User or Employee)
    for (const job of jobs) {
      await populateAssignedAgent(job);
    }

    return {
      jobs,
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
    };
  } catch (error) {
    logger.error('Get jobs error:', error);
    throw error;
  }
};

/**
 * Accept job
 * @param {string} jobId - Job ID
 * @param {string} agentId - Agent/Employee ID
 * @param {string} tenantId - Tenant ID
 * @param {string} timestamp - Timestamp from mobile device (ISO string)
 */
const acceptJob = async (jobId, agentId, tenantId, timestamp) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    if (job.assignedAgentId?.toString() !== agentId.toString()) {
      throw new Error('Job not assigned to you');
    }

    if (job.status !== 'NEW') {
      throw new Error('Job cannot be accepted in current status');
    }

    job.status = 'ACCEPTED';
    // Use timestamp from mobile device, fallback to server time if not provided
    job.acceptedAt = timestamp ? new Date(timestamp) : new Date();
    await job.save();

    // Create activity log
    try {
      // Check if agent is Employee or User
      let agent = await Employee.findById(agentId);
      let performedByModel = 'Employee';
      let agentName = null;
      
      if (agent) {
        agentName = `${agent.firstName} ${agent.lastName}`;
      } else {
        agent = await User.findById(agentId);
        if (agent) {
          performedByModel = 'User';
          agentName = agent.name || agent.email;
        }
      }
      
      if (agent) {
        await createJobActivity({
          jobId,
          tenantId,
          activityType: 'JOB_ACCEPTED',
          performedBy: agentId,
          performedByModel,
          performedByName: agentName,
          description: 'Job accepted by agent',
          timestamp: timestamp ? new Date(timestamp) : new Date(), // Use mobile timestamp
        });
      }
    } catch (activityError) {
      logger.warn('Failed to create job activity:', activityError);
    }

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Accept job error:', error);
    throw error;
  }
};

/**
 * Start job with GPS validation
 * @param {string} jobId - Job ID
 * @param {string} agentId - Agent/Employee ID
 * @param {string} tenantId - Tenant ID
 * @param {object} location - Location object with lat, lng
 * @param {string} timestamp - Timestamp from mobile device (ISO string)
 */
const startJob = async (jobId, agentId, tenantId, location, timestamp, userRole = null, roleName = null) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    // Check agent access (handles both User and Employee IDs)
    if (userRole) {
      await checkAgentJobAccess(job, agentId, tenantId, userRole, roleName);
    } else {
      // Fallback for backward compatibility
      if (job.assignedAgentId?.toString() !== agentId.toString()) {
        throw new Error('Job not assigned to you');
      }
    }

    // Allow starting if:
    // 1. Status is ARRIVED (explicitly marked as arrived)
    // 2. Status is ACCEPTED (can start without arriving first)
    // 3. arrivedAt timestamp exists (job was marked as arrived, even if status wasn't updated)
    // This provides flexibility and handles cases where status might not have been updated properly
    const canStart = 
      job.status === 'ARRIVED' || 
      job.status === 'ACCEPTED' || 
      (job.arrivedAt !== null && job.arrivedAt !== undefined);
    
    if (!canStart) {
      throw new Error('Job must be accepted or marked as arrived before starting');
    }

    // GPS validation removed - no distance check required for starting job

    job.status = 'IN_PROGRESS';
    // Use timestamp from mobile device, fallback to server time if not provided
    job.startedAt = timestamp ? new Date(timestamp) : new Date();

    // Update work details
    let workDetails = await JobWorkDetails.findOne({ jobId });
    if (!workDetails) {
      workDetails = new JobWorkDetails({ jobId });
    }
    workDetails.startLocationVerified = true;
    workDetails.startLocationLat = location.lat;
    workDetails.startLocationLng = location.lng;
    await workDetails.save();

    await job.save();

    // Create activity log
    try {
      // Check if agent is Employee or User
      let agent = await Employee.findById(agentId);
      let performedByModel = 'Employee';
      let agentName = null;
      
      if (agent) {
        agentName = `${agent.firstName} ${agent.lastName}`;
      } else {
        agent = await User.findById(agentId);
        if (agent) {
          performedByModel = 'User';
          agentName = agent.name || agent.email;
        }
      }
      
      if (agent) {
        await createJobActivity({
          jobId,
          tenantId,
          activityType: 'JOB_STARTED',
          performedBy: agentId,
          performedByModel,
          performedByName: agentName,
          description: 'Job started',
          location: {
            latitude: location.lat,
            longitude: location.lng,
            locationName: location.locationName,
          },
          timestamp: timestamp ? new Date(timestamp) : new Date(), // Use mobile timestamp
        });
      }
    } catch (activityError) {
      logger.warn('Failed to create job activity:', activityError);
    }

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Start job error:', error);
    throw error;
  }
};

/**
 * Submit job for approval
 * @param {string} jobId - Job ID
 * @param {string} agentId - Agent/Employee ID
 * @param {string} tenantId - Tenant ID
 * @param {object} workData - Work data including timestamp from mobile device
 */
const submitJob = async (jobId, agentId, tenantId, workData, userRole = null, roleName = null) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    // Check agent access (handles both User and Employee IDs)
    if (userRole) {
      await checkAgentJobAccess(job, agentId, tenantId, userRole, roleName);
    } else {
      // Fallback for backward compatibility
      if (job.assignedAgentId?.toString() !== agentId.toString()) {
        throw new Error('Job not assigned to you');
      }
    }

    if (job.status !== 'IN_PROGRESS') {
      throw new Error('Job must be in progress to submit');
    }

    // Get timestamp from mobile device
    const timestamp = workData.timestamp ? new Date(workData.timestamp) : new Date();

    // Update work details
    let workDetails = await JobWorkDetails.findOne({ jobId });
    if (!workDetails) {
      workDetails = new JobWorkDetails({ jobId });
    }

    workDetails.workNotes = workData.workNotes;
    workDetails.workCompletedFully = workData.workCompletedFully;
    workDetails.sparesRequired = workData.sparesRequired || [];
    workDetails.customerDigitalSignatureUrl = workData.signatureUrl;
    workDetails.customerSignatureData = workData.signatureData;
    workDetails.endLocationVerified = true;
    workDetails.endLocationLat = workData.endLocation?.lat;
    workDetails.endLocationLng = workData.endLocation?.lng;

    if (job.startedAt) {
      // Calculate duration using mobile timestamp if available
      const endTime = timestamp;
      const duration = Math.floor((endTime - job.startedAt) / 60000);
      workDetails.workDurationMinutes = duration;
    }

    await workDetails.save();

    // Update job status
    // If signature provided in workData, persist to Job model and mark signed
    let hasSignature = false;
    if (workData.signatureUrl) {
      job.signatureUrl = workData.signatureUrl;
      job.isSigned = true;
      hasSignature = true;
    }
    if (workData.signatureData) {
      // signatureData is expected to be base64 string
      job.signatureData = workData.signatureData;
      job.isSigned = true;
      hasSignature = true;
    }
    if (workData.signeeName) {
      job.signeeName = workData.signeeName;
    }

    // Set status based on whether signature is provided
    if (hasSignature) {
      job.status = 'PENDING_APPROVAL';
      job.isSigned = true;
    } else {
      job.status = 'SIGNATURE_PENDING';
      job.isSigned = false;
    }
    
    // Use timestamp from mobile device
    job.completedAt = timestamp;
    await job.save();

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Submit job error:', error);
    throw error;
  }
};

/**
 * Calculate distance between two coordinates (Haversine formula)
 */
const calculateDistance = (lat1, lon1, lat2, lon2) => {
  const R = 6371e3; // Earth radius in meters
  const φ1 = lat1 * Math.PI / 180;
  const φ2 = lat2 * Math.PI / 180;
  const Δφ = (lat2 - lat1) * Math.PI / 180;
  const Δλ = (lon2 - lon1) * Math.PI / 180;

  const a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
    Math.cos(φ1) * Math.cos(φ2) *
    Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

  return R * c; // Distance in meters
};

/**
 * Create job activity entry
 * @param {object} activityData - Activity data including optional timestamp from mobile device
 */
const createJobActivity = async (activityData) => {
  try {
    // If timestamp is provided, set it and use it for createdAt
    const activity = new JobActivity(activityData);
    if (activityData.timestamp) {
      activity.createdAt = new Date(activityData.timestamp);
      activity.timestamp = new Date(activityData.timestamp);
    }
    await activity.save();
    return activity;
  } catch (error) {
    logger.error('Create job activity error:', error);
    throw error;
  }
};

/**
 * Helper function to check if agent has access to a job
 * Checks both User and Employee IDs to handle self-assigned jobs
 */
const checkAgentJobAccess = async (job, agentId, tenantId, userRole, roleName = null) => {
  // Build list of agent IDs (both User and Employee IDs) to check access
  const agentIds = [agentId.toString()];
  
  // If agent is an Employee, also check for corresponding User account
  if (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent') {
    const employee = await Employee.findById(agentId);
    if (employee) {
      const agentUser = await User.findOne({
        email: employee.email,
        tenantId,
        role: 'SERVICE_AGENT',
      });
      if (agentUser) {
        agentIds.push(agentUser._id.toString());
        logger.info('✅ [JOB ACCESS] Found User account for Employee agent', { userId: agentUser._id });
      }
    }
  } else if (userRole === 'SERVICE_AGENT') {
    // If agent is a User, also check for corresponding Employee account
    const agentUser = await User.findById(agentId);
    if (agentUser) {
      const employee = await Employee.findOne({
        email: agentUser.email,
        tenantId,
      }).populate('roleId', 'name');
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
        agentIds.push(employee._id.toString());
        logger.info('✅ [JOB ACCESS] Found Employee account for User agent', { employeeId: employee._id });
      }
    }
  }
  
  const jobAgentIdStr = job.assignedAgentId?.toString();
  const hasAccess = agentIds.includes(jobAgentIdStr);
  
  logger.info('🔍 [JOB ACCESS] Agent access check', { jobAgentId: jobAgentIdStr, agentIds, hasAccess });
  
  if (!hasAccess) {
    logger.warn('⚠️ [JOB ACCESS] Agent does not have access to this job', { jobAgentId: jobAgentIdStr, agentIds });
    throw new Error('Job not assigned to you');
  }
  
  return true;
};

/**
 * Mark job as arrived
 * Note: No distance validation is performed - agents can mark as arrived from any location
 * @param {string} jobId - Job ID
 * @param {string} agentId - Agent/Employee ID
 * @param {string} tenantId - Tenant ID
 * @param {object} location - Location object with lat, lng, locationName, address (optional)
 * @param {string} userRole - User role (EMPLOYEE or SERVICE_AGENT)
 * @param {string} timestamp - Timestamp from mobile device (ISO string)
 * @param {string} roleName - Role name from Employee collection (optional)
 */
const arriveJob = async (jobId, agentId, tenantId, location, userRole, timestamp, roleName = null) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    // Check agent access (handles both User and Employee IDs)
    await checkAgentJobAccess(job, agentId, tenantId, userRole, roleName);

    if (job.status === 'CLOSED' || job.status === 'CANCELLED') {
      throw new Error('Cannot arrive at a closed or cancelled job');
    }

    // No distance validation - agents can mark as arrived from any location

    // Get agent details for activity log
    let agent;
    let agentName;
    let performedByModel = 'User';
    
    if (userRole === 'EMPLOYEE') {
      agent = await Employee.findById(agentId);
      if (agent) {
        agentName = `${agent.firstName} ${agent.lastName}`;
        performedByModel = 'Employee';
      }
    } else {
      agent = await User.findById(agentId);
      if (agent) {
        agentName = agent.name || agent.email;
      }
    }

    if (!agent) {
      throw new Error('Agent not found');
    }

    // Update job - use timestamp from mobile device
    if (!job.arrivedAt) {
      job.arrivedAt = timestamp ? new Date(timestamp) : new Date();
    }
    
    // Update job status to ARRIVED if it's NEW or ACCEPTED
    if (job.status === 'NEW' || job.status === 'ACCEPTED') {
      job.status = 'ARRIVED';
    }

    await job.save();

    // Create activity log - use timestamp from mobile device
    await createJobActivity({
      jobId,
      tenantId,
      activityType: 'JOB_ARRIVED',
      performedBy: agentId,
      performedByModel,
      performedByName: agentName,
      description: 'Agent arrived at job location',
      location: location ? {
        latitude: location.lat,
        longitude: location.lng,
        locationName: location.locationName,
        address: location.address,
      } : undefined,
      timestamp: timestamp ? new Date(timestamp) : new Date(), // Use mobile timestamp
    });

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Arrive job error:', error);
    throw error;
  }
};

/**
 * Update job status
 * @param {string} jobId - Job ID
 * @param {string} agentId - Agent/Employee ID
 * @param {string} tenantId - Tenant ID
 * @param {string} newStatus - New status
 * @param {string} description - Status description
 * @param {array} photos - Array of photo URLs
 * @param {string} userRole - User role (EMPLOYEE or SERVICE_AGENT)
 * @param {string} timestamp - Timestamp from mobile device (ISO string)
 * @param {boolean} monitoringRequired - Whether monitoring is required
 * @param {string} signatureUrl - URL of uploaded signature image
 * @param {string} signeeName - Name of the person who signed
 */
const updateJobStatus = async (jobId, agentId, tenantId, newStatus, description, photos, userRole, timestamp, monitoringRequired, signatureUrl, signatureData, signeeName, roleName = null) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    // Check agent access (handles both User and Employee IDs)
    if (userRole && (userRole === 'SERVICE_AGENT' || userRole === 'EMPLOYEE')) {
      await checkAgentJobAccess(job, agentId, tenantId, userRole, roleName);
    } else {
      // Fallback for backward compatibility
      if (job.assignedAgentId?.toString() !== agentId.toString()) {
        throw new Error('Job not assigned to you');
      }
    }

    // Validate status transition
    const validTransitions = {
      'NEW': ['ACCEPTED', 'ARRIVED'],
      'ACCEPTED': ['IN_PROGRESS', 'ARRIVED'],
      'ARRIVED': ['IN_PROGRESS'],
      'IN_PROGRESS': ['PENDING_APPROVAL', 'CLOSED'],
      'PENDING_APPROVAL': ['APPROVED', 'REJECTED'],
      'APPROVED': ['CLOSED'],
      'REJECTED': ['IN_PROGRESS'],
    };
    
    // Also allow status updates if job has arrivedAt timestamp (even if status wasn't updated to ARRIVED)
    // This handles edge cases where the status might not have been properly updated
    const hasArrived = job.arrivedAt !== null && job.arrivedAt !== undefined;

    // Allow transition if it's in valid transitions OR if job has arrived and transitioning to IN_PROGRESS
    const isValidTransition = validTransitions[job.status]?.includes(newStatus);
    const canTransitionFromArrived = hasArrived && newStatus === 'IN_PROGRESS' && (job.status === 'ARRIVED' || job.status === 'ACCEPTED' || job.status === 'NEW');
    
    if (!isValidTransition && !canTransitionFromArrived) {
      throw new Error(`Invalid status transition from ${job.status} to ${newStatus}`);
    }

    // Get agent details
    let agent;
    let agentName;
    let performedByModel = 'User';
    
    if (userRole === 'EMPLOYEE') {
      agent = await Employee.findById(agentId);
      if (agent) {
        agentName = `${agent.firstName} ${agent.lastName}`;
        performedByModel = 'Employee';
      }
    } else {
      agent = await User.findById(agentId);
      if (agent) {
        agentName = agent.name || agent.email;
      }
    }

    if (!agent) {
      throw new Error('Agent not found');
    }

    // Use timestamp from mobile device, fallback to server time if not provided
    const actionTimestamp = timestamp ? new Date(timestamp) : new Date();

    const oldStatus = job.status;
    job.status = newStatus;

    // Update timestamps based on status - use mobile timestamp
    if (newStatus === 'IN_PROGRESS' && !job.startedAt) {
      job.startedAt = actionTimestamp;
    }
    if (newStatus === 'CLOSED' && !job.closedAt) {
      job.closedAt = actionTimestamp;
    }

    // Add photos if provided
    if (photos && photos.length > 0) {
      if (!job.complaintPhotos) {
        job.complaintPhotos = [];
      }
      // Only add photos that don't already exist to avoid duplicates
      const existingPhotos = new Set(job.complaintPhotos.map(p => p.toString()));
      const newPhotos = photos.filter(p => !existingPhotos.has(p.toString()));
      if (newPhotos.length > 0) {
        job.complaintPhotos.push(...newPhotos);
      }
    }

    await job.save();

    // Update JobWorkDetails with all provided data
    let workDetails = await JobWorkDetails.findOne({ jobId });
    if (!workDetails) {
      workDetails = new JobWorkDetails({ jobId });
    }
    
    // Update monitoring required if provided
    if (monitoringRequired !== undefined) {
      workDetails.monitoringRequired = monitoringRequired;
    }
    
    // Update description/remarks if provided
    if (description && description.trim().length > 0) {
      workDetails.statusUpdateDescription = description.trim();
    }
    
    // Update signature if provided
    let hasSignature = false;
    if (signatureUrl) {
      workDetails.customerDigitalSignatureUrl = signatureUrl;
      // Also update job model
      job.signatureUrl = signatureUrl;
      job.isSigned = true;
      hasSignature = true;
    }
    if (signatureData) {
      workDetails.customerSignatureData = signatureData;
      job.signatureData = signatureData;
      job.isSigned = true;
      hasSignature = true;
    }
    
    // Update signee name if provided
    if (signeeName && signeeName.trim().length > 0) {
      workDetails.signeeName = signeeName.trim();
      // Also update job model
      job.signeeName = signeeName.trim();
    }
    
    // Update status based on signature
    // If signature is provided and job is SIGNATURE_PENDING, change to PENDING_APPROVAL
    if (hasSignature && job.status === 'SIGNATURE_PENDING') {
      job.status = 'PENDING_APPROVAL';
      job.isSigned = true;
    } else if (newStatus === 'PENDING_APPROVAL') {
      // If status is being changed to PENDING_APPROVAL
      if (hasSignature) {
        job.status = 'PENDING_APPROVAL';
        job.isSigned = true;
      } else {
        job.status = 'SIGNATURE_PENDING';
        job.isSigned = false;
      }
    } else if (newStatus === 'SIGNATURE_PENDING') {
      // If explicitly setting to SIGNATURE_PENDING, ensure isSigned is false
      job.isSigned = false;
    } else if (hasSignature && (newStatus === 'CLOSED' || newStatus === 'APPROVED')) {
      // If signature is provided when closing/approving, mark as signed
      job.isSigned = true;
      // If was SIGNATURE_PENDING, change to appropriate status
      if (job.status === 'SIGNATURE_PENDING') {
        job.status = newStatus === 'CLOSED' ? 'CLOSED' : 'PENDING_APPROVAL';
      }
    } else if (!hasSignature && (newStatus === 'CLOSED' || newStatus === 'APPROVED')) {
      // If closing/approving without signature, set to SIGNATURE_PENDING
      job.status = 'SIGNATURE_PENDING';
      job.isSigned = false;
    } else if (hasSignature && job.status === 'SIGNATURE_PENDING' && newStatus !== 'SIGNATURE_PENDING') {
      // If signature is added to a SIGNATURE_PENDING job, update status
      if (newStatus === 'IN_PROGRESS' || newStatus === 'PENDING_APPROVAL') {
        job.status = 'PENDING_APPROVAL';
        job.isSigned = true;
      }
    }
    
    await workDetails.save();
    await job.save(); // Save job to persist signature and signeeName

    // Create activity log - use mobile timestamp
    const activityMetadata = {};
    if (photos && photos.length > 0) {
      activityMetadata.photosCount = photos.length;
    }
    if (signatureUrl || signatureData) {
      activityMetadata.hasSignature = true;
      if (signeeName) {
        activityMetadata.signeeName = signeeName;
      }
    }
    if (description) {
      activityMetadata.hasDescription = true;
    }
    
    // Build description with signature info if available
    let activityDescription = description || `Status changed from ${oldStatus} to ${newStatus}`;
    if (signeeName && (signatureUrl || signatureData)) {
      activityDescription += ` - Signed by: ${signeeName}`;
    }
    
    await createJobActivity({
      jobId,
      tenantId,
      activityType: 'STATUS_UPDATED',
      performedBy: agentId,
      performedByModel,
      performedByName: agentName,
      description: activityDescription,
      oldStatus,
      newStatus,
      metadata: activityMetadata,
      timestamp: actionTimestamp, // Use mobile timestamp
    });

    // Create photo activities if photos were added - use mobile timestamp
    if (photos && photos.length > 0) {
      for (const photo of photos) {
        await createJobActivity({
          jobId,
          tenantId,
          activityType: 'PHOTO_ADDED',
          performedBy: agentId,
          performedByModel,
          performedByName: agentName,
          description: 'Photo added to job',
          metadata: { photoUrl: photo },
          timestamp: actionTimestamp, // Use mobile timestamp
        });
      }
    }

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Update job status error:', error);
    throw error;
  }
};

/**
 * Mark job as opened (first time viewing)
 */
const markJobOpened = async (jobId, agentId, tenantId, userRole) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      return; // Don't throw error, just return silently
    }

    // Only mark if not already opened
    if (job.firstOpenedAt) {
      return;
    }

    // Get agent details
    let agent;
    let agentName;
    let performedByModel = 'User';
    
    if (userRole === 'EMPLOYEE') {
      agent = await Employee.findById(agentId);
      if (agent) {
        agentName = `${agent.firstName} ${agent.lastName}`;
        performedByModel = 'Employee';
      }
    } else {
      agent = await User.findById(agentId);
      if (agent) {
        agentName = agent.name || agent.email;
      }
    }

    if (!agent) {
      return;
    }

    job.firstOpenedAt = new Date();
    await job.save();

    // Create activity log
    await createJobActivity({
      jobId,
      tenantId,
      activityType: 'JOB_OPENED',
      performedBy: agentId,
      performedByModel,
      performedByName: agentName,
      description: 'Job opened for the first time',
    });
  } catch (error) {
    logger.error('Mark job opened error:', error);
    // Don't throw, just log
  }
};

/**
 * Get job timeline/activities
 */
const getJobTimeline = async (jobId, tenantId) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    const activities = await JobActivity.find({ jobId, tenantId })
      .sort({ createdAt: 1 }); // Oldest first for timeline

    // Enhance activities with assignment type information
    const enhancedActivities = activities.map(activity => {
      const activityObj = activity.toObject ? activity.toObject() : activity;
      
      // Check if this is a JOB_ASSIGNED activity
      if (activityObj.activityType === 'JOB_ASSIGNED') {
        // Determine if it was assigned by manager or self-assigned by agent
        const description = activityObj.description || '';
        const descriptionLower = description.toLowerCase();
        
        // Check description first (most reliable)
        if (descriptionLower.includes('self-assigned')) {
          activityObj.assignmentType = 'SELF_ASSIGNED';
          activityObj.assignmentInfo = 'Job was taken by agent';
        } else if (descriptionLower.includes('assigned to agent') || descriptionLower.includes('job assigned to')) {
          activityObj.assignmentType = 'MANAGER_ASSIGNED';
          activityObj.assignmentInfo = 'Job was assigned by manager';
        } else {
          // Fallback: check if performedBy matches the job's assignedAgentId or managerId
          const performedById = activityObj.performedBy?.toString();
          const assignedAgentId = job.assignedAgentId?.toString();
          const managerId = job.managerId?.toString();
          
          if (performedById === assignedAgentId) {
            // If performedBy is the assigned agent, it's self-assigned
            activityObj.assignmentType = 'SELF_ASSIGNED';
            activityObj.assignmentInfo = 'Job was taken by agent';
          } else if (performedById === managerId) {
            // If performedBy is the manager, it's manager-assigned
            activityObj.assignmentType = 'MANAGER_ASSIGNED';
            activityObj.assignmentInfo = 'Job was assigned by manager';
          } else {
            // Default: assume manager-assigned if we can't determine
            activityObj.assignmentType = 'MANAGER_ASSIGNED';
            activityObj.assignmentInfo = 'Job was assigned by manager';
          }
        }
      }
      
      return activityObj;
    });

    return enhancedActivities;
  } catch (error) {
    logger.error('Get job timeline error:', error);
    throw error;
  }
};

/**
 * Get pending jobs count for agent
 */
const getPendingJobsCount = async (agentId, tenantId) => {
  try {
    const count = await Job.countDocuments({
      tenantId,
      assignedAgentId: agentId,
      status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS'] },
    });

    return count;
  } catch (error) {
    logger.error('Get pending jobs count error:', error);
    throw error;
  }
};

/**
 * Assign an available agent to a job (tries assigned client -> least loaded employee)
 */
const assignAgentToJob = async (jobId, tenantId) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });
    if (!job) {
      throw new Error('Job not found');
    }

    // If job is already assigned to a User who is SERVICE_AGENT, nothing to do
    if (job.assignedAgentId && job.assignedAgentModel === 'User') {
      const currentUser = await User.findById(job.assignedAgentId).select('role status');
      if (currentUser && currentUser.role === 'SERVICE_AGENT' && currentUser.status === 'ACTIVE') {
        return job;
      }
    }

    // If currently assigned to a non-agent (Employee or User with other role), remove that assignment
    if (job.assignedAgentId) {
      try {
        if (job.assignedAgentModel === 'Employee') {
          const emp = await Employee.findById(job.assignedAgentId);
          if (emp && emp.assignedJobs) {
            emp.assignedJobs = emp.assignedJobs.filter(id => id.toString() !== job._id.toString());
            await emp.save();
          }
        } else if (job.assignedAgentModel === 'User') {
          // If assigned to non-agent user, nothing specific to remove (no assignedJobs field)
        }
      } catch (remErr) {
        logger.warn('Failed to remove previous assignment', remErr);
      }

      job.assignedAgentId = null;
      job.assignedAgentModel = null;
      await job.save();
    }

    // Find candidate SERVICE_AGENT users
    const maxJobsPerAgent = parseInt(process.env.MAX_JOBS_PER_AGENT || '3');

    // 1) Try agents with load < max
    const agents = await User.find({ tenantId, role: 'SERVICE_AGENT', status: 'ACTIVE' });
    if (!agents || agents.length === 0) {
      logger.info('No service agents found for tenant', { tenantId });
      return job;
    }

    let candidate = null;
    for (const agent of agents) {
      const cnt = await Job.countDocuments({ tenantId, assignedAgentId: agent._id, status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS'] } });
      if (cnt < maxJobsPerAgent) {
        if (!candidate || cnt < candidate.count) {
          candidate = { agent, count: cnt };
        }
      }
    }

    // 2) If none found under limit, pick least loaded agent anyway (allow overflow)
    if (!candidate) {
      let best = null;
      for (const agent of agents) {
        const cnt = await Job.countDocuments({ tenantId, assignedAgentId: agent._id, status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS'] } });
        if (!best || cnt < best.count) {
          best = { agent, count: cnt };
        }
      }
      if (best) candidate = best; // may be null if no agents
    }

    if (!candidate || !candidate.agent) {
      logger.info('No suitable agent to assign', { jobId, tenantId });
      return job;
    }

    const agent = candidate.agent;
    job.assignedAgentId = agent._id;
    job.assignedAgentModel = 'User';
    await job.save();

    // Create activity log
    await createJobActivity({
      jobId: job._id,
      tenantId,
      activityType: 'JOB_ASSIGNED',
      performedBy: agent._id,
      performedByModel: 'User',
      performedByName: agent.name || agent.email,
      description: 'Job automatically assigned to service agent by scheduler',
    });

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Assign agent to job error:', error);
    throw error;
  }
};

/**
 * Close job by manager
 * Manager can close jobs from PENDING_APPROVAL status with mandatory closure description
 */
const closeJobByManager = async (jobId, managerId, tenantId, closureDescription, userRole = null, roleName = null) => {
  try {
    logger.info('🔵 [CLOSE JOB] Starting close job by manager', { jobId, managerId, userRole, roleName });
    
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    logger.info('✅ [CLOSE JOB] Job found', { jobId: job._id, status: job.status, assignedAgentId: job.assignedAgentId });

    // Verify manager has access to this job
    // Check if managerId is a User with MANAGER role
    let manager = await User.findById(managerId);
    
    // If not found or not a User model manager, check if it's an Employee with Manager role
    if (!manager || manager.role !== 'MANAGER') {
      logger.info('📱 [CLOSE JOB] Manager not found as User, checking Employee collection...');
      const employee = await Employee.findById(managerId).populate('roleId', 'name');
      
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
        logger.info('✅ [CLOSE JOB] Employee manager found', { employeeId: employee._id, email: employee.email });
        // Find corresponding User account with MANAGER role
        manager = await User.findOne({ 
          email: employee.email, 
          tenantId: tenantId,
          role: 'MANAGER' 
        });
        
        if (!manager) {
          logger.error('❌ [CLOSE JOB] Employee manager found but no corresponding User account');
          throw new Error('Manager User account not found. Please create a User account for this manager.');
        }
        logger.info('✅ [CLOSE JOB] Found corresponding User account for Employee manager', { userId: manager._id });
      } else {
        logger.error('❌ [CLOSE JOB] User is not a manager', { managerId, userRole, roleName });
        throw new Error('Only managers can close jobs');
      }
    } else {
      logger.info('✅ [CLOSE JOB] Manager found as User', { userId: manager._id, email: manager.email });
    }

    // Check if manager has access to this job (job must be assigned to one of manager's agents)
    // Jobs can be assigned to either User IDs or Employee IDs
    let hasAccess = false;
    
    if (manager.assignedAgents && manager.assignedAgents.length > 0) {
      // Get assigned User agents
      const assignedUserAgents = await User.find({
        _id: { $in: manager.assignedAgents },
        tenantId,
        role: 'SERVICE_AGENT',
      }).select('email');
      
      // Get emails of assigned User agents
      const assignedAgentEmails = assignedUserAgents.map(agent => agent.email);
      
      // Find corresponding Employee records for these agents (by email)
      const assignedEmployees = await Employee.find({
        email: { $in: assignedAgentEmails },
        tenantId,
      }).select('_id email');
      
      // Build combined list of agent IDs (both User IDs and Employee IDs)
      const agentUserIds = manager.assignedAgents.map(id => id.toString());
      const agentEmployeeIds = assignedEmployees.map(emp => emp._id.toString());
      const allAgentIds = [...agentUserIds, ...agentEmployeeIds];
      
      logger.info('👥 [CLOSE JOB] Manager assigned agents', { 
        userIds: agentUserIds, 
        employeeIds: agentEmployeeIds,
        jobAssignedAgentId: job.assignedAgentId?.toString() 
      });
      
      // Check if job's assignedAgentId matches any of the manager's agents
      const jobAgentIdStr = job.assignedAgentId?.toString();
      hasAccess = allAgentIds.includes(jobAgentIdStr);
      
      logger.info('🔍 [CLOSE JOB] Access check', { hasAccess, jobAgentId: jobAgentIdStr });
    } else {
      logger.warn('⚠️ [CLOSE JOB] Manager has no assigned agents');
    }
    
    if (!hasAccess) {
      logger.error('❌ [CLOSE JOB] Manager does not have access to this job');
      throw new Error('You do not have permission to close this job');
    }

    // Only allow closing from PENDING_APPROVAL status
    if (job.status !== 'PENDING_APPROVAL') {
      throw new Error('Job can only be closed from PENDING_APPROVAL status');
    }

    // Closure description is mandatory
    if (!closureDescription || closureDescription.trim().length === 0) {
      throw new Error('Closure description is required');
    }

    const oldStatus = job.status;
    job.status = 'CLOSED';
    job.closedAt = new Date();

    // Store closure description in job work details
    const JobWorkDetails = require('../models/JobWorkDetails.model');
    let workDetails = await JobWorkDetails.findOne({ jobId });
    if (!workDetails) {
      workDetails = new JobWorkDetails({ jobId });
    }
    workDetails.closureDescription = closureDescription.trim();
    await workDetails.save();

    await job.save();

    // Create activity log
    await createJobActivity({
      jobId,
      tenantId,
      activityType: 'STATUS_UPDATED',
      performedBy: managerId,
      performedByModel: 'User',
      performedByName: manager.name || manager.email,
      description: `Job closed by manager. Closure remarks: ${closureDescription.trim()}`,
      oldStatus,
      newStatus: 'CLOSED',
      metadata: {
        closureDescription: closureDescription.trim(),
        closedByManager: true,
      },
      timestamp: new Date(),
    });

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Close job by manager error:', error);
    throw error;
  }
};

/**
 * Get unassigned jobs for agents (jobs assigned to agent's manager but not to any agent)
 */
const getUnassignedJobsForAgent = async (filters, tenantId, userRole, userId, roleName = null) => {
  try {
    logger.info('🔵 [GET UNASSIGNED JOBS FOR AGENT] Starting', { userId, userRole, roleName });

    // Determine if user is an agent
    const isAgent = userRole === 'SERVICE_AGENT' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent');
    
    if (!isAgent) {
      throw new Error('Only agents can view unassigned jobs from their manager');
    }

    // Find agent's manager
    let agent = await User.findById(userId);
    let managerId = null;
    
    if (agent && agent.role === 'SERVICE_AGENT' && agent.managerId) {
      managerId = agent.managerId;
      logger.info('✅ [GET UNASSIGNED JOBS FOR AGENT] Found manager from User agent', { agentId: userId, managerId });
    } else {
      // Try Employee model
      const employee = await Employee.findById(userId).populate('roleId', 'name');
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
        logger.info('✅ [GET UNASSIGNED JOBS FOR AGENT] Found Employee agent', { employeeId: userId, employeeManagerId: employee.managerId });
        
        // First check Employee's managerId
        if (employee.managerId) {
          // Find the corresponding User account for this manager
          const managerEmployee = await Employee.findById(employee.managerId).populate('roleId', 'name');
          if (managerEmployee && managerEmployee.roleId && managerEmployee.roleId.name && managerEmployee.roleId.name.toLowerCase() === 'manager') {
            // Find User account for this manager
            const managerUser = await User.findOne({
              email: managerEmployee.email,
              tenantId,
              role: 'MANAGER',
            });
            if (managerUser) {
              managerId = managerUser._id;
              logger.info('✅ [GET UNASSIGNED JOBS FOR AGENT] Found manager User account from Employee manager', { managerId });
            } else {
              // Use Employee ID as fallback
              managerId = employee.managerId;
              logger.info('⚠️ [GET UNASSIGNED JOBS FOR AGENT] Using Employee manager ID as fallback', { managerId });
            }
          } else {
            managerId = employee.managerId;
          }
        } else {
          // Try to find User account for this agent and get managerId from there
          const agentUser = await User.findOne({
            email: employee.email,
            tenantId,
            role: 'SERVICE_AGENT',
          });
          if (agentUser && agentUser.managerId) {
            managerId = agentUser.managerId;
            logger.info('✅ [GET UNASSIGNED JOBS FOR AGENT] Found manager from User account of Employee agent', { managerId });
          }
        }
      }
    }

    if (!managerId) {
      logger.warn('⚠️ [GET UNASSIGNED JOBS FOR AGENT] Agent has no manager - returning empty list');
      return {
        jobs: [],
        pagination: {
          page: parseInt(filters.page) || 1,
          limit: parseInt(filters.limit) || 20,
          total: 0,
          pages: 0,
        },
      };
    }

    // Build query for ALL unassigned jobs (not filtered by managerId)
    // When agent takes a job, their manager will be attached to it
    const query = {
      tenantId,
      $or: [
        { assignedAgentId: { $exists: false } },
        { assignedAgentId: null }
      ],
    };

    // Apply filters
    if (filters.status) {
      if (filters.status === 'ALL') {
        // Don't filter by status
      } else {
        query.status = filters.status;
      }
    }
    if (filters.ticketType) {
      query.ticketType = filters.ticketType;
    }
    if (filters.fromDate || filters.toDate) {
      query.createdAt = {};
      if (filters.fromDate) {
        query.createdAt.$gte = new Date(filters.fromDate);
      }
      if (filters.toDate) {
        const toDate = new Date(filters.toDate);
        toDate.setHours(23, 59, 59, 999);
        query.createdAt.$lte = toDate;
      }
    }

    // Pagination
    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 20;
    const skip = (page - 1) * limit;

    // Get total count
    const total = await Job.countDocuments(query);

    // Get jobs
    const jobs = await Job.find(query)
      .populate('customerId', 'name email phone')
      .populate('managerId', 'name email')
      .populate('productId', 'name')
      .populate('complaintCategories', 'name')
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(limit);

    // Manually populate assignedAgentId for each job
    for (const job of jobs) {
      await populateAssignedAgent(job);
    }

    return {
      jobs,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    };
  } catch (error) {
    logger.error('Get unassigned jobs for agent error:', error);
    throw error;
  }
};

/**
 * Get unassigned jobs (jobs not assigned to any agent)
 * Managers can see all unassigned jobs and assign them to their agents
 */
const getUnassignedJobs = async (filters, tenantId, userRole, userId, roleName = null) => {
  try {
    logger.info('🔵 [GET UNASSIGNED JOBS] Starting', { userId, userRole, roleName });

    // Determine if user is a manager
    const isManager = userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager');
    
    if (!isManager) {
      throw new Error('Only managers can view unassigned jobs');
    }

    // Build query for ALL unassigned jobs (not filtered by managerId)
    // Jobs are no longer auto-assigned to managers at creation
    const query = {
      tenantId,
      $or: [
        { assignedAgentId: { $exists: false } },
        { assignedAgentId: null }
      ],
    };

    // Apply filters
    if (filters.status) {
      query.status = filters.status;
    }
    if (filters.ticketType) {
      query.ticketType = filters.ticketType;
    }
    if (filters.fromDate || filters.toDate) {
      query.createdAt = {};
      if (filters.fromDate) {
        query.createdAt.$gte = new Date(filters.fromDate);
      }
      if (filters.toDate) {
        const toDate = new Date(filters.toDate);
        toDate.setHours(23, 59, 59, 999);
        query.createdAt.$lte = toDate;
      }
    }

    // Pagination
    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 20;
    const skip = (page - 1) * limit;

    // Get total count
    const total = await Job.countDocuments(query);

    // Get jobs
    const jobs = await Job.find(query)
      .populate('customerId', 'name email phone')
      .populate('managerId', 'name email')
      .populate('productId', 'name')
      .populate('complaintCategories', 'name')
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(limit);

    return {
      jobs,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    };
  } catch (error) {
    logger.error('Get unassigned jobs error:', error);
    throw error;
  }
};

/**
 * Assign job to agent (by manager)
 */
const assignJobToAgent = async (jobId, agentId, managerId, tenantId, userRole, roleName = null) => {
  try {
    logger.info('🔵 [ASSIGN JOB TO AGENT] Starting', { jobId, agentId, managerId, userRole, roleName });

    // Verify manager
    const isManager = userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager');
    if (!isManager) {
      throw new Error('Only managers can assign jobs to agents');
    }

    // Find manager
    let manager = await User.findById(managerId);
    if (!manager || manager.role !== 'MANAGER') {
      const employee = await Employee.findById(managerId).populate('roleId', 'name');
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
        manager = await User.findOne({ 
          email: employee.email, 
          tenantId, 
          role: 'MANAGER' 
        });
        if (!manager) {
          throw new Error('Manager User account not found');
        }
      } else {
        throw new Error('Manager not found');
      }
    }

    // Get job
    const job = await Job.findOne({ _id: jobId, tenantId });
    if (!job) {
      throw new Error('Job not found');
    }

    // Verify job is not already assigned to an agent
    if (job.assignedAgentId) {
      throw new Error('Job is already assigned to an agent');
    }

    // Verify agent is assigned to this manager
    const agentUser = await User.findById(agentId);
    if (!agentUser) {
      // Try Employee model
      const agentEmployee = await Employee.findById(agentId).populate('roleId', 'name');
      if (!agentEmployee) {
        throw new Error('Agent not found');
      }
      
      // Check if agent is assigned to manager
      if (agentEmployee.managerId?.toString() !== managerId.toString()) {
        // Check via User model
        const agentUserByEmail = await User.findOne({ 
          email: agentEmployee.email, 
          tenantId, 
          role: 'SERVICE_AGENT' 
        });
        if (!agentUserByEmail || agentUserByEmail.managerId?.toString() !== manager._id.toString()) {
          throw new Error('Agent is not assigned to this manager');
        }
        // Use the User account for assignment
        job.assignedAgentId = agentUserByEmail._id;
        job.assignedAgentModel = 'User';
      } else {
        // Use Employee ID for assignment
        job.assignedAgentId = agentEmployee._id;
        job.assignedAgentModel = 'Employee';
      }
    } else {
      // Verify agent is assigned to this manager
      if (agentUser.managerId?.toString() !== manager._id.toString()) {
        throw new Error('Agent is not assigned to this manager');
      }
      job.assignedAgentId = agentUser._id;
      job.assignedAgentModel = 'User';
    }

    // Set managerId when manager assigns the job
    job.managerId = manager._id;
    job.status = 'NEW'; // Ensure status is NEW when assigned
    await job.save();

    // Create activity log
    const agentName = agentUser?.name || `${agentUser?.firstName || ''} ${agentUser?.lastName || ''}`.trim();
    await createJobActivity({
      jobId: job._id,
      tenantId,
      activityType: 'JOB_ASSIGNED',
      performedBy: managerId,
      performedByModel: 'User',
      performedByName: manager.name || manager.email,
      description: `Job assigned to agent: ${agentName || agentId}`,
    });

    // Create notification for agent
    const Notification = require('../models/Notification.model');
    const agentNotification = new Notification({
      tenantId,
      userId: job.assignedAgentId,
      jobId: job._id,
      title: 'New Job Assigned',
      message: `You have been assigned a new job: ${job.ticketNumber || job._id}`,
      notificationType: 'JOB_ASSIGNED',
    });
    await agentNotification.save();

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Assign job to agent error:', error);
    throw error;
  }
};

/**
 * Agent self-assignment (agent can take jobs assigned to their manager)
 */
const selfAssignJob = async (jobId, agentId, tenantId, userRole, roleName = null) => {
  try {
    logger.info('🔵 [SELF ASSIGN JOB] Starting', { jobId, agentId, userRole, roleName });

    // Verify agent
    const isAgent = userRole === 'SERVICE_AGENT' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent');
    if (!isAgent) {
      throw new Error('Only agents can self-assign jobs');
    }

    // Get job
    const job = await Job.findOne({ _id: jobId, tenantId });
    if (!job) {
      throw new Error('Job not found');
    }

    // Verify job is not already assigned
    if (job.assignedAgentId) {
      throw new Error('Job is already assigned to an agent');
    }

    // Find agent (User or Employee)
    let agent = await User.findById(agentId);
    let agentModel = 'User';
    
    if (!agent || agent.role !== 'SERVICE_AGENT') {
      const employee = await Employee.findById(agentId).populate('roleId', 'name');
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'agent') {
        agent = await User.findOne({ 
          email: employee.email, 
          tenantId, 
          role: 'SERVICE_AGENT' 
        });
        if (!agent) {
          // Use Employee ID if no User account
          agent = employee;
          agentModel = 'Employee';
        }
      } else {
        throw new Error('Agent not found');
      }
    }

    // Get agent's manager ID
    let agentManagerId = null;
    if (agentModel === 'User') {
      agentManagerId = agent.managerId;
    } else {
      // For Employee, get managerId from Employee record
      const employee = await Employee.findById(agentId);
      if (employee && employee.managerId) {
        // Try to find User account for this manager
        const managerEmployee = await Employee.findById(employee.managerId).populate('roleId', 'name');
        if (managerEmployee && managerEmployee.roleId && managerEmployee.roleId.name && managerEmployee.roleId.name.toLowerCase() === 'manager') {
          const managerUser = await User.findOne({
            email: managerEmployee.email,
            tenantId,
            role: 'MANAGER',
          });
          if (managerUser) {
            agentManagerId = managerUser._id;
          } else {
            // Use Employee ID as fallback
            agentManagerId = employee.managerId;
          }
        } else {
          agentManagerId = employee.managerId;
        }
      } else {
        // Try to find User account for this agent and get managerId from there
        const agentUser = await User.findOne({
          email: employee.email,
          tenantId,
          role: 'SERVICE_AGENT',
        });
        if (agentUser && agentUser.managerId) {
          agentManagerId = agentUser.managerId;
        }
      }
    }

    if (!agentManagerId) {
      throw new Error('Agent does not have a manager assigned. Please contact administrator.');
    }

    // Assign job to agent and attach manager
    job.assignedAgentId = agent._id;
    job.assignedAgentModel = agentModel;
    job.managerId = agentManagerId; // Set managerId when agent takes the job
    job.status = 'NEW';
    await job.save();

    // Create activity log
    const agentName = agent.name || `${agent.firstName || ''} ${agent.lastName || ''}`.trim();
    await createJobActivity({
      jobId: job._id,
      tenantId,
      activityType: 'JOB_ASSIGNED',
      performedBy: agentId,
      performedByModel: agentModel,
      performedByName: agentName || agent.email,
      description: 'Job self-assigned by agent',
    });

    // Create notification for manager
    const Notification = require('../models/Notification.model');
    const managerNotification = new Notification({
      tenantId,
      userId: job.managerId,
      jobId: job._id,
      title: 'Job Self-Assigned',
      message: `Agent ${agentName || agent.email} has self-assigned job: ${job.ticketNumber || job._id}`,
      notificationType: 'JOB_ASSIGNED',
    });
    await managerNotification.save();

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Self assign job error:', error);
    throw error;
  }
};

/**
 * Sign job (for managers and customers)
 * @param {string} jobId - Job ID
 * @param {string} userId - User ID (manager or customer)
 * @param {string} tenantId - Tenant ID
 * @param {object} signatureData - Signature data (signatureUrl, signatureData, signeeName)
 * @param {string} userRole - User role (MANAGER or CUSTOMER)
 * @param {string} timestamp - Timestamp from mobile device (ISO string)
 * @param {string} roleName - Role name from Employee collection (optional)
 */
const signJob = async (jobId, userId, tenantId, signatureData, userRole, timestamp, roleName = null) => {
  try {
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    // Check if job is in SIGNATURE_PENDING status
    if (job.status !== 'SIGNATURE_PENDING') {
      throw new Error('Job is not pending signature');
    }

    // Verify access based on role
    if (userRole === 'CUSTOMER') {
      // Customer can only sign their own jobs
      // job.customerId should match userId (both are ObjectIds)
      if (job.customerId?.toString() !== userId.toString()) {
        throw new Error('You can only sign your own jobs');
      }
    } else if (userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager')) {
      // Manager can sign jobs assigned to their agents
      const manager = await User.findById(userId);
      if (!manager || manager.role !== 'MANAGER') {
        // Check if it's an Employee with Manager role
        const employee = await Employee.findById(userId).populate('roleId', 'name');
        if (!employee || !employee.roleId || employee.roleId.name.toLowerCase() !== 'manager') {
          throw new Error('Only managers can sign jobs');
        }
      }
      
      // Verify manager has access to this job (job should be assigned to one of their agents)
      if (job.managerId?.toString() !== userId.toString()) {
        // Check if job is assigned to one of manager's agents
        const managerUser = await User.findById(userId);
        if (managerUser && managerUser.assignedAgents && managerUser.assignedAgents.length > 0) {
          const agentIds = managerUser.assignedAgents.map(id => id.toString());
          if (!agentIds.includes(job.assignedAgentId?.toString())) {
            throw new Error('You can only sign jobs assigned to your agents');
          }
        } else {
          throw new Error('You can only sign jobs assigned to your agents');
        }
      }
    } else {
      throw new Error('Only managers and customers can sign jobs');
    }

    // Get signer details
    let signer;
    let signerName;
    let performedByModel = 'User';
    
    if (userRole === 'CUSTOMER') {
      // Check both User model (with role CUSTOMER) and Customer model
      signer = await User.findById(userId);
      if (signer) {
        signerName = signer.name || signer.email;
      } else {
        // Try Customer model
        const Customer = require('../models/Customer.model');
        const customer = await Customer.findById(userId);
        if (customer) {
          signer = customer;
          signerName = customer.name || customer.email;
          performedByModel = 'Customer';
        }
      }
    } else if (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager') {
      signer = await Employee.findById(userId);
      if (signer) {
        signerName = `${signer.firstName} ${signer.lastName}`;
        performedByModel = 'Employee';
      }
    } else {
      signer = await User.findById(userId);
      if (signer) {
        signerName = signer.name || signer.email;
      }
    }

    if (!signer) {
      throw new Error('Signer not found');
    }

    // Validate signature data
    if (!signatureData.signatureUrl && !signatureData.signatureData) {
      throw new Error('Signature is required');
    }

    // Update job signature
    if (signatureData.signatureUrl) {
      job.signatureUrl = signatureData.signatureUrl;
      job.isSigned = true;
    }
    if (signatureData.signatureData) {
      job.signatureData = signatureData.signatureData;
      job.isSigned = true;
    }
    if (signatureData.signeeName) {
      job.signeeName = signatureData.signeeName;
    }

    // Update status from SIGNATURE_PENDING to PENDING_APPROVAL
    job.status = 'PENDING_APPROVAL';
    job.isSigned = true;

    // Update JobWorkDetails
    let workDetails = await JobWorkDetails.findOne({ jobId });
    if (!workDetails) {
      workDetails = new JobWorkDetails({ jobId });
    }
    
    if (signatureData.signatureUrl) {
      workDetails.customerDigitalSignatureUrl = signatureData.signatureUrl;
    }
    if (signatureData.signatureData) {
      workDetails.customerSignatureData = signatureData.signatureData;
    }
    if (signatureData.signeeName) {
      workDetails.signeeName = signatureData.signeeName;
    }

    await workDetails.save();
    await job.save();

    // Create activity log
    const signerType = userRole === 'CUSTOMER' ? 'Customer' : 'Manager';
    await createJobActivity({
      jobId,
      tenantId,
      activityType: 'JOB_SIGNED',
      performedBy: userId,
      performedByModel,
      performedByName: signerName,
      description: `Job signed by ${signerType}${signatureData.signeeName ? ` - ${signatureData.signeeName}` : ''}`,
      metadata: {
        hasSignature: true,
        signeeName: signatureData.signeeName || signerName,
        signedBy: signerType,
      },
      timestamp: timestamp ? new Date(timestamp) : new Date(),
    });

    await job.populate(['customerId', 'managerId', 'productId', 'complaintCategories']);
    await populateAssignedAgent(job);
    return job;
  } catch (error) {
    logger.error('Sign job error:', error);
    throw error;
  }
};

/**
 * Get jobs with pending signatures (isSigned: false)
 */
const getPendingSignatureJobs = async (filters, tenantId, userRole, userId, roleName = null) => {
  try {
    logger.info('🔵 [GET PENDING SIGNATURE JOBS] Starting', { userId, userRole, roleName });

    const query = { 
      tenantId,
      status: 'SIGNATURE_PENDING', // Only show SIGNATURE_PENDING jobs
    };

    // Determine if user is a manager, agent, or customer
    const isManager = userRole === 'MANAGER' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'manager');
    const isAgent = userRole === 'SERVICE_AGENT' || (userRole === 'EMPLOYEE' && roleName && roleName.toLowerCase() === 'agent');
    const isCustomer = userRole === 'CUSTOMER';

    // Customer isolation - only see their own jobs
    if (isCustomer) {
      query.customerId = userId;
    } else if (isManager) {
      // Manager isolation - only see assigned agents' jobs
      let manager = await User.findById(userId);
      if (!manager || manager.role !== 'MANAGER') {
        const employee = await Employee.findById(userId).populate('roleId', 'name');
        if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
          manager = await User.findOne({ 
            email: employee.email, 
            tenantId, 
            role: 'MANAGER' 
          });
        }
      }
      
      if (manager && manager.assignedAgents && manager.assignedAgents.length > 0) {
        const assignedUserAgents = await User.find({
          _id: { $in: manager.assignedAgents },
          tenantId,
          role: 'SERVICE_AGENT',
        }).select('email');
        
        const assignedAgentEmails = assignedUserAgents.map(agent => agent.email);
        const assignedEmployees = await Employee.find({
          email: { $in: assignedAgentEmails },
          tenantId,
        }).select('_id email');
        
        const agentUserIds = manager.assignedAgents.map(id => id.toString());
        const agentEmployeeIds = assignedEmployees.map(emp => emp._id.toString());
        const allAgentIds = [...agentUserIds, ...agentEmployeeIds];
        
        query.assignedAgentId = { $in: allAgentIds };
      } else {
        return {
          jobs: [],
          pagination: {
            page: parseInt(filters.page) || 1,
            limit: parseInt(filters.limit) || 10,
            total: 0,
            pages: 0,
          },
        };
      }
    } else if (isAgent) {
      // Agent - only see own jobs
      query.assignedAgentId = userId;
    }

    // Apply filters
    if (filters.ticketType) {
      query.ticketType = filters.ticketType;
    }
    if (filters.fromDate || filters.toDate) {
      query.createdAt = {};
      if (filters.fromDate) {
        query.createdAt.$gte = new Date(filters.fromDate);
      }
      if (filters.toDate) {
        const toDate = new Date(filters.toDate);
        toDate.setHours(23, 59, 59, 999);
        query.createdAt.$lte = toDate;
      }
    }

    // Pagination
    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 20;
    const skip = (page - 1) * limit;

    // Get total count
    const total = await Job.countDocuments(query);

    // Get jobs
    const jobs = await Job.find(query)
      .populate('customerId', 'name email phone')
      .populate('managerId', 'name email')
      .populate('productId', 'name')
      .populate('complaintCategories', 'name')
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(limit);

    // Manually populate assignedAgentId for each job
    for (const job of jobs) {
      await populateAssignedAgent(job);
    }

    return {
      jobs,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    };
  } catch (error) {
    logger.error('Get pending signature jobs error:', error);
    throw error;
  }
};

module.exports = {
  signJob,
  createJob,
  getJobById,
  getJobs,
  acceptJob,
  startJob,
  submitJob,
  arriveJob,
  updateJobStatus,
  markJobOpened,
  getJobTimeline,
  getPendingJobsCount,
  assignAgentToJob,
  createJobActivity,
  closeJobByManager,
  getUnassignedJobs,
  getUnassignedJobsForAgent,
  assignJobToAgent,
  selfAssignJob,
  getPendingSignatureJobs,
};

