/**
 * Employee Service
 * Business logic for employee management
 */

const Employee = require('../models/Employee.model');
const Role = require('../models/Role.model');
const { hashPassword } = require('../utils/password');
const logger = require('../utils/logger');

/**
 * Get employees for a company
 */
const getEmployees = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };

    if (filters.status) {
      query.status = filters.status;
    }

    if (filters.role) {
      query.role = filters.role;
    }

    // Filter by role name (e.g., "Agent", "Manager")
    // Match exactly by role name (case-insensitive)
    if (filters.roleName) {
      const role = await Role.findOne({ 
        tenantId, 
        name: { $regex: new RegExp(`^${filters.roleName}$`, 'i') }, // Exact match, case-insensitive
        status: 'ACTIVE'
      });
      if (role) {
        query.roleId = role._id;
        logger.info(`Filtering employees by role: ${filters.roleName}, roleId: ${role._id}`);
      } else {
        // If role not found, return empty array
        logger.warn(`Role not found: ${filters.roleName} for tenant: ${tenantId}`);
        return [];
      }
    }

    if (filters.search) {
      query.$or = [
        { firstName: { $regex: filters.search, $options: 'i' } },
        { lastName: { $regex: filters.search, $options: 'i' } },
        { email: { $regex: filters.search, $options: 'i' } },
        { employeeId: { $regex: filters.search, $options: 'i' } },
      ];
    }

    const employees = await Employee.find(query)
      .populate('assignedClients', 'name email')
      .populate('roleId', 'name description permissions')
      .sort({ createdAt: -1 });

    return employees;
  } catch (error) {
    logger.error('Get employees error:', error);
    throw error;
  }
};

/**
 * Get employee by ID
 */
const getEmployeeById = async (employeeId, tenantId) => {
  try {
    const employee = await Employee.findOne({ _id: employeeId, tenantId })
      .populate('assignedClients', 'name email')
      .populate('assignedJobs', 'jobNumber status')
      .populate('roleId', 'name description permissions');

    if (!employee) {
      throw new Error('Employee not found');
    }

    return employee;
  } catch (error) {
    logger.error('Get employee by ID error:', error);
    throw error;
  }
};

/**
 * Create employee
 */
const createEmployee = async (employeeData, tenantId, createdBy) => {
  try {
    // Check if email already exists
    const existingEmployee = await Employee.findOne({ email: employeeData.email.toLowerCase().trim() });
    if (existingEmployee) {
      throw new Error('Employee with this email already exists');
    }

    // Validate role exists
    if (!employeeData.roleId) {
      throw new Error('Role is required');
    }

    const role = await Role.findOne({ _id: employeeData.roleId, tenantId });
    if (!role) {
      throw new Error('Invalid role selected');
    }

    const employee = new Employee({
      ...employeeData,
      tenantId,
      createdBy,
      email: employeeData.email.toLowerCase().trim(),
    });

    await employee.save();
    
    // Populate role before returning
    await employee.populate('roleId', 'name description permissions');
    
    // Sync to User model if role is Manager
    await syncEmployeeToUser(employee, tenantId, employeeData.password);
    
    logger.info(`Employee created: ${employee.employeeId}`, { employeeId: employee._id, tenantId });

    return employee;
  } catch (error) {
    logger.error('Create employee error:', error);
    throw error;
  }
};

/**
 * Sync Employee to User model (for Manager role)
 * Creates or updates a User account when employee has Manager role
 */
const syncEmployeeToUser = async (employee, tenantId, newPassword = null) => {
  try {
    const User = require('../models/User.model');
    const role = await Role.findById(employee.roleId);
    
    // Only sync if role name contains "Manager" (case-insensitive)
    if (!role || !role.name.toLowerCase().includes('manager')) {
      return null; // Not a manager, no need to sync
    }

    // Check if User account already exists
    let user = await User.findOne({ email: employee.email, tenantId });
    
    if (user) {
      // Update existing user
      user.name = `${employee.firstName} ${employee.lastName}`;
      user.phone = employee.contactNumber;
      user.role = 'MANAGER';
      user.status = employee.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE';
      
      // Update password if new password provided
      if (newPassword && newPassword.trim() !== '') {
        const { hashPassword } = require('../utils/password');
        user.password = await hashPassword(newPassword);
      }
      
      await user.save();
      logger.info(`User account updated for manager employee: ${employee.email}`);
    } else {
      // Create new user account
      // Use provided password or employee's password
      const { hashPassword } = require('../utils/password');
      const passwordToUse = newPassword || 'TempPassword123!';
      const hashedPassword = await hashPassword(passwordToUse);
      
      user = new User({
        email: employee.email,
        password: hashedPassword,
        name: `${employee.firstName} ${employee.lastName}`,
        phone: employee.contactNumber,
        role: 'MANAGER',
        status: employee.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE',
        tenantId: tenantId,
      });
      
      await user.save();
      logger.info(`User account created for manager employee: ${employee.email}`);
    }
    
    return user;
  } catch (error) {
    logger.error('Sync employee to user error:', error);
    // Don't throw error, just log it - employee update should still succeed
    return null;
  }
};

/**
 * Update employee
 */
const updateEmployee = async (employeeId, employeeData, tenantId) => {
  try {
    const employee = await Employee.findOne({ _id: employeeId, tenantId });

    if (!employee) {
      throw new Error('Employee not found');
    }

    // If email is being updated, check for duplicates
    if (employeeData.email && employeeData.email.toLowerCase().trim() !== employee.email) {
      const existingEmployee = await Employee.findOne({ 
        email: employeeData.email.toLowerCase().trim(),
        _id: { $ne: employeeId }
      });
      if (existingEmployee) {
        throw new Error('Employee with this email already exists');
      }
      employeeData.email = employeeData.email.toLowerCase().trim();
    }

    // Track if password is being updated
    let passwordUpdated = false;
    let newPassword = null;
    
    // Update password if provided
    if (employeeData.password && employeeData.password.trim() !== '') {
      newPassword = employeeData.password;
      employee.password = employeeData.password; // Will be hashed by pre-save hook
      employee.markModified('password');
      passwordUpdated = true;
      delete employeeData.password;
    }

    // Validate role if being updated
    let roleChanged = false;
    if (employeeData.roleId && employeeData.roleId.toString() !== employee.roleId.toString()) {
      const role = await Role.findOne({ _id: employeeData.roleId, tenantId });
      if (!role) {
        throw new Error('Invalid role selected');
      }
      roleChanged = true;
    }

    // Update other fields
    Object.assign(employee, employeeData);
    await employee.save();
    
    // Populate role before returning
    await employee.populate('roleId', 'name description permissions');

    // Sync to User model if role is Manager (always check after update)
    await syncEmployeeToUser(employee, tenantId, passwordUpdated ? newPassword : null);

    logger.info(`Employee updated: ${employee.employeeId}`, { employeeId: employee._id, tenantId });

    return employee;
  } catch (error) {
    logger.error('Update employee error:', error);
    throw error;
  }
};

/**
 * Delete employee
 */
const deleteEmployee = async (employeeId, tenantId) => {
  try {
    const employee = await Employee.findOneAndDelete({ _id: employeeId, tenantId });

    if (!employee) {
      throw new Error('Employee not found');
    }

    logger.info(`Employee deleted: ${employee.employeeId}`, { employeeId: employee._id, tenantId });

    return { success: true };
  } catch (error) {
    logger.error('Delete employee error:', error);
    throw error;
  }
};

/**
 * Assign clients to employee
 */
const assignClients = async (employeeId, clientIds, tenantId) => {
  try {
    const employee = await Employee.findOne({ _id: employeeId, tenantId });

    if (!employee) {
      throw new Error('Employee not found');
    }

    employee.assignedClients = clientIds;
    await employee.save();

    logger.info(`Clients assigned to employee: ${employee.employeeId}`, { employeeId: employee._id, clientIds });

    return employee;
  } catch (error) {
    logger.error('Assign clients error:', error);
    throw error;
  }
};

/**
 * Assign jobs to employee
 */
const assignJobs = async (employeeId, jobIds, tenantId) => {
  try {
    const employee = await Employee.findOne({ _id: employeeId, tenantId });

    if (!employee) {
      throw new Error('Employee not found');
    }

    employee.assignedJobs = jobIds;
    await employee.save();

    logger.info(`Jobs assigned to employee: ${employee.employeeId}`, { employeeId: employee._id, jobIds });

    return employee;
  } catch (error) {
    logger.error('Assign jobs error:', error);
    throw error;
  }
};

/**
 * Create or sync User account for employee (for Manager role)
 * This can be called manually from the frontend if User account doesn't exist
 */
const createUserAccountForEmployee = async (employeeId, tenantId) => {
  try {
    const employee = await Employee.findOne({ _id: employeeId, tenantId });
    
    if (!employee) {
      throw new Error('Employee not found');
    }
    
    await employee.populate('roleId', 'name description permissions');
    const user = await syncEmployeeToUser(employee, tenantId);
    
    if (!user) {
      throw new Error('Employee role is not Manager. User account can only be created for Manager role.');
    }
    
    return user;
  } catch (error) {
    logger.error('Create user account for employee error:', error);
    throw error;
  }
};

module.exports = {
  getEmployees,
  getEmployeeById,
  createEmployee,
  updateEmployee,
  deleteEmployee,
  assignClients,
  assignJobs,
  createUserAccountForEmployee,
};

