/**
 * Email Service
 * Handles email notifications using nodemailer
 */

const nodemailer = require('nodemailer');
const logger = require('../utils/logger');

let transporter = null;

/**
 * Initialize email transporter
 */
const initializeEmailService = () => {
  try {
    transporter = nodemailer.createTransport({
      host: process.env.EMAIL_HOST || 'smtp.sendgrid.net',
      port: parseInt(process.env.EMAIL_PORT) || 587,
      secure: false, // true for 465, false for other ports
      auth: {
        user: process.env.EMAIL_USER || 'apikey',
        pass: process.env.EMAIL_PASSWORD,
      },
    });

    logger.info('Email service initialized');
  } catch (error) {
    logger.error('Failed to initialize email service:', error);
  }
};

/**
 * Send email
 */
const sendEmail = async (to, subject, html, text = null) => {
  try {
    if (!transporter) {
      initializeEmailService();
    }

    if (!transporter) {
      logger.warn('Email service not configured, skipping email send');
      return { success: false, message: 'Email service not configured' };
    }

    const mailOptions = {
      from: process.env.EMAIL_FROM || 'noreply@hrq.com',
      to,
      subject,
      html,
      text: text || html.replace(/<[^>]*>/g, ''), // Strip HTML for text version
    };

    const info = await transporter.sendMail(mailOptions);
    logger.info(`Email sent successfully to ${to}: ${info.messageId}`);
    
    return { success: true, messageId: info.messageId };
  } catch (error) {
    logger.error('Error sending email:', error);
    return { success: false, error: error.message };
  }
};

/**
 * Send job assignment notification
 */
const sendJobAssignmentEmail = async (user, job) => {
  const subject = `New Job Assigned: ${job.ticketNumber}`;
  const html = `
    <h2>New Job Assigned</h2>
    <p>Hello ${user.name},</p>
    <p>A new job has been assigned to you:</p>
    <ul>
      <li><strong>Ticket Number:</strong> ${job.ticketNumber}</li>
      <li><strong>Type:</strong> ${job.ticketType}</li>
      <li><strong>Customer:</strong> ${job.customerName}</li>
      <li><strong>Location:</strong> ${job.jobLocationAddress}</li>
      <li><strong>SLA Deadline:</strong> ${new Date(job.slaDeadline).toLocaleString()}</li>
    </ul>
    <p>Please log in to accept and start the job.</p>
  `;

  return await sendEmail(user.email, subject, html);
};

/**
 * Send approval required notification
 */
const sendApprovalRequiredEmail = async (manager, job) => {
  const subject = `Approval Required: ${job.ticketNumber}`;
  const html = `
    <h2>Job Approval Required</h2>
    <p>Hello ${manager.name},</p>
    <p>A job is pending your approval:</p>
    <ul>
      <li><strong>Ticket Number:</strong> ${job.ticketNumber}</li>
      <li><strong>Agent:</strong> ${job.assignedAgentId?.name || 'N/A'}</li>
      <li><strong>Customer:</strong> ${job.customerName}</li>
      <li><strong>Completed At:</strong> ${new Date(job.completedAt).toLocaleString()}</li>
    </ul>
    <p>Please log in to review and approve the job.</p>
  `;

  return await sendEmail(manager.email, subject, html);
};

/**
 * Send report delivery email
 */
const sendReportEmail = async (customer, report, job) => {
  const subject = `Service Report: ${job.ticketNumber}`;
  const html = `
    <h2>Service Report Available</h2>
    <p>Hello ${customer.name},</p>
    <p>Your service report is ready for download:</p>
    <ul>
      <li><strong>Report Number:</strong> ${report.reportNumber}</li>
      <li><strong>Ticket Number:</strong> ${job.ticketNumber}</li>
      <li><strong>Service Date:</strong> ${new Date(job.completedAt).toLocaleDateString()}</li>
    </ul>
    <p><a href="${process.env.API_BASE_URL || 'http://localhost:3000'}${report.pdfFileUrl}">Download Report</a></p>
    <p>Thank you for using our services!</p>
  `;

  return await sendEmail(customer.email, subject, html);
};

/**
 * Send SLA breach notification
 */
const sendSLABreachEmail = async (user, job) => {
  const subject = `URGENT: SLA Breached - ${job.ticketNumber}`;
  const html = `
    <h2>SLA Breach Alert</h2>
    <p>Hello ${user.name},</p>
    <p><strong>URGENT:</strong> The following job has breached its SLA deadline:</p>
    <ul>
      <li><strong>Ticket Number:</strong> ${job.ticketNumber}</li>
      <li><strong>Customer:</strong> ${job.customerName}</li>
      <li><strong>SLA Deadline:</strong> ${new Date(job.slaDeadline).toLocaleString()}</li>
      <li><strong>Priority:</strong> ${job.priority || 'N/A'}</li>
    </ul>
    <p>Immediate action is required. Please log in to address this issue.</p>
  `;

  return await sendEmail(user.email, subject, html);
};

module.exports = {
  initializeEmailService,
  sendEmail,
  sendJobAssignmentEmail,
  sendApprovalRequiredEmail,
  sendReportEmail,
  sendSLABreachEmail,
};

