/**
 * Dashboard Service
 */

const Job = require('../models/Job.model');
const User = require('../models/User.model');
const Tenant = require('../models/Tenant.model');
const Notification = require('../models/Notification.model');
const KPIScore = require('../models/KPIScore.model');
const logger = require('../utils/logger');
const { isSLABreached } = require('../utils/sla');

/**
 * Get dashboard stats (Admin/Super Admin)
 */
const getAdminDashboardStats = async (tenantId) => {
  try {
    const [
      totalUsers,
      totalJobs,
      pendingApprovals,
      activeAgents,
      slaBreaches,
      completedJobs,
    ] = await Promise.all([
      User.countDocuments({ tenantId, status: 'ACTIVE' }),
      Job.countDocuments({ tenantId }),
      Job.countDocuments({ tenantId, status: 'PENDING_APPROVAL' }),
      User.countDocuments({ tenantId, role: 'SERVICE_AGENT', status: 'ACTIVE' }),
      Job.countDocuments({ tenantId, slaBreached: true, status: { $ne: 'CLOSED' } }),
      Job.countDocuments({ tenantId, status: 'CLOSED' }),
    ]);

    // Get recent jobs
    const recentJobs = await Job.find({ tenantId })
      .populate('customerId', 'name')
      .populate('assignedAgentId', 'name')
      .sort({ createdAt: -1 })
      .limit(10);

    return {
      totalUsers,
      totalJobs,
      pendingApprovals,
      activeAgents,
      slaBreaches,
      completedJobs,
      recentJobs,
    };
  } catch (error) {
    logger.error('Get admin dashboard stats error:', error);
    throw error;
  }
};

/**
 * Get manager dashboard stats
 */
const getManagerDashboardStats = async (managerId, tenantId) => {
  try {
    const manager = await User.findById(managerId);
    if (!manager || !manager.assignedAgents || manager.assignedAgents.length === 0) {
      return {
        pendingApprovals: 0,
        teamAgents: 0,
        teamJobs: 0,
        slaBreaches: 0,
        teamPerformance: [],
      };
    }

    const [
      pendingApprovals,
      teamJobs,
      slaBreaches,
    ] = await Promise.all([
      Job.countDocuments({
        tenantId,
        managerId,
        assignedAgentId: { $in: manager.assignedAgents },
        status: 'PENDING_APPROVAL',
      }),
      Job.countDocuments({
        tenantId,
        managerId,
        assignedAgentId: { $in: manager.assignedAgents },
      }),
      Job.countDocuments({
        tenantId,
        managerId,
        assignedAgentId: { $in: manager.assignedAgents },
        slaBreached: true,
        status: { $ne: 'CLOSED' },
      }),
    ]);

    // Get team performance (KPI scores)
    const teamPerformance = await KPIScore.find({
      tenantId,
      userId: { $in: manager.assignedAgents },
      role: 'SERVICE_AGENT',
    })
      .populate('userId', 'name email')
      .sort({ scoreDate: -1, totalScore: -1 })
      .limit(10);

    return {
      pendingApprovals,
      teamAgents: manager.assignedAgents.length,
      teamJobs,
      slaBreaches,
      teamPerformance,
    };
  } catch (error) {
    logger.error('Get manager dashboard stats error:', error);
    throw error;
  }
};

/**
 * Get agent dashboard stats
 */
const getAgentDashboardStats = async (agentId, tenantId) => {
  try {
    const [
      newJobs,
      acceptedJobs,
      inProgressJobs,
      pendingJobs,
      completedJobs,
      myKPI,
      unreadNotifications,
    ] = await Promise.all([
      Job.countDocuments({ tenantId, assignedAgentId: agentId, status: 'NEW' }),
      Job.countDocuments({ tenantId, assignedAgentId: agentId, status: 'ACCEPTED' }),
      Job.countDocuments({ tenantId, assignedAgentId: agentId, status: 'IN_PROGRESS' }),
      Job.countDocuments({ tenantId, assignedAgentId: agentId, status: 'PENDING_APPROVAL' }),
      Job.countDocuments({ tenantId, assignedAgentId: agentId, status: 'CLOSED' }),
      KPIScore.findOne({
        tenantId,
        userId: agentId,
        role: 'SERVICE_AGENT',
      }).sort({ scoreDate: -1 }),
      Notification.countDocuments({
        tenantId,
        userId: agentId,
        isRead: false,
      }),
    ]);

    // Get my jobs queue
    const myJobs = await Job.find({
      tenantId,
      assignedAgentId: agentId,
      status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS', 'PENDING_APPROVAL'] },
    })
      .populate('customerId', 'name phone')
      .sort({ createdAt: -1 })
      .limit(20);

    return {
      newJobs,
      acceptedJobs,
      inProgressJobs,
      pendingJobs,
      completedJobs,
      myKPI: myKPI || { totalScore: 0 },
      unreadNotifications,
      myJobs,
    };
  } catch (error) {
    logger.error('Get agent dashboard stats error:', error);
    throw error;
  }
};

/**
 * Get customer dashboard stats
 */
const getCustomerDashboardStats = async (customerId, tenantId) => {
  try {
    const [
      activeRequests,
      completedServices,
      pendingSignature,
      upcomingMaintenance,
    ] = await Promise.all([
      Job.countDocuments({
        tenantId,
        customerId,
        status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS', 'PENDING_APPROVAL'] },
      }),
      Job.countDocuments({
        tenantId,
        customerId,
        status: 'CLOSED',
      }),
      Job.countDocuments({
        tenantId,
        customerId,
        status: 'PENDING_APPROVAL',
      }),
      Job.countDocuments({
        tenantId,
        customerId,
        ticketType: 'PERIODIC',
        status: { $in: ['NEW', 'ACCEPTED'] },
        scheduledDate: { $gte: new Date() },
      }),
    ]);

    // Get service history
    const serviceHistory = await Job.find({
      tenantId,
      customerId,
      status: 'CLOSED',
    })
      .populate('assignedAgentId', 'name')
      .sort({ closedAt: -1 })
      .limit(10);

    return {
      activeRequests,
      completedServices,
      pendingSignature,
      upcomingMaintenance,
      serviceHistory,
    };
  } catch (error) {
    logger.error('Get customer dashboard stats error:', error);
    throw error;
  }
};

/**
 * Get super admin dashboard stats (all tenants)
 */
const getSuperAdminDashboardStats = async () => {
  try {
    const [
      totalCompanies,
      activeCompanies,
      totalUsers,
      pendingActivations,
      totalJobs,
      activeJobs,
    ] = await Promise.all([
      Tenant.countDocuments(),
      Tenant.countDocuments({ isActive: true, status: 'ACTIVE' }),
      User.countDocuments({ status: 'ACTIVE' }),
      Tenant.countDocuments({ status: 'INACTIVE' }),
      Job.countDocuments(),
      Job.countDocuments({ status: { $in: ['NEW', 'ACCEPTED', 'IN_PROGRESS'] } }),
    ]);

    // Get recent companies
    const recentCompanies = await Tenant.find()
      .sort({ createdAt: -1 })
      .limit(10)
      .select('name email status isActive createdAt');

    // Get companies by status
    const companiesByStatus = await Tenant.aggregate([
      {
        $group: {
          _id: '$status',
          count: { $sum: 1 },
        },
      },
    ]);

    return {
      totalCompanies,
      activeCompanies,
      totalUsers,
      pendingActivations,
      totalJobs,
      activeJobs,
      recentCompanies,
      companiesByStatus,
    };
  } catch (error) {
    logger.error('Get super admin dashboard stats error:', error);
    throw error;
  }
};

module.exports = {
  getAdminDashboardStats,
  getManagerDashboardStats,
  getAgentDashboardStats,
  getCustomerDashboardStats,
  getSuperAdminDashboardStats,
};

