/**
 * Customer Service
 * Business logic for customer management
 */

const Customer = require('../models/Customer.model');
const logger = require('../utils/logger');

/**
 * Get customers for a company
 */
const getCustomers = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };

    if (filters.status) {
      query.status = filters.status;
    }

    if (filters.search) {
      query.$or = [
        { name: { $regex: filters.search, $options: 'i' } },
        { email: { $regex: filters.search, $options: 'i' } },
        { customerId: { $regex: filters.search, $options: 'i' } },
        { contactNumber: { $regex: filters.search, $options: 'i' } },
      ];
    }

    const customers = await Customer.find(query)
      .populate('products', 'name productId productType hsnCode')
      .sort({ createdAt: -1 });

    return customers;
  } catch (error) {
    logger.error('Get customers error:', error);
    throw error;
  }
};

/**
 * Get customer by ID
 */
const getCustomerById = async (customerId, tenantId) => {
  try {
    const customer = await Customer.findOne({ _id: customerId, tenantId })
      .populate('products', 'name productId productType hsnCode description year');

    if (!customer) {
      throw new Error('Customer not found');
    }

    return customer;
  } catch (error) {
    logger.error('Get customer by ID error:', error);
    throw error;
  }
};

/**
 * Create customer
 */
const createCustomer = async (customerData, tenantId, createdBy) => {
  try {
    // Check if email already exists
    const existingCustomer = await Customer.findOne({ email: customerData.email.toLowerCase().trim() });
    if (existingCustomer) {
      throw new Error('Customer with this email already exists');
    }

    const customer = new Customer({
      ...customerData,
      tenantId,
      createdBy,
      email: customerData.email.toLowerCase().trim(),
    });

    await customer.save();
    
    // Populate products before returning
    await customer.populate('products', 'name productId productType');
    
    logger.info(`Customer created: ${customer.customerId}`, { customerId: customer._id, tenantId });

    return customer;
  } catch (error) {
    logger.error('Create customer error:', error);
    throw error;
  }
};

/**
 * Update customer
 */
const updateCustomer = async (customerId, customerData, tenantId) => {
  try {
    const customer = await Customer.findOne({ _id: customerId, tenantId });

    if (!customer) {
      throw new Error('Customer not found');
    }

    // If email is being updated, check for duplicates
    if (customerData.email && customerData.email.toLowerCase().trim() !== customer.email) {
      const existingCustomer = await Customer.findOne({ 
        email: customerData.email.toLowerCase().trim(),
        _id: { $ne: customerId }
      });
      if (existingCustomer) {
        throw new Error('Customer with this email already exists');
      }
      customerData.email = customerData.email.toLowerCase().trim();
    }

    // Update password if provided
    if (customerData.password && customerData.password.trim() !== '') {
      customer.password = customerData.password; // Will be hashed by pre-save hook
      customer.markModified('password');
      delete customerData.password;
    }

    // Update address fullAddress if address fields changed
    if (customerData.address) {
      const addressParts = [];
      if (customerData.address.blockName) addressParts.push(`Block: ${customerData.address.blockName}`);
      if (customerData.address.streetName) addressParts.push(customerData.address.streetName);
      if (customerData.address.floorNumber) addressParts.push(`Floor: ${customerData.address.floorNumber}`);
      if (customerData.address.flatNumber) addressParts.push(`Flat: ${customerData.address.flatNumber}`);
      customerData.address.fullAddress = addressParts.join(', ');
    }

    // Update other fields
    Object.assign(customer, customerData);
    await customer.save();
    
    // Populate products before returning
    await customer.populate('products', 'name productId productType');

    logger.info(`Customer updated: ${customer.customerId}`, { customerId: customer._id, tenantId });

    return customer;
  } catch (error) {
    logger.error('Update customer error:', error);
    throw error;
  }
};

/**
 * Delete customer
 */
const deleteCustomer = async (customerId, tenantId) => {
  try {
    const customer = await Customer.findOneAndDelete({ _id: customerId, tenantId });

    if (!customer) {
      throw new Error('Customer not found');
    }

    logger.info(`Customer deleted: ${customer.customerId}`, { customerId: customer._id, tenantId });

    return { success: true };
  } catch (error) {
    logger.error('Delete customer error:', error);
    throw error;
  }
};

module.exports = {
  getCustomers,
  getCustomerById,
  createCustomer,
  updateCustomer,
  deleteCustomer,
};

