/**
 * Compliance Management Service
 */

const Job = require('../models/Job.model');
const User = require('../models/User.model');
const Employee = require('../models/Employee.model');
const UserActivity = require('../models/UserActivity.model');
const Role = require('../models/Role.model');
const Notification = require('../models/Notification.model');
const { calculateSLADeadline } = require('../utils/sla');
const { resolveJobLocation, calculateDistance } = require('../utils/geocoding');
const logger = require('../utils/logger');

/**
 * Create complaint
 */
const createComplaint = async (complaintData, tenantId, customerId, userRole) => {
  try {
    // Get customer details - check Customer model for CUSTOMER role, User model for others
    let customer;
    if (userRole === 'CUSTOMER') {
      const Customer = require('../models/Customer.model');
      customer = await Customer.findById(customerId);
      if (!customer || customer.tenantId.toString() !== tenantId.toString()) {
        throw new Error('Customer not found');
      }
    } else {
      customer = await User.findById(customerId);
      if (!customer || customer.tenantId.toString() !== tenantId.toString()) {
        throw new Error('Customer not found');
      }
    }

    // Determine priority - use priority from complaintData if provided (from category), otherwise determine from description
    // Map Critical/Non-Critical to IMPORTANT/NOT_IMPORTANT for backend
    let priority = complaintData.priority;
    if (priority === 'Critical') {
      priority = 'IMPORTANT';
    } else if (priority === 'Non-Critical') {
      priority = 'NOT_IMPORTANT';
    } else {
      // Fallback to auto-determination
      priority = determinePriority(complaintData.description, complaintData.category);
    }

    // Prepare job location address for fallback
    const jobLocationAddress = complaintData.address?.fullAddress || 
      `${complaintData.address?.streetAddress || customer.address?.streetName || customer.address?.street || ''}, ${complaintData.address?.city || customer.address?.city || ''}`.trim() ||
      complaintData.address?.streetAddress || '';

    // Resolve job location from coordinates or address
    const jobLocation = await resolveJobLocation(
      complaintData.location?.latitude || 0,
      complaintData.location?.longitude || 0,
      jobLocationAddress
    );

    // No auto-assignment to manager - jobs are unassigned until agent takes them or manager assigns them
    // When agent takes a job, their manager will be attached to it

    // Calculate SLA deadline
    const slaDeadline = calculateSLADeadline('COMPLIANCE', priority, new Date());

    // Create job
    const job = new Job({
      tenantId,
      ticketType: 'COMPLIANCE',
      customerId,
      customerName: customer.name,
      customerPhone: customer.contactNumber || customer.phone,
      customerEmail: customer.email,
      assignedAgentId: null, // No agent assignment at creation
      assignedAgentModel: 'User',
      managerId: null, // No manager assignment at creation - will be set when agent takes job
      status: 'NEW',
      priority,
      jobLocationLat: complaintData.location?.latitude || customer.address?.coordinates?.lat || 0,
      jobLocationLng: complaintData.location?.longitude || customer.address?.coordinates?.lng || 0,
      jobLocationAddress: complaintData.address?.fullAddress || 
                          `${complaintData.address?.streetAddress || customer.address?.streetName || customer.address?.street || ''}, ${complaintData.address?.city || customer.address?.city || ''}`.trim() ||
                          complaintData.address?.streetAddress || '',
      complaintDescription: complaintData.description,
      complaintCategory: complaintData.categories && complaintData.categories.length > 0 
                          ? complaintData.categories.join(', ') 
                          : complaintData.category,
      complaintCategories: complaintData.categories || [],
      complaintAddress: {
        flatHouseNumber: complaintData.address?.flatHouseNumber || '',
        floorNumber: complaintData.address?.floorNumber || '',
        streetAddress: complaintData.address?.streetAddress || '',
        area: complaintData.address?.area || '',
        landmark: complaintData.address?.landmark || '',
        city: complaintData.address?.city || '',
        pincode: complaintData.address?.pincode || '',
        state: complaintData.address?.state || '',
        fullAddress: complaintData.address?.fullAddress || '',
      },
      complaintContact: {
        name: complaintData.contactName || '',
        number: complaintData.contactNumber || '',
        alternativeNumber: complaintData.alternativeContactNumber || '',
      },
      complaintPhotos: complaintData.photos || [], // Array of photo URLs
      productId: complaintData.productId || null,
      slaDeadline,
    });

    await job.save();

    // No notification to manager at creation - managers will see unassigned jobs in their view

    // Populate job with related data
    await job.populate([
      { path: 'customerId', select: 'name email phone' },
      { path: 'assignedAgentId', select: 'name email phone firstName lastName' },
      { path: 'managerId', select: 'name email' },
    ]);
    
    return job;
  } catch (error) {
    logger.error('Create complaint error:', error);
    throw error;
  }
};

/**
 * Determine priority based on description and category
 */
const determinePriority = (description, category) => {
  const urgentKeywords = ['urgent', 'emergency', 'broken', 'not working', 'down', 'critical'];
  const descriptionLower = description.toLowerCase();

  // Check for urgent keywords
  if (urgentKeywords.some(keyword => descriptionLower.includes(keyword))) {
    return 'IMPORTANT';
  }

  // Category-based priority
  const urgentCategories = ['BREAKDOWN', 'EMERGENCY', 'SAFETY'];
  if (urgentCategories.includes(category)) {
    return 'IMPORTANT';
  }

  return 'NOT_IMPORTANT';
};

/**
 * Find manager with lowest workload for job assignment
 * Considers both User (MANAGER role) and Employee (with Manager role) models
 * Workload = previous jobs + open jobs + pending jobs + ongoing jobs
 */
const findManagerWithLowestWorkload = async (tenantId) => {
  try {
    logger.info('🔵 [FIND MANAGER] Starting workload-based manager assignment', { tenantId });

    // Get Manager role ID for this tenant
    const managerRole = await Role.findOne({
      tenantId,
      name: 'Manager',
      status: 'ACTIVE',
    });

    // Get all active managers - both User model (MANAGER) and Employee model (with Manager role)
    const userManagers = await User.find({
      tenantId,
      role: 'MANAGER',
      status: 'ACTIVE',
    }).select('_id name email');

    const employeeManagers = managerRole
      ? await Employee.find({
          tenantId,
          roleId: managerRole._id,
          status: 'ACTIVE',
        }).select('_id firstName lastName email roleId')
      : [];

    if (userManagers.length === 0 && employeeManagers.length === 0) {
      logger.warn(`No active managers found for tenant: ${tenantId}`);
      return null;
    }

    // Combine managers with their model type
    const allManagers = [
      ...userManagers.map(manager => ({ ...manager.toObject(), modelType: 'User', managerId: manager._id })),
      ...employeeManagers.map(manager => ({
        ...manager.toObject(),
        modelType: 'Employee',
        managerId: manager._id,
        name: `${manager.firstName} ${manager.lastName}`,
      })),
    ];

    // Calculate workload for each manager
    const managersWithWorkload = await Promise.all(
      allManagers.map(async (manager) => {
        // For User model managers, use manager._id directly
        // For Employee model managers, find corresponding User account
        let managerUserId = manager.managerId;
        if (manager.modelType === 'Employee') {
          const userManager = await User.findOne({
            email: manager.email,
            tenantId,
            role: 'MANAGER',
          });
          if (userManager) {
            managerUserId = userManager._id;
          } else {
            // If no User account exists, use Employee ID (but jobs are assigned via User model)
            // In this case, we'll still calculate workload but may need to create User account
            logger.warn(`⚠️ [FIND MANAGER] Employee manager ${manager.email} has no corresponding User account`);
          }
        }

        // Calculate workload: previous jobs + open jobs + pending jobs + ongoing jobs
        // Previous jobs: CLOSED, CANCELLED
        // Open jobs: NEW
        // Pending jobs: PENDING_APPROVAL
        // Ongoing jobs: ACCEPTED, ARRIVED, IN_PROGRESS

        const [previousJobs, openJobs, pendingJobs, ongoingJobs] = await Promise.all([
          Job.countDocuments({
            tenantId,
            managerId: managerUserId,
            status: { $in: ['CLOSED', 'CANCELLED'] },
          }),
          Job.countDocuments({
            tenantId,
            managerId: managerUserId,
            status: 'NEW',
            assignedAgentId: null, // Unassigned jobs
          }),
          Job.countDocuments({
            tenantId,
            managerId: managerUserId,
            status: 'PENDING_APPROVAL',
          }),
          Job.countDocuments({
            tenantId,
            managerId: managerUserId,
            status: { $in: ['ACCEPTED', 'ARRIVED', 'IN_PROGRESS'] },
          }),
        ]);

        const totalWorkload = previousJobs + openJobs + pendingJobs + ongoingJobs;

        logger.info(`📊 [FIND MANAGER] Manager workload`, {
          managerId: manager.managerId,
          email: manager.email,
          previousJobs,
          openJobs,
          pendingJobs,
          ongoingJobs,
          totalWorkload,
        });

        return {
          ...manager,
          managerUserId,
          workload: {
            previousJobs,
            openJobs,
            pendingJobs,
            ongoingJobs,
            total: totalWorkload,
          },
        };
      })
    );

    // Sort by total workload (ascending) and return manager with lowest workload
    managersWithWorkload.sort((a, b) => a.workload.total - b.workload.total);

    const selectedManager = managersWithWorkload[0];
    
    if (selectedManager) {
      logger.info(`✅ [FIND MANAGER] Selected manager with lowest workload`, {
        managerId: selectedManager.managerId,
        email: selectedManager.email,
        workload: selectedManager.workload.total,
      });

      // Return the User model manager (create if needed for Employee managers)
      if (selectedManager.modelType === 'Employee') {
        let userManager = await User.findOne({
          email: selectedManager.email,
          tenantId,
          role: 'MANAGER',
        });

        if (!userManager) {
          logger.warn(`⚠️ [FIND MANAGER] Creating User account for Employee manager ${selectedManager.email}`);
          // Create User account for Employee manager (without password - they use Employee login)
          userManager = new User({
            tenantId,
            email: selectedManager.email,
            name: selectedManager.name,
            role: 'MANAGER',
            status: 'ACTIVE',
            password: 'temp_password_should_not_be_used', // Placeholder - Employee uses Employee model for login
          });
          await userManager.save();
        }

        return userManager;
      } else {
        return await User.findById(selectedManager.managerId);
      }
    }

    return null;
  } catch (error) {
    logger.error('Find manager with lowest workload error:', error);
    throw error;
  }
};

/**
 * Find nearest available agent based on location and workload
 * Considers both User (SERVICE_AGENT) and Employee (with Agent role) models
 * NOTE: This function is kept for backward compatibility but is no longer used in complaint creation
 */
const findNearestAvailableAgent = async (jobLocation, tenantId) => {
  try {
    // Get Agent role ID for this tenant
    const agentRole = await Role.findOne({
      tenantId,
      name: 'Agent',
      status: 'ACTIVE',
    });

    // Get all active agents - both User model (SERVICE_AGENT) and Employee model (with Agent role)
    const userAgents = await User.find({
      tenantId,
      role: 'SERVICE_AGENT',
      status: 'ACTIVE',
    }).select('_id name email phone managerId');

    const employeeAgents = agentRole
      ? await Employee.find({
          tenantId,
          roleId: agentRole._id,
          status: 'ACTIVE',
        }).select('_id firstName lastName email contactNumber roleId')
      : [];

    if (userAgents.length === 0 && employeeAgents.length === 0) {
      logger.warn(`No active agents found for tenant: ${tenantId}`);
      return null;
    }

    // Combine agents with their model type
    const allAgents = [
      ...userAgents.map(agent => ({ ...agent.toObject(), modelType: 'User', agentId: agent._id })),
      ...employeeAgents.map(agent => ({
        ...agent.toObject(),
        modelType: 'Employee',
        agentId: agent._id,
        name: `${agent.firstName} ${agent.lastName}`,
      })),
    ];

    // If no job location, assign to agent with least workload
    if (!jobLocation || !jobLocation.lat || !jobLocation.lng) {
      logger.info('No job location available. Assigning based on workload only.');
      return await findAgentByWorkload(allAgents, tenantId);
    }

    // Get most recent location for each agent from UserActivity
    const agentsWithLocation = await Promise.all(
      allAgents.map(async (agent) => {
        // Get most recent location from UserActivity
        const recentActivity = await UserActivity.findOne({
          userId: agent.agentId,
          tenantId,
          activityType: { $in: ['LOCATION_UPDATE', 'LOGIN'] },
          'location.latitude': { $exists: true, $ne: null },
          'location.longitude': { $exists: true, $ne: null },
        })
          .sort({ createdAt: -1 })
          .select('location');

        const agentLocation = recentActivity?.location
          ? {
              lat: recentActivity.location.latitude,
              lng: recentActivity.location.longitude,
            }
          : null;

        // Get active job count for this agent
        const activeJobCount = await Job.countDocuments({
          tenantId,
          assignedAgentId: agent.agentId,
          status: { $in: ['NEW', 'IN_PROGRESS', 'ASSIGNED'] },
        });

        // Calculate distance if location is available
        let distance = Infinity;
        if (agentLocation) {
          distance = calculateDistance(
            jobLocation.lat,
            jobLocation.lng,
            agentLocation.lat,
            agentLocation.lng
          );
        }

        return {
          ...agent,
          location: agentLocation,
          distance,
          activeJobCount,
        };
      })
    );

    // Filter out agents without location if we have location-based job
    const agentsWithValidLocation = agentsWithLocation.filter(
      (agent) => agent.location !== null
    );

    // If we have agents with location, use location + workload scoring
    if (agentsWithValidLocation.length > 0) {
      // Score agents: lower distance and lower workload = better score
      const scoredAgents = agentsWithValidLocation.map((agent) => {
        // Normalize distance (convert to km, then score inversely)
        const distanceKm = agent.distance / 1000;
        // Normalize workload (assume max 20 jobs for normalization)
        const normalizedWorkload = Math.min(agent.activeJobCount / 20, 1);

        // Weighted score: 60% distance, 40% workload
        // Lower score is better
        const score = distanceKm * 0.6 + normalizedWorkload * 40;

        return {
          ...agent,
          score,
        };
      });

      // Sort by score (lower is better)
      scoredAgents.sort((a, b) => a.score - b.score);

      const selectedAgent = scoredAgents[0];
      logger.info(
        `Selected agent: ${selectedAgent.name} (${selectedAgent.modelType}) - Distance: ${(selectedAgent.distance / 1000).toFixed(2)}km, Workload: ${selectedAgent.activeJobCount} jobs`
      );

      // Return agent in the format expected by the caller
      let agent;
      if (selectedAgent.modelType === 'User') {
        agent = await User.findById(selectedAgent.agentId);
      } else {
        agent = await Employee.findById(selectedAgent.agentId);
      }
      return { agent, modelType: selectedAgent.modelType };
    }

    // Fallback: no location available, use workload-based assignment
    logger.info('No agents with location found. Assigning based on workload only.');
    return await findAgentByWorkload(allAgents, tenantId);
  } catch (error) {
    logger.error('Find nearest agent error:', error);
    return null;
  }
};

/**
 * Find agent with least workload
 */
const findAgentByWorkload = async (agents, tenantId) => {
  try {
    const agentsWithWorkload = await Promise.all(
      agents.map(async (agent) => {
        const activeJobCount = await Job.countDocuments({
          tenantId,
          assignedAgentId: agent.agentId,
          status: { $in: ['NEW', 'IN_PROGRESS', 'ASSIGNED'] },
        });

        return {
          ...agent,
          activeJobCount,
        };
      })
    );

    // Sort by workload (ascending)
    agentsWithWorkload.sort((a, b) => a.activeJobCount - b.activeJobCount);

    const selectedAgent = agentsWithWorkload[0];
    logger.info(
      `Selected agent by workload: ${selectedAgent.name} (${selectedAgent.modelType}) - Workload: ${selectedAgent.activeJobCount} jobs`
    );

    // Return agent in the format expected by the caller
    let agent;
    if (selectedAgent.modelType === 'User') {
      agent = await User.findById(selectedAgent.agentId);
    } else {
      agent = await Employee.findById(selectedAgent.agentId);
    }
    return { agent, modelType: selectedAgent.modelType };
  } catch (error) {
    logger.error('Find agent by workload error:', error);
    // Fallback to first available agent
    if (agents.length > 0) {
      const firstAgent = agents[0];
      let agent;
      if (firstAgent.modelType === 'User') {
        agent = await User.findById(firstAgent.agentId);
      } else {
        agent = await Employee.findById(firstAgent.agentId);
      }
      return { agent, modelType: firstAgent.modelType };
    }
    return null;
  }
};


/**
 * Get complaints
 */
const getComplaints = async (filters, tenantId, userRole, userId) => {
  try {
    const query = {
      tenantId,
      ticketType: 'COMPLIANCE',
    };

    // Apply filters
    if (filters.status) {
      query.status = filters.status;
    }
    if (filters.priority) {
      query.priority = filters.priority;
    }

    // Customer can only see own complaints
    if (userRole === 'CUSTOMER') {
      query.customerId = userId;
    }

    // Agent/Employee can only see assigned complaints
    if (userRole === 'SERVICE_AGENT' || userRole === 'EMPLOYEE') {
      query.assignedAgentId = userId;
    }

    // Manager isolation
    if (userRole === 'MANAGER') {
      const manager = await User.findById(userId);
      if (manager && manager.assignedAgents && manager.assignedAgents.length > 0) {
        query.assignedAgentId = { $in: manager.assignedAgents };
      } else {
        return [];
      }
    }

    const complaints = await Job.find(query)
      .populate('customerId', 'name email phone')
      .populate('managerId', 'name email')
      .sort({ createdAt: -1 });

    // Manually populate assignedAgentId from both User and Employee models
    for (const complaint of complaints) {
      if (complaint.assignedAgentId) {
        // Try User model first
        const userAgent = await User.findById(complaint.assignedAgentId).select('name email phone');
        if (userAgent) {
          complaint.assignedAgentId = userAgent;
        } else {
          // Try Employee model
          const employeeAgent = await Employee.findById(complaint.assignedAgentId).select('firstName lastName email contactNumber');
          if (employeeAgent) {
            complaint.assignedAgentId = employeeAgent;
          }
        }
      }
    }

    return complaints;
  } catch (error) {
    logger.error('Get complaints error:', error);
    throw error;
  }
};

module.exports = {
  createComplaint,
  getComplaints,
  findNearestAvailableAgent,
  findManagerWithLowestWorkload,
};

