/**
 * Complaint Category Service
 */

const ComplaintCategory = require('../models/ComplaintCategory.model');
const logger = require('../utils/logger');

/**
 * Get all complaint categories for a tenant
 */
const getCategories = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };
    
    if (filters.isActive !== undefined) {
      query.isActive = filters.isActive === 'true';
    }
    
    if (filters.search) {
      query.$or = [
        { name: { $regex: filters.search, $options: 'i' } },
        { description: { $regex: filters.search, $options: 'i' } },
      ];
    }

    const categories = await ComplaintCategory.find(query)
      .sort({ name: 1 })
      .select('-__v');

    return categories;
  } catch (error) {
    logger.error('Get categories error:', error);
    throw error;
  }
};

/**
 * Get category by ID
 */
const getCategoryById = async (categoryId, tenantId) => {
  try {
    const category = await ComplaintCategory.findOne({
      _id: categoryId,
      tenantId,
    });

    if (!category) {
      throw new Error('Category not found');
    }

    return category;
  } catch (error) {
    logger.error('Get category by ID error:', error);
    throw error;
  }
};

/**
 * Create complaint category
 */
const createCategory = async (categoryData, tenantId, createdBy) => {
  try {
    // Check if category with same name already exists for this tenant
    const existing = await ComplaintCategory.findOne({
      name: categoryData.name,
      tenantId,
    });

    if (existing) {
      throw new Error('Category with this name already exists');
    }

    const category = new ComplaintCategory({
      ...categoryData,
      tenantId,
      createdBy,
    });

    await category.save();

    logger.info(`Category created: ${category.name}`, { categoryId: category._id, tenantId });

    return category;
  } catch (error) {
    logger.error('Create category error:', error);
    throw error;
  }
};

/**
 * Update complaint category
 */
const updateCategory = async (categoryId, categoryData, tenantId) => {
  try {
    const category = await ComplaintCategory.findOne({
      _id: categoryId,
      tenantId,
    });

    if (!category) {
      throw new Error('Category not found');
    }

    // Check if new name conflicts with existing category
    if (categoryData.name && categoryData.name !== category.name) {
      const existing = await ComplaintCategory.findOne({
        name: categoryData.name,
        tenantId,
        _id: { $ne: categoryId },
      });

      if (existing) {
        throw new Error('Category with this name already exists');
      }
    }

    Object.assign(category, categoryData);
    await category.save();

    logger.info(`Category updated: ${category.name}`, { categoryId: category._id, tenantId });

    return category;
  } catch (error) {
    logger.error('Update category error:', error);
    throw error;
  }
};

/**
 * Delete complaint category
 */
const deleteCategory = async (categoryId, tenantId) => {
  try {
    const category = await ComplaintCategory.findOneAndDelete({
      _id: categoryId,
      tenantId,
    });

    if (!category) {
      throw new Error('Category not found');
    }

    logger.info(`Category deleted: ${category.name}`, { categoryId, tenantId });

    return { message: 'Category deleted successfully' };
  } catch (error) {
    logger.error('Delete category error:', error);
    throw error;
  }
};

module.exports = {
  getCategories,
  getCategoryById,
  createCategory,
  updateCategory,
  deleteCategory,
};

