/**
 * Authentication Service
 */

const User = require('../models/User.model');
const { generateAccessToken, generateRefreshToken } = require('../utils/jwt');
const { comparePassword } = require('../utils/password');
const logger = require('../utils/logger');

/**
 * Login user
 */
const login = async (email, password) => {
  try {
    // Normalize email to lowercase
    const normalizedEmail = email.toLowerCase().trim();
    
    // Check if it's Super Admin (from users collection)
    const superAdmin = await User.findOne({ 
      email: normalizedEmail, 
      role: 'SUPER_ADMIN' 
    }).select('+password').populate('tenantId');

    if (superAdmin) {
      // Super Admin login
      const isPasswordValid = await comparePassword(password, superAdmin.password);
      if (!isPasswordValid) {
        logger.warn(`Invalid password attempt for Super Admin: ${normalizedEmail}`);
        throw new Error('Invalid credentials');
      }

      if (superAdmin.status !== 'ACTIVE') {
        throw new Error('User account is not active');
      }

      // Generate tokens for Super Admin
      const payload = {
        userId: superAdmin._id.toString(),
        email: superAdmin.email,
        role: superAdmin.role,
        tenantId: superAdmin.tenantId?._id?.toString() || superAdmin.tenantId?.toString(),
      };

      const accessToken = generateAccessToken(payload);
      const refreshToken = generateRefreshToken(payload);

      superAdmin.password = undefined;

      return {
        accessToken,
        refreshToken,
        user: superAdmin,
      };
    }

    // Check if it's a Customer login
    const Customer = require('../models/Customer.model');
    const customer = await Customer.findOne({ email: normalizedEmail }).select('+password').populate('tenantId');

    if (customer) {
      // Customer login
      const isPasswordValid = await comparePassword(password, customer.password);
      if (!isPasswordValid) {
        logger.warn(`Invalid password attempt for customer: ${normalizedEmail}`);
        throw new Error('Invalid credentials');
      }

      if (customer.status !== 'ACTIVE') {
        throw new Error('Customer account is not active');
      }

      // Check if company is active
      if (customer.tenantId) {
        const tenant = customer.tenantId._id ? customer.tenantId : await require('../models/Tenant.model').findById(customer.tenantId);
        if (!tenant || !tenant.isActive || tenant.status !== 'ACTIVE') {
          throw new Error('Company account is not active. Please contact support.');
        }
      }

      // Create user object for customer login (for compatibility)
      const customerUser = {
        _id: customer._id,
        email: customer.email,
        name: customer.name,
        role: 'CUSTOMER',
        tenantId: customer.tenantId?._id || customer.tenantId,
        status: customer.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE',
        customerId: customer.customerId,
        contactNumber: customer.contactNumber,
        address: customer.address,
        products: customer.products,
      };

      // Generate tokens for customer
      const payload = {
        userId: customerUser._id.toString(),
        email: customerUser.email,
        role: customerUser.role,
        tenantId: customerUser.tenantId.toString(),
      };

      const accessToken = generateAccessToken(payload);
      const refreshToken = generateRefreshToken(payload);

      return {
        accessToken,
        refreshToken,
        user: customerUser,
      };
    }

    // Check if it's an Employee login (from employees collection)
    // Check employees FIRST before companies to avoid conflicts
    logger.info(`Checking employees collection for: ${normalizedEmail}`);
    const Employee = require('../models/Employee.model');
    const employee = await Employee.findOne({ email: normalizedEmail }).select('+password').populate('tenantId');

    if (employee) {
      logger.info(`Employee found in employees collection: ${normalizedEmail}`, {
        employeeId: employee._id,
        email: employee.email,
        status: employee.status,
        tenantId: employee.tenantId?._id || employee.tenantId
      });
      
      // Employee login - check password from employees collection
      logger.info(`Comparing password for employee: ${normalizedEmail}`);
      const isPasswordValid = await comparePassword(password, employee.password);
      
      if (!isPasswordValid) {
        logger.warn(`Invalid password attempt for employee: ${normalizedEmail}`, {
          employeeId: employee._id,
          email: employee.email
        });
        throw new Error('Invalid credentials');
      }

      logger.info(`Password valid for employee: ${normalizedEmail}`);

      if (employee.status !== 'ACTIVE') {
        logger.warn(`Employee account is not active: ${normalizedEmail}`, {
          status: employee.status
        });
        throw new Error('Employee account is not active');
      }

      // Check if company is active
      if (employee.tenantId) {
        const tenant = employee.tenantId._id ? employee.tenantId : await require('../models/Tenant.model').findById(employee.tenantId);
        if (!tenant || !tenant.isActive || tenant.status !== 'ACTIVE') {
          logger.warn(`Company account is not active for employee: ${normalizedEmail}`, {
            tenantId: employee.tenantId?._id || employee.tenantId,
            tenantStatus: tenant?.status,
            tenantIsActive: tenant?.isActive
          });
          throw new Error('Company account is not active. Please contact support.');
        }
      }

      // Populate role to get permissions
      await employee.populate('roleId', 'name description permissions');

      // Create user object for employee login
      // Employees have EMPLOYEE role with role-based permissions
      const employeeUser = {
        _id: employee._id,
        email: employee.email,
        name: `${employee.firstName} ${employee.lastName}`,
        role: 'EMPLOYEE', // Employees have EMPLOYEE role with role-based permissions
        tenantId: employee.tenantId?._id || employee.tenantId,
        status: employee.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE',
        permissions: employee.roleId?.permissions || {},
        roleName: employee.roleId?.name || 'Employee',
        assignedClients: employee.assignedClients,
        assignedJobs: employee.assignedJobs,
      };

      logger.info(`Employee login successful: ${normalizedEmail}`, {
        employeeId: employee._id,
        role: 'EMPLOYEE',
        tenantId: employeeUser.tenantId,
        hasPermissions: !!employeeUser.permissions,
        roleName: employeeUser.roleName
      });

      // Generate tokens for employee
      const payload = {
        userId: employeeUser._id.toString(),
        email: employeeUser.email,
        role: employeeUser.role, // EMPLOYEE role
        tenantId: employeeUser.tenantId.toString(),
      };

      const accessToken = generateAccessToken(payload);
      const refreshToken = generateRefreshToken(payload);

      return {
        accessToken,
        refreshToken,
        user: employeeUser,
      };
    } else {
      logger.info(`No employee found with email: ${normalizedEmail}, checking companies collection...`);
    }

    // Company login (from companies collection) - Check LAST
    // Companies have ADMIN role with FULL permissions
    const Tenant = require('../models/Tenant.model');
    const company = await Tenant.findOne({ email: normalizedEmail }).select('+password');

    if (company) {
      logger.info(`Company login attempt: ${normalizedEmail}`);
      
      // Company login - check password from companies collection
      const isPasswordValid = await comparePassword(password, company.password);
      if (!isPasswordValid) {
        logger.warn(`Invalid password attempt for company: ${normalizedEmail}`);
        throw new Error('Invalid credentials');
      }

      // Check if company is active
      if (!company.isActive || company.status !== 'ACTIVE') {
        throw new Error('Company account is not active. Please contact support.');
      }

      // Create user object for company login
      // Companies always have ADMIN role with FULL permissions
      const companyUser = {
        _id: company._id,
        email: company.email,
        name: company.name,
        role: 'ADMIN', // Companies login as ADMIN with FULL access
        tenantId: company._id, // Company is its own tenant
        status: company.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE',
      };

      logger.info(`Company login successful: ${normalizedEmail}`, {
        companyId: company._id,
        role: 'ADMIN',
        tenantId: companyUser.tenantId,
        hasFullAccess: true
      });

      // Generate tokens for company
      const payload = {
        userId: companyUser._id.toString(),
        email: companyUser.email,
        role: companyUser.role, // ADMIN role with full permissions
        tenantId: companyUser.tenantId.toString(),
      };

      const accessToken = generateAccessToken(payload);
      const refreshToken = generateRefreshToken(payload);

      return {
        accessToken,
        refreshToken,
        user: companyUser,
      };
    }

    // No matching user found in any collection
    logger.warn(`Login attempt with non-existent email: ${normalizedEmail}`);
    throw new Error('Invalid credentials');
  } catch (error) {
    logger.error('Login error:', error);
    throw error;
  }
};

/**
 * Refresh access token
 */
const refreshToken = async (refreshToken) => {
  try {
    const { verifyRefreshToken } = require('../utils/jwt');
    const decoded = verifyRefreshToken(refreshToken);

    // Check if it's Super Admin (from users collection)
    if (decoded.role === 'SUPER_ADMIN') {
      const user = await User.findById(decoded.userId).populate('tenantId');

      if (!user || user.status !== 'ACTIVE') {
        throw new Error('Invalid refresh token');
      }

      // Generate new access token
      const payload = {
        userId: user._id.toString(),
        email: user.email,
        role: user.role,
        tenantId: user.tenantId?._id?.toString() || user.tenantId?.toString(),
      };

      const newAccessToken = generateAccessToken(payload);

      return {
        accessToken: newAccessToken,
      };
    }

    // Check if it's a Customer refresh token
    if (decoded.role === 'CUSTOMER') {
      const Customer = require('../models/Customer.model');
      const customer = await Customer.findById(decoded.userId);

      if (!customer || customer.status !== 'ACTIVE') {
        throw new Error('Invalid refresh token');
      }

      // Check if company is active
      if (customer.tenantId) {
        const Tenant = require('../models/Tenant.model');
        const tenant = await Tenant.findById(customer.tenantId);
        if (!tenant || !tenant.isActive || tenant.status !== 'ACTIVE') {
          throw new Error('Company account is not active');
        }
      }

      // Generate new access token for customer
      const payload = {
        userId: customer._id.toString(),
        email: customer.email,
        role: 'CUSTOMER',
        tenantId: customer.tenantId.toString(),
      };

      const newAccessToken = generateAccessToken(payload);

      return {
        accessToken: newAccessToken,
      };
    }

    // Check if it's an Employee refresh token
    if (decoded.role === 'EMPLOYEE') {
      const Employee = require('../models/Employee.model');
      const employee = await Employee.findById(decoded.userId);

      if (!employee || employee.status !== 'ACTIVE') {
        throw new Error('Invalid refresh token');
      }

      // Check if company is active
      if (employee.tenantId) {
        const Tenant = require('../models/Tenant.model');
        const tenant = await Tenant.findById(employee.tenantId);
        if (!tenant || !tenant.isActive || tenant.status !== 'ACTIVE') {
          throw new Error('Company account is not active');
        }
      }

      // Generate new access token for employee
      const payload = {
        userId: employee._id.toString(),
        email: employee.email,
        role: 'EMPLOYEE',
        tenantId: employee.tenantId.toString(),
      };

      const newAccessToken = generateAccessToken(payload);

      return {
        accessToken: newAccessToken,
      };
    }

    // Company refresh token (from companies collection)
    const Tenant = require('../models/Tenant.model');
    const company = await Tenant.findById(decoded.userId);

    if (!company || !company.isActive || company.status !== 'ACTIVE') {
      throw new Error('Invalid refresh token');
    }

    // Generate new access token for company
    const payload = {
      userId: company._id.toString(),
      email: company.email,
      role: 'ADMIN', // Companies login as ADMIN
      tenantId: company._id.toString(),
    };

    const newAccessToken = generateAccessToken(payload);

    return {
      accessToken: newAccessToken,
    };
  } catch (error) {
    logger.error('Refresh token error:', error);
    throw error;
  }
};

module.exports = {
  login,
  refreshToken,
};

