/**
 * Audit Log Service
 */

const AuditLog = require('../models/AuditLog.model');
const logger = require('../utils/logger');

/**
 * Create audit log entry
 */
const createAuditLog = async (logData) => {
  try {
    const auditLog = new AuditLog(logData);
    await auditLog.save();
    return auditLog;
  } catch (error) {
    logger.error('Create audit log error:', error);
    // Don't throw - audit logging should not break the application
  }
};

/**
 * Get audit logs
 */
const getAuditLogs = async (filters, tenantId, userRole) => {
  try {
    const query = {};

    if (tenantId) {
      query.tenantId = tenantId;
    }

    if (filters.userId) {
      query.userId = filters.userId;
    }

    if (filters.entityType) {
      query.entityType = filters.entityType;
    }

    if (filters.entityId) {
      query.entityId = filters.entityId;
    }

    if (filters.action) {
      query.action = filters.action;
    }

    if (filters.startDate || filters.endDate) {
      query.createdAt = {};
      if (filters.startDate) {
        query.createdAt.$gte = new Date(filters.startDate);
      }
      if (filters.endDate) {
        query.createdAt.$lte = new Date(filters.endDate);
      }
    }

    const auditLogs = await AuditLog.find(query)
      .populate('userId', 'name email role')
      .populate('tenantId', 'name')
      .sort({ createdAt: -1 })
      .limit(filters.limit || 100);

    return auditLogs;
  } catch (error) {
    logger.error('Get audit logs error:', error);
    throw error;
  }
};

/**
 * Get audit logs by entity
 */
const getAuditLogsByEntity = async (entityType, entityId, tenantId) => {
  try {
    const auditLogs = await AuditLog.find({
      tenantId,
      entityType,
      entityId,
    })
      .populate('userId', 'name email role')
      .sort({ createdAt: -1 });

    return auditLogs;
  } catch (error) {
    logger.error('Get audit logs by entity error:', error);
    throw error;
  }
};

module.exports = {
  createAuditLog,
  getAuditLogs,
  getAuditLogsByEntity,
};

