/**
 * Attendance Service
 * Business logic for attendance management
 */

const Attendance = require('../models/Attendance.model');
const Employee = require('../models/Employee.model');
const logger = require('../utils/logger');

/**
 * Get attendances with filters
 */
const getAttendances = async (filters, tenantId) => {
  try {
    const query = { tenantId };

    if (filters.employeeId) {
      query.employeeId = filters.employeeId;
    }

    if (filters.startDate && filters.endDate) {
      query.date = {
        $gte: new Date(filters.startDate),
        $lte: new Date(filters.endDate),
      };
    } else if (filters.date) {
      const date = new Date(filters.date);
      const startOfDay = new Date(date.setHours(0, 0, 0, 0));
      const endOfDay = new Date(date.setHours(23, 59, 59, 999));
      query.date = {
        $gte: startOfDay,
        $lte: endOfDay,
      };
    }

    if (filters.status) {
      query.status = filters.status;
    }

    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 10;
    const skip = (page - 1) * limit;

    const total = await Attendance.countDocuments(query);
    const attendances = await Attendance.find(query)
      .populate('employeeId', 'firstName lastName email employeeId')
      .populate('approvedBy', 'name email')
      .sort({ date: -1, createdAt: -1 })
      .skip(skip)
      .limit(limit);

    return {
      attendances,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    };
  } catch (error) {
    logger.error('Get attendances error:', error);
    throw error;
  }
};

/**
 * Get attendance by ID
 */
const getAttendanceById = async (attendanceId, tenantId) => {
  try {
    const attendance = await Attendance.findOne({ _id: attendanceId, tenantId })
      .populate('employeeId', 'firstName lastName email employeeId')
      .populate('approvedBy', 'name email');
    
    if (!attendance) {
      throw new Error('Attendance not found');
    }
    
    return attendance;
  } catch (error) {
    logger.error('Get attendance by ID error:', error);
    throw error;
  }
};

/**
 * Create or update attendance (check-in)
 */
const checkIn = async (employeeId, tenantId, checkInData) => {
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    let attendance = await Attendance.findOne({
      tenantId,
      employeeId,
      date: today,
    });

    if (attendance && attendance.checkIn && attendance.checkIn.time) {
      throw new Error('Already checked in today');
    }

    if (!attendance) {
      attendance = new Attendance({
        tenantId,
        employeeId,
        date: today,
        status: 'PRESENT',
      });
    }

    attendance.checkIn = {
      time: new Date(),
      location: checkInData.location || {},
      method: checkInData.method || 'MOBILE_APP',
    };

    await attendance.save();
    await attendance.populate('employeeId', 'firstName lastName email employeeId');
    
    return attendance;
  } catch (error) {
    logger.error('Check-in error:', error);
    throw error;
  }
};

/**
 * Check-out
 */
const checkOut = async (employeeId, tenantId, checkOutData) => {
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const attendance = await Attendance.findOne({
      tenantId,
      employeeId,
      date: today,
    });

    if (!attendance || !attendance.checkIn || !attendance.checkIn.time) {
      throw new Error('Please check in first');
    }

    if (attendance.checkOut && attendance.checkOut.time) {
      throw new Error('Already checked out today');
    }

    attendance.checkOut = {
      time: new Date(),
      location: checkOutData.location || {},
      method: checkOutData.method || 'MOBILE_APP',
    };

    // Calculate total work hours
    const checkInTime = new Date(attendance.checkIn.time);
    const checkOutTime = new Date(attendance.checkOut.time);
    const totalMinutes = (checkOutTime - checkInTime) / (1000 * 60);
    
    let breakMinutes = 0;
    if (attendance.breaks && attendance.breaks.length > 0) {
      attendance.breaks.forEach(breakItem => {
        if (breakItem.duration) {
          breakMinutes += breakItem.duration;
        } else if (breakItem.startTime && breakItem.endTime) {
          const breakStart = new Date(breakItem.startTime);
          const breakEnd = new Date(breakItem.endTime);
          breakMinutes += (breakEnd - breakStart) / (1000 * 60);
        }
      });
    }
    
    attendance.totalWorkHours = (totalMinutes - breakMinutes) / 60;

    await attendance.save();
    await attendance.populate('employeeId', 'firstName lastName email employeeId');
    
    return attendance;
  } catch (error) {
    logger.error('Check-out error:', error);
    throw error;
  }
};

/**
 * Add break
 */
const addBreak = async (employeeId, tenantId, breakData) => {
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const attendance = await Attendance.findOne({
      tenantId,
      employeeId,
      date: today,
    });

    if (!attendance || !attendance.checkIn || !attendance.checkIn.time) {
      throw new Error('Please check in first');
    }

    if (!attendance.breaks) {
      attendance.breaks = [];
    }

    attendance.breaks.push({
      startTime: new Date(),
      type: breakData.type || 'OTHER',
      location: breakData.location || {},
    });

    await attendance.save();
    await attendance.populate('employeeId', 'firstName lastName email employeeId');
    
    return attendance;
  } catch (error) {
    logger.error('Add break error:', error);
    throw error;
  }
};

/**
 * End break
 */
const endBreak = async (employeeId, tenantId, breakIndex) => {
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const attendance = await Attendance.findOne({
      tenantId,
      employeeId,
      date: today,
    });

    if (!attendance || !attendance.breaks || !attendance.breaks[breakIndex]) {
      throw new Error('Break not found');
    }

    const breakItem = attendance.breaks[breakIndex];
    if (breakItem.endTime) {
      throw new Error('Break already ended');
    }

    breakItem.endTime = new Date();
    const breakStart = new Date(breakItem.startTime);
    const breakEnd = new Date(breakItem.endTime);
    breakItem.duration = (breakEnd - breakStart) / (1000 * 60);

    await attendance.save();
    await attendance.populate('employeeId', 'firstName lastName email employeeId');
    
    return attendance;
  } catch (error) {
    logger.error('End break error:', error);
    throw error;
  }
};

/**
 * Update attendance (manual entry by admin)
 */
const updateAttendance = async (attendanceId, tenantId, updateData) => {
  try {
    const attendance = await Attendance.findOneAndUpdate(
      { _id: attendanceId, tenantId },
      updateData,
      { new: true, runValidators: true }
    )
      .populate('employeeId', 'firstName lastName email employeeId')
      .populate('approvedBy', 'name email');

    if (!attendance) {
      throw new Error('Attendance not found');
    }

    return attendance;
  } catch (error) {
    logger.error('Update attendance error:', error);
    throw error;
  }
};

/**
 * Get attendance statistics
 */
const getAttendanceStats = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };

    if (filters.startDate && filters.endDate) {
      query.date = {
        $gte: new Date(filters.startDate),
        $lte: new Date(filters.endDate),
      };
    }

    if (filters.employeeId) {
      query.employeeId = filters.employeeId;
    }

    const attendances = await Attendance.find(query)
      .populate('employeeId', 'firstName lastName email employeeId');

    const stats = {
      total: attendances.length,
      present: attendances.filter(a => a.status === 'PRESENT').length,
      absent: attendances.filter(a => a.status === 'ABSENT').length,
      late: attendances.filter(a => a.status === 'LATE').length,
      halfDay: attendances.filter(a => a.status === 'HALF_DAY').length,
      onLeave: attendances.filter(a => a.status === 'ON_LEAVE').length,
      averageWorkHours: 0,
    };

    const presentAttendances = attendances.filter(a => a.totalWorkHours);
    if (presentAttendances.length > 0) {
      const totalHours = presentAttendances.reduce((sum, a) => sum + (a.totalWorkHours || 0), 0);
      stats.averageWorkHours = totalHours / presentAttendances.length;
    }

    return stats;
  } catch (error) {
    logger.error('Get attendance stats error:', error);
    throw error;
  }
};

module.exports = {
  getAttendances,
  getAttendanceById,
  checkIn,
  checkOut,
  addBreak,
  endBreak,
  updateAttendance,
  getAttendanceStats,
};

