/**
 * Approval Service
 */

const Job = require('../models/Job.model');
const JobApproval = require('../models/JobApproval.model');
const User = require('../models/User.model');
const Employee = require('../models/Employee.model');
const logger = require('../utils/logger');

/**
 * Get pending approvals for manager
 */
const getPendingApprovals = async (managerId, tenantId) => {
  try {
    // Get manager's assigned agents
    const manager = await User.findById(managerId);
    if (!manager || !manager.assignedAgents || manager.assignedAgents.length === 0) {
      return [];
    }

    // Get pending jobs from assigned agents (both PENDING_APPROVAL and SIGNATURE_PENDING)
    const jobs = await Job.find({
      tenantId,
      managerId: managerId,
      assignedAgentId: { $in: manager.assignedAgents },
      status: { $in: ['PENDING_APPROVAL', 'SIGNATURE_PENDING'] },
    })
      .populate('customerId', 'name email phone')
      .populate('assignedAgentId', 'name email phone')
      .populate('managerId', 'name email')
      .sort({ completedAt: -1 });

    return jobs;
  } catch (error) {
    logger.error('Get pending approvals error:', error);
    throw error;
  }
};

/**
 * Approve job
 */
const approveJob = async (jobId, managerId, tenantId, comments, userRole = null, roleName = null) => {
  try {
    logger.info('🔵 [APPROVE JOB] Starting approve job', { jobId, managerId, userRole, roleName });
    
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    logger.info('✅ [APPROVE JOB] Job found', { jobId: job._id, status: job.status, assignedAgentId: job.assignedAgentId });

    // Verify manager has access to this job
    // Check if managerId is a User with MANAGER role
    let manager = await User.findById(managerId);
    
    // If not found or not a User model manager, check if it's an Employee with Manager role
    if (!manager || manager.role !== 'MANAGER') {
      logger.info('📱 [APPROVE JOB] Manager not found as User, checking Employee collection...');
      const employee = await Employee.findById(managerId).populate('roleId', 'name');
      
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
        logger.info('✅ [APPROVE JOB] Employee manager found', { employeeId: employee._id, email: employee.email });
        // Find corresponding User account with MANAGER role
        manager = await User.findOne({ 
          email: employee.email, 
          tenantId: tenantId,
          role: 'MANAGER' 
        });
        
        if (!manager) {
          logger.error('❌ [APPROVE JOB] Employee manager found but no corresponding User account');
          throw new Error('Manager User account not found. Please create a User account for this manager.');
        }
        logger.info('✅ [APPROVE JOB] Found corresponding User account for Employee manager', { userId: manager._id });
      } else {
        logger.error('❌ [APPROVE JOB] User is not a manager', { managerId, userRole, roleName });
        throw new Error('Only managers can approve jobs');
      }
    } else {
      logger.info('✅ [APPROVE JOB] Manager found as User', { userId: manager._id, email: manager.email });
    }

    // Check if manager has access to this job (job must be assigned to one of manager's agents)
    // Jobs can be assigned to either User IDs or Employee IDs
    let hasAccess = false;
    
    if (!manager.assignedAgents || manager.assignedAgents.length === 0) {
      logger.warn('⚠️ [APPROVE JOB] Manager has no assigned agents');
      throw new Error('You do not have permission to approve this job - no assigned agents');
    }
    
    // Get assigned User agents
    const assignedUserAgents = await User.find({
      _id: { $in: manager.assignedAgents },
      tenantId,
      role: 'SERVICE_AGENT',
    }).select('email');
    
    // Get emails of assigned User agents
    const assignedAgentEmails = assignedUserAgents.map(agent => agent.email);
    
    // Find corresponding Employee records for these agents (by email)
    const assignedEmployees = await Employee.find({
      email: { $in: assignedAgentEmails },
      tenantId,
    }).select('_id email');
    
    // Build combined list of agent IDs (both User IDs and Employee IDs)
    const agentUserIds = manager.assignedAgents.map(id => id.toString());
    const agentEmployeeIds = assignedEmployees.map(emp => emp._id.toString());
    const allAgentIds = [...agentUserIds, ...agentEmployeeIds];
    
    logger.info('👥 [APPROVE JOB] Manager assigned agents', { 
      userIds: agentUserIds, 
      employeeIds: agentEmployeeIds,
      jobAssignedAgentId: job.assignedAgentId?.toString() 
    });
    
    // Check if job's assignedAgentId matches any of the manager's agents
    const jobAgentIdStr = job.assignedAgentId?.toString();
    hasAccess = allAgentIds.includes(jobAgentIdStr);
    
    logger.info('🔍 [APPROVE JOB] Access check', { hasAccess, jobAgentId: jobAgentIdStr });
    
    if (!hasAccess) {
      logger.error('❌ [APPROVE JOB] Manager does not have access to this job');
      throw new Error('You do not have permission to approve this job');
    }

    // Allow approval for both PENDING_APPROVAL and SIGNATURE_PENDING jobs
    // But check if signature is required
    if (job.status !== 'PENDING_APPROVAL' && job.status !== 'SIGNATURE_PENDING') {
      throw new Error('Job is not pending approval');
    }
    
    // If job is SIGNATURE_PENDING, check if signature exists
    if (job.status === 'SIGNATURE_PENDING' && !job.isSigned) {
      throw new Error('Job requires signature before approval. Please add signature first.');
    }

    // Create or update approval record
    let approval = await JobApproval.findOne({ jobId });
    if (!approval) {
      approval = new JobApproval({
        jobId,
        tenantId,
        managerId,
      });
    }

    approval.approvalStatus = 'APPROVED';
    approval.comments = comments;
    approval.approvedAt = new Date();
    await approval.save();

    // Update job status
    job.status = 'APPROVED';
    await job.save();

    // Send email notification to manager
    try {
      const emailService = require('./email.service');
      const manager = await User.findById(managerId);
      if (manager) {
        await emailService.sendApprovalRequiredEmail(manager, job);
      }
    } catch (error) {
      logger.warn('Failed to send approval email:', error);
    }

    // Generate report automatically
    try {
      const reportService = require('./report.service');
      await reportService.generateReport(jobId, tenantId);
    } catch (error) {
      logger.warn('Failed to auto-generate report on approval:', error);
    }

    return job.populate(['customerId', 'assignedAgentId', 'managerId']);
  } catch (error) {
    logger.error('Approve job error:', error);
    throw error;
  }
};

/**
 * Reject job
 */
const rejectJob = async (jobId, managerId, tenantId, rejectionReason, comments, userRole = null, roleName = null) => {
  try {
    logger.info('🔵 [REJECT JOB] Starting reject job', { jobId, managerId, userRole, roleName });
    
    const job = await Job.findOne({ _id: jobId, tenantId });

    if (!job) {
      throw new Error('Job not found');
    }

    logger.info('✅ [REJECT JOB] Job found', { jobId: job._id, status: job.status, assignedAgentId: job.assignedAgentId });

    // Verify manager has access to this job
    // Check if managerId is a User with MANAGER role
    let manager = await User.findById(managerId);
    
    // If not found or not a User model manager, check if it's an Employee with Manager role
    if (!manager || manager.role !== 'MANAGER') {
      logger.info('📱 [REJECT JOB] Manager not found as User, checking Employee collection...');
      const employee = await Employee.findById(managerId).populate('roleId', 'name');
      
      if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
        logger.info('✅ [REJECT JOB] Employee manager found', { employeeId: employee._id, email: employee.email });
        // Find corresponding User account with MANAGER role
        manager = await User.findOne({ 
          email: employee.email, 
          tenantId: tenantId,
          role: 'MANAGER' 
        });
        
        if (!manager) {
          logger.error('❌ [REJECT JOB] Employee manager found but no corresponding User account');
          throw new Error('Manager User account not found. Please create a User account for this manager.');
        }
        logger.info('✅ [REJECT JOB] Found corresponding User account for Employee manager', { userId: manager._id });
      } else {
        logger.error('❌ [REJECT JOB] User is not a manager', { managerId, userRole, roleName });
        throw new Error('Only managers can reject jobs');
      }
    } else {
      logger.info('✅ [REJECT JOB] Manager found as User', { userId: manager._id, email: manager.email });
    }

    // Check if manager has access to this job (job must be assigned to one of manager's agents)
    // Jobs can be assigned to either User IDs or Employee IDs
    let hasAccess = false;
    
    if (!manager.assignedAgents || manager.assignedAgents.length === 0) {
      logger.warn('⚠️ [REJECT JOB] Manager has no assigned agents');
      throw new Error('You do not have permission to reject this job - no assigned agents');
    }
    
    // Get assigned User agents
    const assignedUserAgents = await User.find({
      _id: { $in: manager.assignedAgents },
      tenantId,
      role: 'SERVICE_AGENT',
    }).select('email');
    
    // Get emails of assigned User agents
    const assignedAgentEmails = assignedUserAgents.map(agent => agent.email);
    
    // Find corresponding Employee records for these agents (by email)
    const assignedEmployees = await Employee.find({
      email: { $in: assignedAgentEmails },
      tenantId,
    }).select('_id email');
    
    // Build combined list of agent IDs (both User IDs and Employee IDs)
    const agentUserIds = manager.assignedAgents.map(id => id.toString());
    const agentEmployeeIds = assignedEmployees.map(emp => emp._id.toString());
    const allAgentIds = [...agentUserIds, ...agentEmployeeIds];
    
    logger.info('👥 [REJECT JOB] Manager assigned agents', { 
      userIds: agentUserIds, 
      employeeIds: agentEmployeeIds,
      jobAssignedAgentId: job.assignedAgentId?.toString() 
    });
    
    // Check if job's assignedAgentId matches any of the manager's agents
    const jobAgentIdStr = job.assignedAgentId?.toString();
    hasAccess = allAgentIds.includes(jobAgentIdStr);
    
    logger.info('🔍 [REJECT JOB] Access check', { hasAccess, jobAgentId: jobAgentIdStr });
    
    if (!hasAccess) {
      logger.error('❌ [REJECT JOB] Manager does not have access to this job');
      throw new Error('You do not have permission to reject this job');
    }

    // Allow rejection for both PENDING_APPROVAL and SIGNATURE_PENDING jobs
    if (job.status !== 'PENDING_APPROVAL' && job.status !== 'SIGNATURE_PENDING') {
      throw new Error('Job is not pending approval');
    }

    // Create or update approval record
    let approval = await JobApproval.findOne({ jobId });
    if (!approval) {
      approval = new JobApproval({
        jobId,
        tenantId,
        managerId,
      });
    }

    approval.approvalStatus = 'REJECTED';
    approval.rejectionReason = rejectionReason;
    approval.comments = comments;
    approval.rejectedAt = new Date();
    await approval.save();

    // Update job status - agent needs to rework
    job.status = 'REJECTED';
    await job.save();

    return job.populate(['customerId', 'assignedAgentId', 'managerId']);
  } catch (error) {
    logger.error('Reject job error:', error);
    throw error;
  }
};

module.exports = {
  getPendingApprovals,
  approveJob,
  rejectJob,
};

