/**
 * Activity Tracking Service
 * Business logic for tracking employee activities
 */

const UserActivity = require('../models/UserActivity.model');
const Employee = require('../models/Employee.model');
const logger = require('../utils/logger');

/**
 * Get activities with filters
 */
const getActivities = async (filters, tenantId) => {
  try {
    const query = { tenantId };

    if (filters.userId) {
      query.userId = filters.userId;
    }

    if (filters.userModel) {
      query.userModel = filters.userModel;
    }

    if (filters.activityType) {
      query.activityType = filters.activityType;
    }

    if (filters.startDate && filters.endDate) {
      query.createdAt = {
        $gte: new Date(filters.startDate),
        $lte: new Date(filters.endDate),
      };
    }

    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 50;
    const skip = (page - 1) * limit;

    const total = await UserActivity.countDocuments(query);
    const activities = await UserActivity.find(query)
      .populate('userId', 'firstName lastName email employeeId name')
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(limit);

    return {
      activities,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    };
  } catch (error) {
    logger.error('Get activities error:', error);
    throw error;
  }
};

/**
 * Get activity statistics
 */
const getActivityStats = async (tenantId, filters = {}) => {
  try {
    const query = { tenantId };

    if (filters.userId) {
      query.userId = filters.userId;
    }

    if (filters.startDate && filters.endDate) {
      query.createdAt = {
        $gte: new Date(filters.startDate),
        $lte: new Date(filters.endDate),
      };
    }

    const activities = await UserActivity.find(query)
      .populate('userId', 'firstName lastName email employeeId name');

    const stats = {
      total: activities.length,
      byType: {},
      byUser: {},
      recentActivities: [],
    };

    // Group by activity type
    activities.forEach(activity => {
      const type = activity.activityType;
      stats.byType[type] = (stats.byType[type] || 0) + 1;
    });

    // Group by user
    activities.forEach(activity => {
      const userId = activity.userId?._id?.toString() || activity.userId?.toString();
      if (userId) {
        const userName = activity.userId?.firstName 
          ? `${activity.userId.firstName} ${activity.userId.lastName}`
          : activity.userId?.name || 'Unknown';
        if (!stats.byUser[userId]) {
          stats.byUser[userId] = {
            name: userName,
            count: 0,
          };
        }
        stats.byUser[userId].count++;
      }
    });

    // Get recent activities (last 10)
    stats.recentActivities = activities.slice(0, 10).map(activity => ({
      id: activity._id,
      type: activity.activityType,
      userName: activity.userId?.firstName 
        ? `${activity.userId.firstName} ${activity.userId.lastName}`
        : activity.userId?.name || 'Unknown',
      timestamp: activity.createdAt,
      location: activity.location,
      metadata: activity.metadata,
    }));

    return stats;
  } catch (error) {
    logger.error('Get activity stats error:', error);
    throw error;
  }
};

/**
 * Get user activity timeline
 */
const getUserActivityTimeline = async (userId, tenantId, filters = {}) => {
  try {
    const query = {
      tenantId,
      userId,
    };

    if (filters.startDate && filters.endDate) {
      query.createdAt = {
        $gte: new Date(filters.startDate),
        $lte: new Date(filters.endDate),
      };
    }

    const activities = await UserActivity.find(query)
      .sort({ createdAt: -1 })
      .limit(100);

    return activities;
  } catch (error) {
    logger.error('Get user activity timeline error:', error);
    throw error;
  }
};

module.exports = {
  getActivities,
  getActivityStats,
  getUserActivityTimeline,
};

