/**
 * HQR Backend Server
 * Main entry point for the application
 */

require('dotenv').config();
const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const mongoose = require('mongoose');
const logger = require('./utils/logger');

// Import routes
const authRoutes = require('./routes/auth.routes');
const userRoutes = require('./routes/user.routes');
const tenantRoutes = require('./routes/tenant.routes');
const jobRoutes = require('./routes/job.routes');
const approvalRoutes = require('./routes/approval.routes');
const mediaRoutes = require('./routes/media.routes');
const reportRoutes = require('./routes/report.routes');
const periodicMaintenanceRoutes = require('./routes/periodicMaintenance.routes');
const complianceRoutes = require('./routes/compliance.routes');
const kpiRoutes = require('./routes/kpi.routes');
const kpiMatrixRoutes = require('./routes/kpiMatrix.routes');
const slaRoutes = require('./routes/sla.routes');
const notificationRoutes = require('./routes/notification.routes');
const auditLogRoutes = require('./routes/auditLog.routes');
const dashboardRoutes = require('./routes/dashboard.routes');
const employeeRoutes = require('./routes/employee.routes');
const roleRoutes = require('./routes/role.routes');
const settingsRoutes = require('./routes/settings.routes');
const productRoutes = require('./routes/product.routes');
const customerRoutes = require('./routes/customer.routes');
const complaintCategoryRoutes = require('./routes/complaintCategory.routes');
const periodicMaintenanceConfigRoutes = require('./routes/periodicMaintenanceConfig.routes');
const attendanceRoutes = require('./routes/attendance.routes');
const activityTrackingRoutes = require('./routes/activityTracking.routes');

// Import middleware
const errorHandler = require('./middleware/errorHandler.middleware');
const { authenticateToken } = require('./middleware/auth.middleware');

const app = express();
const PORT = process.env.PORT || 3000;

// Health check endpoints (BEFORE all middleware for quick access)
// These endpoints are placed before all middleware to ensure they're always accessible
app.get('/health', (req, res) => {
  const clientIp = req.ip || req.connection.remoteAddress || req.socket.remoteAddress;
  console.log('🏥 Health check requested');
  console.log('   Client IP: ', clientIp);
  console.log('   User-Agent: ', req.get('user-agent') || 'Unknown');
  console.log('   Origin: ', req.get('origin') || 'No origin (mobile app)');
  logger.info('Health check accessed', { 
    ip: clientIp,
    userAgent: req.get('user-agent'),
    origin: req.get('origin')
  });
  
  res.status(200).json({
    success: true,
    message: 'Server is running',
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    server: 'HQR Backend',
    version: '1.0.0'
  });
});

app.get('/api/v1/health', (req, res) => {
  const clientIp = req.ip || req.connection.remoteAddress || req.socket.remoteAddress;
  console.log('🏥 API health check requested');
  console.log('   Client IP: ', clientIp);
  logger.info('API health check accessed', { 
    ip: clientIp,
    userAgent: req.get('user-agent'),
    origin: req.get('origin')
  });
  
  res.status(200).json({
    success: true,
    message: 'API is running',
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    server: 'HQR Backend API',
    version: '1.0.0'
  });
});

// Simple test endpoint (no middleware, fastest response)
app.get('/test', (req, res) => {
  const clientIp = req.ip || req.connection.remoteAddress || req.socket.remoteAddress;
  console.log('✅ Test endpoint hit from IP:', clientIp);
  res.status(200).send('OK');
});

// Security middleware
app.use(helmet());

// Request logging middleware (before CORS)
app.use((req, res, next) => {
  const timestamp = new Date().toISOString();
  console.log(`[${timestamp}] 📥 Incoming ${req.method} ${req.url}`);
  console.log(`   IP: ${req.ip || req.connection.remoteAddress}`);
  console.log(`   Origin: ${req.get('origin') || 'No origin (mobile app)'}`);
  console.log(`   User-Agent: ${req.get('user-agent') || 'Unknown'}`);
  
  logger.info('📥 [SERVER] Incoming request', {
    method: req.method,
    url: req.url,
    ip: req.ip || req.connection.remoteAddress,
    origin: req.get('origin'),
    userAgent: req.get('user-agent'),
    timestamp
  });
  next();
});

app.use(cors({
  origin: function (origin, callback) {
    // Allow requests with no origin (like mobile apps or Postman)
    if (!origin) return callback(null, true);
    
    const allowedOrigins = process.env.CORS_ORIGIN?.split(',') || [
      'http://localhost:3001',  // Admin Panel
      'http://localhost:3002',  // White-Label Panel
      'http://localhost:5173',   // Vite default
      'http://192.168.0.3:3001', // Admin Panel (network)
      'http://192.168.0.3:3002', // White-Label Panel (network)
    ];
    
    // Allow all origins for mobile apps (you can restrict this in production)
    // For now, allow all to debug
    logger.info('🌐 [CORS] Checking origin', { origin, allowedOrigins });
    
    if (allowedOrigins.includes(origin) || origin.includes('192.168.') || origin.includes('localhost')) {
      callback(null, true);
    } else {
      // Allow mobile apps (no origin or any origin for debugging)
      callback(null, true);
    }
  },
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'PATCH', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With']
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: parseInt(process.env.RATE_LIMIT_WINDOW_MS) || 60000,
  max: parseInt(process.env.RATE_LIMIT_MAX_REQUESTS) || 100,
  message: 'Too many requests from this IP, please try again later.'
});
app.use('/api/', limiter);

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Serve static files (uploads) with CORS headers
app.use('/uploads', (req, res, next) => {
  res.header('Access-Control-Allow-Origin', '*');
  res.header('Access-Control-Allow-Methods', 'GET, OPTIONS');
  res.header('Access-Control-Allow-Headers', 'Content-Type');
  next();
}, express.static('uploads'));

// API Routes
app.use('/api/v1/auth', authRoutes);

// ============================================================================
// CRITICAL: Register /update-location route BEFORE general user routes
// This ensures it's matched before any parameterized routes like /:id
// This endpoint is accessible to ALL authenticated users:
// - EMPLOYEE (same access as SERVICE_AGENT for job operations)
// - SERVICE_AGENT (agents)
// - CUSTOMER
// - ADMIN
// - MANAGER
// ============================================================================
const userController = require('./controllers/user.controller');
// Register this route FIRST, before any other /api/v1/users routes
app.post('/api/v1/users/update-location', authenticateToken, async (req, res) => {
  console.log('🔍 [SERVER.JS] /update-location route matched');
  console.log('   URL:', req.originalUrl);
  console.log('   Path:', req.path);
  console.log('   Method:', req.method);
  console.log('   User Role:', req.userRole);
  console.log('   User ID:', req.userId);
  console.log('   Tenant ID:', req.tenantId);
  
  // Verify user is authenticated (authenticateToken already did this)
  if (!req.user || !req.userId || !req.userRole) {
    console.error('❌ [SERVER.JS] Missing user data after authenticateToken');
    return res.status(401).json({
      success: false,
      message: 'Authentication required'
    });
  }
  
  // Allow all authenticated users (same as job update access for agents/employees)
  // EMPLOYEE and SERVICE_AGENT have job update access, so they should have location update access too
  const allowedRoles = ['EMPLOYEE', 'SERVICE_AGENT', 'CUSTOMER', 'ADMIN', 'MANAGER', 'SUPER_ADMIN'];
  if (!allowedRoles.includes(req.userRole)) {
    console.warn('⚠️ [SERVER.JS] Unrecognized role:', req.userRole, '- but allowing anyway since authenticated');
  }
  
  console.log('✅ [SERVER.JS] User authenticated, proceeding to controller');
  // Call the controller directly (it's async and handles req, res)
  try {
    await userController.updateLocation(req, res);
  } catch (err) {
    console.error('❌ [SERVER.JS] Error calling updateLocation:', err);
    if (!res.headersSent) {
      res.status(500).json({
        success: false,
        message: err.message || 'Failed to update location'
      });
    }
  }
});

// General user routes (all require authentication)
app.use('/api/v1/users', authenticateToken, userRoutes);
app.use('/api/v1/tenants', authenticateToken, tenantRoutes);
app.use('/api/v1/jobs', authenticateToken, jobRoutes);
app.use('/api/v1/approvals', authenticateToken, approvalRoutes);
app.use('/api/v1/media', authenticateToken, mediaRoutes);
app.use('/api/v1/reports', authenticateToken, reportRoutes);
app.use('/api/v1/periodic-maintenance', authenticateToken, periodicMaintenanceRoutes);
app.use('/api/v1/compliance', authenticateToken, complianceRoutes);
app.use('/api/v1/kpi', authenticateToken, kpiRoutes);
app.use('/api/v1/kpi-matrix', authenticateToken, kpiMatrixRoutes);
app.use('/api/v1/sla', authenticateToken, slaRoutes);
app.use('/api/v1/notifications', authenticateToken, notificationRoutes);
app.use('/api/v1/audit-logs', authenticateToken, auditLogRoutes);
app.use('/api/v1/dashboard', authenticateToken, dashboardRoutes);
app.use('/api/v1/employees', employeeRoutes);
app.use('/api/v1/roles', roleRoutes);
app.use('/api/v1/settings', settingsRoutes);
app.use('/api/v1/products', productRoutes);
app.use('/api/v1/customers', customerRoutes);
app.use('/api/v1/complaint-categories', complaintCategoryRoutes);
app.use('/api/v1/periodic-maintenance-config', authenticateToken, periodicMaintenanceConfigRoutes);
app.use('/api/v1/attendance', attendanceRoutes);
app.use('/api/v1/activity-tracking', activityTrackingRoutes);

// Error handling middleware (must be last)
app.use(errorHandler);

// Database connection
const connectDB = async () => {
  try {
    const mongoURI = process.env.MONGODB_URI || 'mongodb://135.181.103.182:1200/hyderabad';
    await mongoose.connect(mongoURI, {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
    logger.info('MongoDB connected successfully');
  } catch (error) {
    logger.error('MongoDB connection error:', error);
    process.exit(1);
  }
};

// Start server
const startServer = async () => {
  try {
    await connectDB();
    
    // Start periodic maintenance cron job
    try {
      const { startPeriodicMaintenanceCron } = require('./jobs/periodicMaintenance.job');
      startPeriodicMaintenanceCron();
    } catch (error) {
      logger.warn('Could not start periodic maintenance cron job:', error.message);
    }

    // Auto-assign cron job DISABLED - Jobs are now assigned to managers, not agents
    // Managers manually assign jobs to their agents
    // try {
    //   const { startAutoAssignCron } = require('./jobs/autoAssign.job');
    //   startAutoAssignCron();
    // } catch (error) {
    //   logger.warn('Could not start auto-assign cron job:', error.message);
    // }
    
    // Initialize email service
    try {
      const { initializeEmailService } = require('./services/email.service');
      initializeEmailService();
    } catch (error) {
      logger.warn('Could not initialize email service:', error.message);
    }
    
    // Listen on all interfaces (0.0.0.0) to accept connections from mobile devices
    const server = app.listen(PORT, '0.0.0.0', () => {
      const address = server.address();
      logger.info(`🚀 HQR Backend Server running on port ${PORT}`);
      logger.info(`   Local: http://localhost:${PORT}`);
      logger.info(`   Network: http://192.168.0.3:${PORT}`);
      logger.info(`   API Base: http://0.0.0.0:${PORT}/api/v1`);
      logger.info(`   Server Address: ${JSON.stringify(address)}`);
      logger.info(`   Environment: ${process.env.NODE_ENV || 'development'}`);
      console.log('\n========================================');
      console.log('🚀 SERVER IS RUNNING');
      console.log(`   Port: ${PORT}`);
      console.log(`   Local: http://localhost:${PORT}`);
      console.log(`   Network: http://192.168.0.3:${PORT}`);
      console.log(`   Health Check: http://192.168.0.3:${PORT}/health`);
      console.log(`   API Health: http://192.168.0.3:${PORT}/api/v1/health`);
      console.log('========================================\n');
    });
    
    // Add error handlers for the server
    server.on('error', (error) => {
      if (error.code === 'EADDRINUSE') {
        logger.error(`❌ Port ${PORT} is already in use. Please stop the other process or use a different port.`);
        console.error(`❌ Port ${PORT} is already in use!`);
        console.error('   Try: netstat -ano | findstr :3000 (Windows)');
        console.error('   Or: lsof -i :3000 (Mac/Linux)');
      } else {
        logger.error('❌ Server error:', error);
        console.error('❌ Server error:', error);
      }
      process.exit(1);
    });
    
    server.on('listening', () => {
      logger.info('✅ Server is listening and ready to accept connections');
      console.log('✅ Server is listening and ready to accept connections');
    });
  } catch (error) {
    logger.error('Failed to start server:', error);
    process.exit(1);
  }
};

// Handle unhandled promise rejections
process.on('unhandledRejection', (err) => {
  logger.error('Unhandled Rejection:', err);
  process.exit(1);
});

// Handle uncaught exceptions
process.on('uncaughtException', (err) => {
  logger.error('Uncaught Exception:', err);
  process.exit(1);
});

startServer();

module.exports = app;

