/**
 * Migration Script: Update all companies to have role: 'ADMIN'
 * This script updates all existing companies in the companies collection
 * to have the role field set to 'ADMIN'
 */

require('dotenv').config();
const mongoose = require('mongoose');
const Tenant = require('../models/Tenant.model');
const logger = require('../utils/logger');

async function updateCompaniesRole() {
  try {
    // Connect to MongoDB
    const mongoUri = process.env.MONGODB_URI || 'mongodb://135.181.103.182:1200/hyderabad';
    await mongoose.connect(mongoUri);
    logger.info('Connected to MongoDB');

    // Find all companies
    const companies = await Tenant.find({});
    logger.info(`Found ${companies.length} companies to update`);

    // Update all companies to have role: 'ADMIN'
    const updateResult = await Tenant.updateMany(
      { role: { $exists: false } }, // Only update companies without role field
      { $set: { role: 'ADMIN' } }
    );

    logger.info(`Updated ${updateResult.modifiedCount} companies with role: 'ADMIN'`);

    // Also update companies that might have null or undefined role
    const updateResult2 = await Tenant.updateMany(
      { $or: [{ role: null }, { role: { $exists: false } }] },
      { $set: { role: 'ADMIN' } }
    );

    logger.info(`Total companies updated: ${updateResult2.modifiedCount + updateResult.modifiedCount}`);

    // Verify all companies have role: 'ADMIN'
    const companiesWithoutRole = await Tenant.countDocuments({ 
      $or: [
        { role: { $exists: false } },
        { role: null },
        { role: { $ne: 'ADMIN' } }
      ]
    });

    if (companiesWithoutRole === 0) {
      logger.info('✅ All companies now have role: ADMIN');
    } else {
      logger.warn(`⚠️  ${companiesWithoutRole} companies still don't have role: ADMIN`);
    }

    // List all companies and their roles
    const allCompanies = await Tenant.find({}).select('name email role');
    logger.info('Companies and their roles:');
    allCompanies.forEach(company => {
      logger.info(`  - ${company.name} (${company.email}): role = ${company.role || 'NOT SET'}`);
    });

    await mongoose.disconnect();
    logger.info('Migration completed successfully');
    process.exit(0);
  } catch (error) {
    logger.error('Migration failed:', error);
    await mongoose.disconnect();
    process.exit(1);
  }
}

// Run the migration
updateCompaniesRole();

