/**
 * Test Login Script
 * Test if login credentials work
 * 
 * Usage: node scripts/testLogin.js
 */

require('dotenv').config();
const mongoose = require('mongoose');
const User = require('../models/User.model');
const { comparePassword } = require('../utils/password');
const logger = require('../utils/logger');

const testLogin = async () => {
  try {
    // Connect to MongoDB
    const mongoURI = process.env.MONGODB_URI || 'mongodb://135.181.103.182:1200/hyderabad';
    await mongoose.connect(mongoURI, {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
    logger.info('Connected to MongoDB');

    const email = 'superadmin@hrq.com';
    const password = 'SuperAdmin@123';

    // Find user (without populate to avoid model registration issue)
    const user = await User.findOne({ email: email.toLowerCase().trim() }).select('+password');
    
    if (!user) {
      logger.error('❌ User not found!');
      logger.info('Email searched: ' + email.toLowerCase().trim());
      await mongoose.disconnect();
      process.exit(1);
    }

    logger.info('✅ User found!');
    logger.info('Email: ' + user.email);
    logger.info('Role: ' + user.role);
    logger.info('Status: ' + user.status);
    logger.info('Password hash: ' + user.password.substring(0, 20) + '...');

    // Test password
    logger.info('');
    logger.info('Testing password: ' + password);
    const isPasswordValid = await comparePassword(password, user.password);
    
    if (isPasswordValid) {
      logger.info('✅ Password is CORRECT!');
    } else {
      logger.error('❌ Password is INCORRECT!');
      logger.info('Trying to hash and compare again...');
      
      // Try with bcrypt directly
      const bcrypt = require('bcryptjs');
      const directCompare = await bcrypt.compare(password, user.password);
      logger.info('Direct bcrypt.compare result: ' + directCompare);
    }

    await mongoose.disconnect();
    process.exit(0);
  } catch (error) {
    logger.error('Error testing login:', error);
    await mongoose.disconnect();
    process.exit(1);
  }
};

testLogin();

