/**
 * Seed Super Admin User
 * Run this script to create the initial Super Admin user for the white-label panel
 * 
 * Usage: node scripts/seedSuperAdmin.js
 */

require('dotenv').config();
const mongoose = require('mongoose');
const User = require('../models/User.model');
const Tenant = require('../models/Tenant.model');
const { hashPassword } = require('../utils/password');
const logger = require('../utils/logger');

const seedSuperAdmin = async () => {
  try {
    // Connect to MongoDB
    const mongoURI = process.env.MONGODB_URI || 'mongodb://135.181.103.182:1200/hyderabad';
    await mongoose.connect(mongoURI, {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
    logger.info('Connected to MongoDB');

    // Check if Super Admin already exists
    const existingSuperAdmin = await User.findOne({ role: 'SUPER_ADMIN' }).select('+password');
    if (existingSuperAdmin) {
      logger.warn('⚠️  Super Admin user already exists!');
      logger.info(`📧 Email: ${existingSuperAdmin.email}`);
      logger.info(`👤 Name: ${existingSuperAdmin.name}`);
      logger.info(`🔐 Status: ${existingSuperAdmin.status}`);
      
      // Ask if user wants to reset password
      const args = process.argv.slice(2);
      if (args.includes('--reset-password') || args.includes('-r')) {
        logger.info('🔄 Resetting password...');
        const newPassword = process.env.SUPER_ADMIN_PASSWORD || 'SuperAdmin@123';
        
        // Mark password as modified to trigger pre-save hook
        existingSuperAdmin.password = newPassword; // Set plain password
        existingSuperAdmin.markModified('password'); // Mark as modified
        existingSuperAdmin.status = 'ACTIVE';
        
        // Save will trigger the pre-save hook to hash the password
        await existingSuperAdmin.save();
        
        // Verify the password was saved correctly
        const verifyUser = await User.findById(existingSuperAdmin._id).select('+password');
        const { comparePassword } = require('../utils/password');
        const isValid = await comparePassword(newPassword, verifyUser.password);
        
        if (isValid) {
          logger.info('✅ Password reset successfully!');
          logger.info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          logger.info('📧 Email: ' + existingSuperAdmin.email);
          logger.info('🔑 New Password: ' + newPassword);
          logger.info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        } else {
          logger.error('❌ Password reset failed - verification failed!');
          logger.info('Trying alternative method...');
          
          // Alternative: Update directly in database
          const hashedPassword = await hashPassword(newPassword);
          await User.updateOne(
            { _id: existingSuperAdmin._id },
            { 
              $set: { 
                password: hashedPassword,
                status: 'ACTIVE'
              } 
            }
          );
          
          // Verify again
          const verifyUser2 = await User.findById(existingSuperAdmin._id).select('+password');
          const isValid2 = await comparePassword(newPassword, verifyUser2.password);
          
          if (isValid2) {
            logger.info('✅ Password reset successfully (alternative method)!');
            logger.info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
            logger.info('📧 Email: ' + existingSuperAdmin.email);
            logger.info('🔑 New Password: ' + newPassword);
            logger.info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          } else {
            logger.error('❌ Password reset failed completely!');
          }
        }
        
        await mongoose.disconnect();
        process.exit(0);
      } else {
        logger.info('');
        logger.info('💡 To reset password, run:');
        logger.info('   npm run seed:superadmin -- --reset-password');
        logger.info('   or');
        logger.info('   npm run seed:superadmin -- -r');
        await mongoose.disconnect();
        process.exit(0);
      }
    }

    // Create a default tenant for Super Admin (optional, can be null)
    // For Super Admin, tenantId can be null or a special system tenant
    let systemTenant = await Tenant.findOne({ name: 'System' });
    if (!systemTenant) {
      systemTenant = new Tenant({
        name: 'System',
        email: 'system@hrq.com',
        isActive: true,
        status: 'ACTIVE',
      });
      await systemTenant.save();
      logger.info('Created system tenant');
    }

    // Create Super Admin user
    const superAdminData = {
      tenantId: systemTenant._id, // Can be null for Super Admin
      email: process.env.SUPER_ADMIN_EMAIL || 'superadmin@hrq.com',
      password: await hashPassword(process.env.SUPER_ADMIN_PASSWORD || 'SuperAdmin@123'),
      name: process.env.SUPER_ADMIN_NAME || 'Super Admin',
      phone: process.env.SUPER_ADMIN_PHONE || '',
      role: 'SUPER_ADMIN',
      status: 'ACTIVE',
    };

    const superAdmin = new User(superAdminData);
    await superAdmin.save();

    logger.info('✅ Super Admin user created successfully!');
    logger.info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    logger.info('📧 Email: ' + superAdminData.email);
    logger.info('🔑 Password: ' + (process.env.SUPER_ADMIN_PASSWORD || 'SuperAdmin@123'));
    logger.info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    logger.info('⚠️  Please change the password after first login!');

    await mongoose.disconnect();
    process.exit(0);
  } catch (error) {
    logger.error('Error seeding Super Admin:', error);
    await mongoose.disconnect();
    process.exit(1);
  }
};

seedSuperAdmin();

