/**
 * Script to create default Manager and Agent roles for all existing companies
 * Run this script once to add default roles to companies that were created before this feature
 * 
 * Usage: node scripts/createDefaultRolesForExistingCompanies.js
 */

require('dotenv').config();
const mongoose = require('mongoose');
const Tenant = require('../models/Tenant.model');
const Role = require('../models/Role.model');
const logger = require('../utils/logger');

/**
 * Create default Manager and Agent roles for a company
 */
const createDefaultRoles = async (tenantId) => {
  try {
    // Check if roles already exist
    const existingManager = await Role.findOne({ tenantId, name: 'Manager' });
    const existingAgent = await Role.findOne({ tenantId, name: 'Agent' });

    let created = [];

    // Create Manager role if it doesn't exist
    if (!existingManager) {
      const managerRole = new Role({
        tenantId,
        name: 'Manager',
        description: 'Default Manager role with full management permissions',
        permissions: {
          reports: {
            create: true,
            view: true,
            editUpdate: true,
            delete: true,
          },
          customerManagement: {
            create: true,
            view: true,
            editUpdate: true,
            delete: true,
          },
          agentsManagement: {
            create: true,
            view: true,
            editUpdate: true,
            delete: true,
          },
          jobsManagement: {
            create: true,
            view: true,
            edit: true,
            update: true,
            assignToAgents: true,
            approveReject: true,
          },
        },
        isDefault: false,
        status: 'ACTIVE',
        createdBy: tenantId,
      });
      await managerRole.save();
      created.push('Manager');
      logger.info(`Manager role created for tenant: ${tenantId}`);
    }

    // Create Agent role if it doesn't exist
    if (!existingAgent) {
      const agentRole = new Role({
        tenantId,
        name: 'Agent',
        description: 'Default Agent role with job execution permissions',
        permissions: {
          reports: {
            create: true,
            view: true,
            editUpdate: true,
            delete: false,
          },
          customerManagement: {
            create: false,
            view: true,
            editUpdate: false,
            delete: false,
          },
          agentsManagement: {
            create: false,
            view: false,
            editUpdate: false,
            delete: false,
          },
          jobsManagement: {
            create: false,
            view: true,
            edit: true,
            update: true,
            assignToAgents: false,
            approveReject: false,
          },
        },
        isDefault: false,
        status: 'ACTIVE',
        createdBy: tenantId,
      });
      await agentRole.save();
      created.push('Agent');
      logger.info(`Agent role created for tenant: ${tenantId}`);
    }

    return created;
  } catch (error) {
    logger.error('Create default roles error:', error);
    throw error;
  }
};

/**
 * Main function
 */
const main = async () => {
  try {
    // Connect to MongoDB
    const mongoURI = process.env.MONGODB_URI || 'mongodb://135.181.103.182:1200/hyderabad';
    await mongoose.connect(mongoURI, {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
    console.log('✅ Connected to MongoDB');

    // Get all companies
    const companies = await Tenant.find({});
    console.log(`📋 Found ${companies.length} companies`);

    let totalCreated = 0;
    let totalSkipped = 0;
    let errors = [];

    // Process each company
    for (const company of companies) {
      try {
        const created = await createDefaultRoles(company._id);
        if (created.length > 0) {
          totalCreated += created.length;
          console.log(`✅ Created roles [${created.join(', ')}] for company: ${company.name} (${company._id})`);
        } else {
          totalSkipped += 2; // Both roles already exist
          console.log(`⏭️  Skipped company: ${company.name} (roles already exist)`);
        }
      } catch (error) {
        errors.push({ company: company.name, error: error.message });
        console.error(`❌ Error processing company ${company.name}:`, error.message);
      }
    }

    // Summary
    console.log('\n📊 Summary:');
    console.log(`   Total companies processed: ${companies.length}`);
    console.log(`   Roles created: ${totalCreated}`);
    console.log(`   Roles skipped (already exist): ${totalSkipped}`);
    if (errors.length > 0) {
      console.log(`   Errors: ${errors.length}`);
      errors.forEach(err => {
        console.log(`     - ${err.company}: ${err.error}`);
      });
    }

    console.log('\n✅ Script completed successfully');
    process.exit(0);
  } catch (error) {
    console.error('❌ Script failed:', error);
    process.exit(1);
  }
};

// Run the script
main();

