/**
 * Periodic Maintenance Routes
 */

const express = require('express');
const router = express.Router();
const periodicMaintenanceController = require('../controllers/periodicMaintenance.controller');
const { authorize } = require('../middleware/auth.middleware');

// Get schedules (for managers - unassigned schedules)
router.get('/schedules', authorize('ADMIN', 'MANAGER'), periodicMaintenanceController.getSchedules);

// Get agent schedules (for agents - their assigned schedules)
router.get('/schedules/agent/my-schedules', authorize('SERVICE_AGENT', 'EMPLOYEE'), periodicMaintenanceController.getAgentSchedules);

// Get customer schedules (for customers - their assigned schedules)
router.get('/schedules/customer/my-schedules', authorize('CUSTOMER'), periodicMaintenanceController.getCustomerSchedules);

// Get schedule by ID
router.get('/schedules/:id', authorize('ADMIN', 'MANAGER', 'SERVICE_AGENT', 'EMPLOYEE', 'CUSTOMER'), periodicMaintenanceController.getScheduleById);

// Create schedule
router.post('/schedules', authorize('ADMIN'), periodicMaintenanceController.validateCreateSchedule, periodicMaintenanceController.createSchedule);

// Update schedule
router.patch('/schedules/:id', authorize('ADMIN'), periodicMaintenanceController.updateSchedule);

// Delete schedule
router.delete('/schedules/:id', authorize('ADMIN'), periodicMaintenanceController.deleteSchedule);

// Generate jobs from schedule
router.post('/schedules/:scheduleId/generate-jobs', authorize('ADMIN'), periodicMaintenanceController.generateJobs);

// Get schedule jobs
router.get('/schedules/:scheduleId/jobs', authorize('ADMIN', 'MANAGER'), periodicMaintenanceController.getScheduleJobs);

// Assign schedule to agent (by manager)
router.patch('/schedules/:id/assign-agent', authorize('ADMIN', 'MANAGER'), periodicMaintenanceController.assignScheduleToAgent);

// Assign agent to specific date in schedule
router.patch('/schedules/:id/assign-agent-date', authorize('ADMIN', 'MANAGER'), periodicMaintenanceController.assignAgentToScheduleDate);

module.exports = router;

