/**
 * Job Routes
 */

const express = require('express');
const router = express.Router();
const jobController = require('../controllers/job.controller');
const { authorize } = require('../middleware/auth.middleware');

// Get jobs
router.get('/', jobController.getJobs);

// Get unassigned jobs (MUST be before /:id route to avoid route conflict)
router.get('/unassigned', authorize('MANAGER', 'EMPLOYEE'), jobController.getUnassignedJobs);

// Get unassigned jobs for agent (MUST be before /:id route to avoid route conflict)
router.get('/unassigned-for-agent', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.getUnassignedJobsForAgent);

// Get pending signature jobs (MUST be before /:id route to avoid route conflict)
router.get('/pending-signatures', authorize('MANAGER', 'SERVICE_AGENT', 'EMPLOYEE', 'CUSTOMER'), jobController.getPendingSignatureJobs);

// Get job by ID
router.get('/:id', jobController.getJobById);

// Create job (Admin only)
router.post('/', authorize('ADMIN', 'SUPER_ADMIN'), jobController.createJob);

// Accept job (Agent/Employee only - same access as job updates)
router.patch('/:id/accept', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.acceptJob);

// Start job (Agent/Employee only - same access as job updates)
router.patch('/:id/start', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.validateStartJob, jobController.startJob);

// Submit job (Agent/Employee only - same access as job updates)
router.patch('/:id/submit', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.validateSubmitJob, jobController.submitJob);

// Arrive at job (Agent/Employee only)
router.patch('/:id/arrive', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.arriveJob);

// Update job status (Agent/Employee only)
router.patch('/:id/status', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.validateUpdateJobStatus, jobController.updateJobStatus);

// Get job timeline (Agent/Employee/Manager/Admin)
router.get('/:id/timeline', authorize('SERVICE_AGENT', 'EMPLOYEE', 'MANAGER', 'ADMIN'), jobController.getJobTimeline);

// Get pending jobs count (Agent/Employee only)
router.get('/pending/count', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.getPendingJobsCount);

// Close job by manager (Manager only - including Employee with Manager role)
router.patch('/:id/close', authorize('MANAGER', 'EMPLOYEE'), jobController.validateCloseJob, jobController.closeJobByManager);

// Assign job to agent (Manager only)
router.patch('/:id/assign-agent', authorize('MANAGER', 'EMPLOYEE'), jobController.validateAssignAgent, jobController.assignJobToAgent);

// Agent self-assignment (Agent only - can take jobs assigned to their manager)
router.patch('/:id/self-assign', authorize('SERVICE_AGENT', 'EMPLOYEE'), jobController.selfAssignJob);

// Sign job (Manager and Customer only - for SIGNATURE_PENDING jobs)
router.patch('/:id/sign', authorize('MANAGER', 'CUSTOMER', 'EMPLOYEE'), jobController.signJob);

module.exports = router;

