/**
 * Tenant Model
 */

const mongoose = require('mongoose');

const tenantSchema = new mongoose.Schema({
  name: {
    type: String,
    required: true,
    trim: true,
  },
  email: {
    type: String,
    required: true,
    lowercase: true,
    trim: true,
    unique: true,
  },
  password: {
    type: String,
    required: true,
    select: false, // Don't return password by default
  },
  phone: {
    type: String,
    trim: true,
  },
  address: {
    street: String,
    city: String,
    state: String,
    zipCode: String,
    country: String,
  },
  // White-label configuration
  logoUrl: {
    type: String,
  },
  primaryColor: {
    type: String,
    default: '#3B82F6', // Default blue
  },
  secondaryColor: {
    type: String,
    default: '#10B981', // Default green
  },
  // Subscription
  subscriptionTier: {
    type: String,
    enum: ['BASIC', 'PRO', 'ENTERPRISE'],
    default: 'BASIC',
  },
  subscriptionExpiry: {
    type: Date,
  },
  isActive: {
    type: Boolean,
    default: true,
  },
  status: {
    type: String,
    enum: ['ACTIVE', 'INACTIVE', 'SUSPENDED'],
    default: 'ACTIVE',
  },
  // Role field - all companies are ADMIN by default
  role: {
    type: String,
    enum: ['ADMIN'],
    default: 'ADMIN',
  },
}, {
  timestamps: true,
  collection: 'companies', // Store in 'companies' collection
});

// Hash password before saving
const bcrypt = require('bcryptjs');
tenantSchema.pre('save', async function(next) {
  if (!this.isModified('password')) {
    return next();
  }

  try {
    const salt = await bcrypt.genSalt(12);
    this.password = await bcrypt.hash(this.password, salt);
    next();
  } catch (error) {
    next(error);
  }
});

module.exports = mongoose.model('Tenant', tenantSchema);

