/**
 * Product Model
 * Products available for customers (products, spares, accessories)
 */

const mongoose = require('mongoose');

const productSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  productId: {
    type: String,
    required: false, // Auto-generated in pre-save hook
    unique: true,
    sparse: true, // Allow null values for uniqueness check
    index: true,
  },
  name: {
    type: String,
    required: true,
    trim: true,
  },
  hsnCode: {
    type: String,
    trim: true,
  },
  description: {
    type: String,
    trim: true,
  },
  year: {
    type: Number,
  },
  productType: {
    type: String,
    enum: ['product', 'spare', 'accessories'],
    required: true,
    default: 'product',
  },
  status: {
    type: String,
    enum: ['ACTIVE', 'INACTIVE'],
    default: 'ACTIVE',
  },
  createdBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
  },
}, {
  timestamps: true,
});

// Generate product ID before saving (only for new documents)
productSchema.pre('save', async function(next) {
  // Only generate productId for new documents that don't have one
  if (this.isNew && !this.productId) {
    try {
      const Product = mongoose.model('Product');
      const Tenant = mongoose.model('Tenant');
      
      const count = await Product.countDocuments({ tenantId: this.tenantId });
      const tenant = await Tenant.findById(this.tenantId);
      
      // Generate prefix from company name (first 3 letters) or use default
      let prefix = 'PRD';
      if (tenant && tenant.name) {
        const nameParts = tenant.name.trim().split(' ');
        if (nameParts.length > 0) {
          prefix = nameParts[0].substring(0, 3).toUpperCase().replace(/[^A-Z]/g, '');
          if (prefix.length < 2) prefix = 'PRD';
        }
      }
      
      this.productId = `${prefix}-${String(count + 1).padStart(4, '0')}`;
    } catch (error) {
      return next(error);
    }
  }
  next();
});

module.exports = mongoose.model('Product', productSchema);

