/**
 * Periodic Maintenance Schedule Model
 */

const mongoose = require('mongoose');

const scheduleSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  // Customer and Product
  customerId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Customer',
    required: true,
  },
  customerName: {
    type: String,
    required: true,
  },
  productId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Product',
  },
  productName: {
    type: String,
  },
  // Work Order
  workOrderNo: {
    type: String,
    required: true,
    unique: true,
    index: true,
  },
  // Manager
  managerId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true,
  },
  managerName: {
    type: String,
    required: true,
  },
  // Assigned Agent (auto-assigned based on least schedules)
  assignedAgentId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
  },
  assignedAgentName: {
    type: String,
  },
  // Date-specific agent assignments (for individual date agent changes)
  dateAgentAssignments: [{
    date: {
      type: Date,
      required: true,
    },
    agentId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User',
      required: true,
    },
    agentName: {
      type: String,
      required: true,
    },
  }],
  // Work Type (from Periodic Maintenance Config)
  workType: {
    type: String,
  },
  workTypeId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'PeriodicMaintenanceConfig',
  },
  // Frequency
  frequency: {
    type: String,
    enum: ['DAILY', 'WEEKLY', 'MONTHLY', 'QUARTERLY', 'YEARLY'],
    required: true,
  },
  // Period
  periodFrom: {
    type: Date,
    required: true,
  },
  periodTo: {
    type: Date,
    required: true,
  },
  // Address
  address: {
    doorNumber: { type: String },
    floor: { type: String },
    apartmentBlockName: { type: String },
    native: { type: String },
    pincode: { type: String },
    district: { type: String },
    state: { type: String },
  },
  // Remarks
  remarks: {
    type: String,
  },
  // Status
  isActive: {
    type: Boolean,
    default: true,
  },
  // Legacy fields for backward compatibility
  name: {
    type: String,
  },
  description: {
    type: String,
  },
  serviceType: {
    type: String,
  },
  recurrenceInterval: {
    type: String,
    enum: ['DAILY', 'WEEKLY', 'MONTHLY', 'QUARTERLY', 'SEMI_ANNUAL', 'YEARLY'],
  },
  assignedAgentIds: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
  }],
  customerIds: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
  }],
  startDate: {
    type: Date,
  },
  endDate: {
    type: Date,
  },
  lastGeneratedAt: {
    type: Date,
  },
  nextGenerationDate: {
    type: Date,
  },
}, {
  timestamps: true,
});

// Indexes
scheduleSchema.index({ tenantId: 1, isActive: 1 });
scheduleSchema.index({ tenantId: 1, nextGenerationDate: 1 });
scheduleSchema.index({ tenantId: 1, customerId: 1 });
scheduleSchema.index({ tenantId: 1, managerId: 1 });
scheduleSchema.index({ tenantId: 1, assignedAgentId: 1 });
scheduleSchema.index({ workOrderNo: 1 });

module.exports = mongoose.model('PeriodicMaintenanceSchedule', scheduleSchema);

