/**
 * Periodic Maintenance Configuration Model
 * Stores configuration templates for periodic maintenance with dynamic fields
 */

const mongoose = require('mongoose');

const periodicMaintenanceConfigSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  name: {
    type: String,
    required: true,
    trim: true,
  },
  productId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Product',
    required: true,
    unique: true, // One configuration per product
    index: true,
  },
  // Dynamic fields array
  fields: [{
    fieldType: {
      type: String,
      enum: ['CHECKLIST', 'NUMBER_INPUT', 'RADIO_BUTTON'],
      required: true,
    },
    fieldName: {
      type: String,
      required: true,
      trim: true,
    },
    isRequired: {
      type: Boolean,
      default: false,
    },
    // For CHECKLIST fields
    checklistItems: [{
      name: {
        type: String,
        required: true,
        trim: true,
      },
    }],
    // For NUMBER_INPUT fields
    numberInput: {
      minValue: {
        type: Number,
      },
      maxValue: {
        type: Number,
      },
      rangeName: {
        type: String,
        trim: true,
      },
      noRange: {
        type: Boolean,
        default: false, // If true, min/max are not required
      },
    },
    // For RADIO_BUTTON fields
    radioButton: {
      groupName: {
        type: String,
        trim: true,
      },
      options: [{
        label: {
          type: String,
          required: true,
          trim: true,
        },
        value: {
          type: String,
          required: true,
          trim: true,
        },
      }],
    },
    order: {
      type: Number,
      default: 0,
    },
  }],
  isActive: {
    type: Boolean,
    default: true,
  },
  createdBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
  },
}, {
  timestamps: true,
});

// Indexes
periodicMaintenanceConfigSchema.index({ tenantId: 1, productId: 1 }, { unique: true });
periodicMaintenanceConfigSchema.index({ tenantId: 1, isActive: 1 });

module.exports = mongoose.model('PeriodicMaintenanceConfig', periodicMaintenanceConfigSchema);

