/**
 * KPI Matrix Model
 * Based on the KPI Matrix Report structure from the PDF
 */

const mongoose = require('mongoose');

const kpiMatrixSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  contractor: {
    type: String,
    required: true,
  },
  site: {
    type: String,
    required: true,
  },
  month: {
    type: Number,
    required: true,
    min: 1,
    max: 12,
  },
  year: {
    type: Number,
    required: true,
  },
  // Indicator 1: Response time to phone call
  responseTimePhoneCall: {
    officeHours: {
      limit: { type: Number, default: 0.5 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
    nonOfficeHours: {
      limit: { type: Number, default: 2 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
    sundayHolidaysAfterMidnight: {
      limit: { type: Number, default: 2 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
  },
  // Response time (on site)
  responseTimeOnSite: {
    critical: {
      limit: { type: Number, default: 0.5 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
    nonCritical: {
      limit: { type: Number, default: 1 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
  },
  // Response time (off site)
  responseTimeOffSite: {
    critical: {
      limit: { type: Number, default: 2 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
    nonCritical: {
      limit: { type: Number, default: 3 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
  },
  // Indicator 2: Number of maintenance executed as per schedule
  maintenanceExecuted: {
    target: { type: Number, default: 100 }, // Percentage
    actual: { type: Number, default: 0 },
    pointsDeducted: { type: Number, default: 0 },
    equipmentMissed: { type: Number, default: 0 },
    remarks: String,
    clientAgreement: String,
  },
  // Indicator 3: Uptime for equipment (Monthly)
  equipmentUptime: {
    target: { type: Number, default: 98 }, // Percentage
    actual: { type: Number, default: 0 },
    pointsDeducted: { type: Number, default: 0 },
    remarks: String,
    clientAgreement: String,
  },
  // Indicator 4: Response time to resolve breakdown (downtime)
  breakdownResolutionTime: {
    critical: {
      limit: { type: Number, default: 8 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
    urgent: {
      limit: { type: Number, default: 12 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
    normal: {
      limit: { type: Number, default: 48 }, // Hours
      pointsDeducted: { type: Number, default: 0 },
      instances: { type: Number, default: 0 },
      remarks: String,
      clientAgreement: String,
    },
  },
  // Indicator 5: Number of emergency / call-outs attended
  emergencyCallouts: {
    target: { type: Number, default: 100 }, // Percentage
    actual: { type: Number, default: 0 },
    total: { type: Number, default: 0 },
    attended: { type: Number, default: 0 },
    pointsDeducted: { type: Number, default: 0 },
    instances: { type: Number, default: 0 },
    remarks: String,
    clientAgreement: String,
  },
  // Indicator 6: Late reporting to work at Building
  lateReporting: {
    pointsDeducted: { type: Number, default: 0 },
    instances: { type: Number, default: 0 },
    remarks: String,
    clientAgreement: String,
  },
  // Indicator 7: Plant efficiency (GM2021 <=0.74 kW/RT (TSE) GM NRB 4.0 <=0.57 kW/RT)
  plantEfficiency: {
    gm2021Limit: { type: Number, default: 0.74 }, // kW/RT
    gm2021Actual: { type: Number, default: 0 },
    gmNRB40Limit: { type: Number, default: 0.57 }, // kW/RT
    gmNRB40Actual: { type: Number, default: 0 },
    pointsDeducted: { type: Number, default: 0 },
    remarks: String,
    clientAgreement: String,
  },
  // Indicator 8: Operational improvement measures suggested
  operationalImprovements: {
    target: { type: Number, default: 20 },
    actual: { type: Number, default: 0 },
    pointsDeducted: { type: Number, default: 0 },
    suggestions: [{
      title: String,
      description: String,
      suggestedDate: Date,
    }],
    remarks: String,
    clientAgreement: String,
  },
  // Total points deduction
  totalDeduction: {
    type: Number,
    default: 0,
  },
  // Service Grade
  serviceGrade: {
    type: String,
    enum: ['A', 'B', 'C', 'D'],
    default: 'A',
  },
  // Status
  status: {
    type: String,
    enum: ['DRAFT', 'SUBMITTED', 'APPROVED', 'REJECTED'],
    default: 'DRAFT',
  },
  // Created and updated by
  createdBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
  },
  updatedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
  },
}, {
  timestamps: true,
});

// Indexes
kpiMatrixSchema.index({ tenantId: 1, year: 1, month: 1 });
kpiMatrixSchema.index({ tenantId: 1, contractor: 1, site: 1 });
kpiMatrixSchema.index({ tenantId: 1, status: 1 });

// Calculate service grade based on total deduction
kpiMatrixSchema.methods.calculateServiceGrade = function() {
  if (this.totalDeduction <= 15) {
    this.serviceGrade = 'A';
  } else if (this.totalDeduction <= 30) {
    this.serviceGrade = 'B';
  } else if (this.totalDeduction <= 50) {
    this.serviceGrade = 'C';
  } else {
    this.serviceGrade = 'D';
  }
};

// Calculate total deduction before saving
kpiMatrixSchema.pre('save', function(next) {
  let total = 0;
  
  // Response time phone call
  total += this.responseTimePhoneCall.officeHours.pointsDeducted || 0;
  total += this.responseTimePhoneCall.nonOfficeHours.pointsDeducted || 0;
  total += this.responseTimePhoneCall.sundayHolidaysAfterMidnight.pointsDeducted || 0;
  
  // Response time on site
  total += this.responseTimeOnSite.critical.pointsDeducted || 0;
  total += this.responseTimeOnSite.nonCritical.pointsDeducted || 0;
  
  // Response time off site
  total += this.responseTimeOffSite.critical.pointsDeducted || 0;
  total += this.responseTimeOffSite.nonCritical.pointsDeducted || 0;
  
  // Maintenance executed
  total += this.maintenanceExecuted.pointsDeducted || 0;
  
  // Equipment uptime
  total += this.equipmentUptime.pointsDeducted || 0;
  
  // Breakdown resolution
  total += this.breakdownResolutionTime.critical.pointsDeducted || 0;
  total += this.breakdownResolutionTime.urgent.pointsDeducted || 0;
  total += this.breakdownResolutionTime.normal.pointsDeducted || 0;
  
  // Emergency callouts
  total += this.emergencyCallouts.pointsDeducted || 0;
  
  // Late reporting
  total += this.lateReporting.pointsDeducted || 0;
  
  // Plant efficiency
  total += this.plantEfficiency.pointsDeducted || 0;
  
  // Operational improvements
  total += this.operationalImprovements.pointsDeducted || 0;
  
  this.totalDeduction = total;
  this.calculateServiceGrade();
  next();
});

module.exports = mongoose.model('KPIMatrix', kpiMatrixSchema);

