/**
 * Job Activity Model
 * Tracks timeline of job activities (created, arrived, opened, status updates, etc.)
 */

const mongoose = require('mongoose');

const jobActivitySchema = new mongoose.Schema({
  jobId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Job',
    required: true,
    index: true,
  },
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  activityType: {
    type: String,
    enum: [
      'JOB_CREATED',
      'JOB_ASSIGNED',
      'JOB_ACCEPTED',
      'JOB_ARRIVED',
      'JOB_OPENED',
      'JOB_STARTED',
      'STATUS_UPDATED',
      'PHOTO_ADDED',
      'JOB_SUBMITTED',
      'JOB_APPROVED',
      'JOB_REJECTED',
      'JOB_CLOSED',
    ],
    required: true,
  },
  performedBy: {
    type: mongoose.Schema.Types.ObjectId,
    refPath: 'performedByModel',
    required: true,
  },
  performedByModel: {
    type: String,
    enum: ['User', 'Employee'],
    required: true,
  },
  performedByName: {
    type: String,
    required: true,
  },
  description: {
    type: String,
    trim: true,
  },
  // For status updates
  oldStatus: {
    type: String,
  },
  newStatus: {
    type: String,
  },
  // For location-based activities
  location: {
    latitude: { type: Number },
    longitude: { type: Number },
    locationName: { type: String },
    address: { type: String },
  },
  // Additional metadata
  metadata: {
    type: mongoose.Schema.Types.Mixed,
    default: {},
  },
  // Custom timestamp from mobile device (if provided, overrides createdAt)
  timestamp: {
    type: Date,
  },
}, {
  timestamps: true,
  collection: 'job_activities',
});

// Pre-save hook to use custom timestamp if provided
jobActivitySchema.pre('save', function(next) {
  // If custom timestamp is provided, use it for createdAt
  if (this.timestamp && !this.createdAt) {
    this.createdAt = this.timestamp;
  }
  next();
});

// Indexes for efficient queries
jobActivitySchema.index({ jobId: 1, createdAt: -1 });
jobActivitySchema.index({ tenantId: 1, createdAt: -1 });
jobActivitySchema.index({ activityType: 1, createdAt: -1 });

module.exports = mongoose.model('JobActivity', jobActivitySchema);

