/**
 * Job Model
 */

const mongoose = require('mongoose');

const jobSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  ticketType: {
    type: String,
    enum: ['PERIODIC', 'COMPLIANCE'],
    required: true,
  },
  ticketNumber: {
    type: String,
    unique: true,
    index: true,
  },
  // Customer information
  customerId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true,
    index: true,
  },
  customerName: {
    type: String,
    required: true,
  },
  customerPhone: {
    type: String,
  },
  customerEmail: {
    type: String,
  },
  // Assignment - can be either User (SERVICE_AGENT) or Employee
  assignedAgentId: {
    type: mongoose.Schema.Types.ObjectId,
    refPath: 'assignedAgentModel',
    index: true,
  },
  assignedAgentModel: {
    type: String,
    enum: ['User', 'Employee'],
    default: 'User',
  },
  managerId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    index: true,
  },
  // Status tracking
  status: {
    type: String,
    enum: ['NEW', 'ACCEPTED', 'ARRIVED', 'IN_PROGRESS', 'PENDING_APPROVAL', 'SIGNATURE_PENDING', 'APPROVED', 'REJECTED', 'CLOSED', 'CANCELLED'],
    default: 'NEW',
    index: true,
  },
  // Location
  jobLocationLat: {
    type: Number,
    required: true,
  },
  jobLocationLng: {
    type: Number,
    required: true,
  },
  jobLocationAddress: {
    type: String,
    required: true,
  },
  // SLA
  slaDeadline: {
    type: Date,
    index: true,
  },
  slaBreached: {
    type: Boolean,
    default: false,
  },
  // Priority (for compliance tickets)
  priority: {
    type: String,
    enum: ['IMPORTANT', 'NOT_IMPORTANT', 'GENERAL'],
  },
  // Timestamps
  createdAt: {
    type: Date,
    default: Date.now,
  },
  acceptedAt: {
    type: Date,
  },
  arrivedAt: {
    type: Date,
  },
  startedAt: {
    type: Date,
  },
  completedAt: {
    type: Date,
  },
  closedAt: {
    type: Date,
  },
  // Track when job was first opened/viewed by agent
  firstOpenedAt: {
    type: Date,
  },
  // Periodic maintenance specific
  scheduleId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'PeriodicMaintenanceSchedule',
  },
  scheduledDate: {
    type: Date,
  },
  // Compliance specific
  complaintDescription: {
    type: String,
  },
  complaintCategory: {
    type: String,
  },
  complaintCategories: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'ComplaintCategory',
  }],
  complaintAddress: {
    flatHouseNumber: { type: String },
    floorNumber: { type: String },
    streetAddress: { type: String },
    area: { type: String },
    landmark: { type: String },
    city: { type: String },
    pincode: { type: String },
    state: { type: String },
    fullAddress: { type: String },
  },
  complaintContact: {
    name: { type: String },
    number: { type: String },
    alternativeNumber: { type: String },
  },
  complaintPhotos: [{
    type: String, // URLs to uploaded photos
  }],
  productId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Product',
  },
  // Signature
  signatureUrl: {
    type: String,
  },
  signatureData: {
    type: String, // Base64 encoded signature image
  },
  signeeName: {
    type: String,
  },
  isSigned: {
    type: Boolean,
    default: false,
  },
  // Linked jobs (for spares)
  parentJobId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Job',
  },
  linkedJobIds: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Job',
  }],
}, {
  timestamps: true,
});

// Indexes
jobSchema.index({ tenantId: 1, status: 1 });
jobSchema.index({ tenantId: 1, assignedAgentId: 1 });
jobSchema.index({ tenantId: 1, managerId: 1 });
jobSchema.index({ tenantId: 1, customerId: 1 });
jobSchema.index({ tenantId: 1, ticketType: 1 });
jobSchema.index({ slaDeadline: 1 });

// Generate ticket number before saving
jobSchema.pre('save', async function(next) {
  if (!this.ticketNumber) {
    const prefix = this.ticketType === 'PERIODIC' ? 'PM' : 'CM';
    const count = await mongoose.model('Job').countDocuments({ tenantId: this.tenantId });
    this.ticketNumber = `${prefix}-${String(count + 1).padStart(6, '0')}`;
  }
  next();
});

module.exports = mongoose.model('Job', jobSchema);

