/**
 * Employee Model
 * Employees are users created by company admins with specific roles and permissions
 */

const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');

const employeeSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  employeeId: {
    type: String,
    required: false, // Auto-generated in pre-save hook
    unique: true,
    sparse: true, // Allow null values for uniqueness check
    index: true,
  },
  firstName: {
    type: String,
    required: true,
    trim: true,
  },
  lastName: {
    type: String,
    required: true,
    trim: true,
  },
  email: {
    type: String,
    required: true,
    lowercase: true,
    trim: true,
    unique: true,
    index: true,
  },
  contactNumber: {
    type: String,
    required: true,
    trim: true,
  },
  password: {
    type: String,
    required: true,
    select: false, // Don't return password by default
  },
  role: {
    type: String,
    required: true,
    enum: ['EMPLOYEE'],
    default: 'EMPLOYEE',
  },
  // Reference to Role model
  roleId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Role',
    required: true,
  },
  // Assigned clients (for data isolation)
  assignedClients: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User', // Customer users
  }],
  // Assigned jobs (for data isolation)
  assignedJobs: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Job',
  }],
  status: {
    type: String,
    enum: ['ACTIVE', 'INACTIVE', 'SUSPENDED'],
    default: 'ACTIVE',
  },
  createdBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant', // Company admin who created this employee
  },
}, {
  timestamps: true,
});

// Hash password before saving
employeeSchema.pre('save', async function(next) {
  if (!this.isModified('password')) {
    return next();
  }

  try {
    const salt = await bcrypt.genSalt(12);
    this.password = await bcrypt.hash(this.password, salt);
    next();
  } catch (error) {
    next(error);
  }
});

// Method to compare password
employeeSchema.methods.comparePassword = async function(candidatePassword) {
  return bcrypt.compare(candidatePassword, this.password);
};

// Generate employee ID before saving (only for new documents)
employeeSchema.pre('save', async function(next) {
  // Only generate employeeId for new documents that don't have one
  if (this.isNew && !this.employeeId) {
    try {
      const Employee = mongoose.model('Employee');
      const Tenant = mongoose.model('Tenant');
      
      const count = await Employee.countDocuments({ tenantId: this.tenantId });
      const tenant = await Tenant.findById(this.tenantId);
      
      // Generate prefix from company name (first 3 letters) or use default
      let prefix = 'EMP';
      if (tenant && tenant.name) {
        const nameParts = tenant.name.trim().split(' ');
        if (nameParts.length > 0) {
          prefix = nameParts[0].substring(0, 3).toUpperCase().replace(/[^A-Z]/g, '');
          if (prefix.length < 2) prefix = 'EMP';
        }
      }
      
      this.employeeId = `${prefix}-${String(count + 1).padStart(4, '0')}`;
    } catch (error) {
      return next(error);
    }
  }
  next();
});

// Virtual for full name
employeeSchema.virtual('fullName').get(function() {
  return `${this.firstName} ${this.lastName}`;
});

module.exports = mongoose.model('Employee', employeeSchema);

