/**
 * Customer Model
 * Customers with products assigned
 */

const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');

const customerSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  customerId: {
    type: String,
    required: false, // Auto-generated in pre-save hook
    unique: true,
    index: true,
  },
  name: {
    type: String,
    required: true,
    trim: true,
  },
  email: {
    type: String,
    required: true,
    lowercase: true,
    trim: true,
    unique: true,
    index: true,
  },
  contactNumber: {
    type: String,
    required: true,
    trim: true,
  },
  password: {
    type: String,
    required: true,
    select: false,
  },
  address: {
    blockName: { type: String, trim: true },
    streetName: { type: String, trim: true },
    floorNumber: { type: String, trim: true },
    flatNumber: { type: String, trim: true },
    fullAddress: { type: String, trim: true }, // Combined address for display
  },
  products: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Product',
  }],
  status: {
    type: String,
    enum: ['ACTIVE', 'INACTIVE', 'SUSPENDED'],
    default: 'ACTIVE',
  },
  createdBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
  },
}, {
  timestamps: true,
});

// Hash password before saving
customerSchema.pre('save', async function(next) {
  if (!this.isModified('password')) {
    return next();
  }

  try {
    const salt = await bcrypt.genSalt(12);
    this.password = await bcrypt.hash(this.password, salt);
    next();
  } catch (error) {
    next(error);
  }
});

// Method to compare password
customerSchema.methods.comparePassword = async function(candidatePassword) {
  return bcrypt.compare(candidatePassword, this.password);
};

// Generate customer ID before saving (runs before validation)
customerSchema.pre('save', async function(next) {
  // Only generate if this is a new document and customerId is not already set
  if (!this.isNew || this.customerId) {
    // Build full address string even for existing documents
    if (this.isModified('address') && this.address) {
      const addressParts = [];
      if (this.address.blockName) addressParts.push(`Block: ${this.address.blockName}`);
      if (this.address.streetName) addressParts.push(this.address.streetName);
      if (this.address.floorNumber) addressParts.push(`Floor: ${this.address.floorNumber}`);
      if (this.address.flatNumber) addressParts.push(`Flat: ${this.address.flatNumber}`);
      this.address.fullAddress = addressParts.join(', ');
    }
    return next();
  }

  try {
    const Customer = mongoose.model('Customer');
    const Tenant = mongoose.model('Tenant');
    
    // Ensure tenantId is set
    if (!this.tenantId) {
      return next(new Error('Tenant ID is required'));
    }
    
    // Count existing customers for this tenant
    const count = await Customer.countDocuments({ tenantId: this.tenantId });
    const tenant = await Tenant.findById(this.tenantId);
    
    // Generate prefix from company name (first 3 letters) or use default
    let prefix = 'CUS';
    if (tenant && tenant.name) {
      const nameParts = tenant.name.trim().split(' ');
      if (nameParts.length > 0) {
        prefix = nameParts[0].substring(0, 3).toUpperCase().replace(/[^A-Z]/g, '');
        if (prefix.length < 2) prefix = 'CUS';
      }
    }
    
    // Generate unique customer ID
    this.customerId = `${prefix}-${String(count + 1).padStart(4, '0')}`;
    
    // Build full address string
    if (this.address) {
      const addressParts = [];
      if (this.address.blockName) addressParts.push(`Block: ${this.address.blockName}`);
      if (this.address.streetName) addressParts.push(this.address.streetName);
      if (this.address.floorNumber) addressParts.push(`Floor: ${this.address.floorNumber}`);
      if (this.address.flatNumber) addressParts.push(`Flat: ${this.address.flatNumber}`);
      this.address.fullAddress = addressParts.join(', ');
    }
    
    next();
  } catch (error) {
    next(error);
  }
});

module.exports = mongoose.model('Customer', customerSchema);

