/**
 * Attendance Model
 * Tracks employee attendance (check-in, check-out, breaks, etc.)
 */

const mongoose = require('mongoose');

const attendanceSchema = new mongoose.Schema({
  tenantId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Tenant',
    required: true,
    index: true,
  },
  employeeId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Employee',
    required: true,
    index: true,
  },
  date: {
    type: Date,
    required: true,
    index: true,
  },
  checkIn: {
    time: {
      type: Date,
    },
    location: {
      latitude: { type: Number },
      longitude: { type: Number },
      locationName: { type: String },
      address: { type: String },
    },
    method: {
      type: String,
      enum: ['MOBILE_APP', 'WEB', 'MANUAL'],
      default: 'MOBILE_APP',
    },
  },
  checkOut: {
    time: {
      type: Date,
    },
    location: {
      latitude: { type: Number },
      longitude: { type: Number },
      locationName: { type: String },
      address: { type: String },
    },
    method: {
      type: String,
      enum: ['MOBILE_APP', 'WEB', 'MANUAL'],
      default: 'MOBILE_APP',
    },
  },
  breaks: [{
    startTime: {
      type: Date,
      required: true,
    },
    endTime: {
      type: Date,
    },
    duration: {
      type: Number, // in minutes
    },
    type: {
      type: String,
      enum: ['LUNCH', 'TEA', 'PERSONAL', 'OTHER'],
      default: 'OTHER',
    },
    location: {
      latitude: { type: Number },
      longitude: { type: Number },
      locationName: { type: String },
    },
  }],
  totalWorkHours: {
    type: Number, // in hours, calculated
  },
  status: {
    type: String,
    enum: ['PRESENT', 'ABSENT', 'LATE', 'HALF_DAY', 'ON_LEAVE'],
    default: 'PRESENT',
  },
  notes: {
    type: String,
    trim: true,
  },
  approvedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
  },
  approvedAt: {
    type: Date,
  },
}, {
  timestamps: true,
  collection: 'attendances',
});

// Index for efficient queries
attendanceSchema.index({ tenantId: 1, employeeId: 1, date: 1 }, { unique: true });
attendanceSchema.index({ tenantId: 1, date: -1 });
attendanceSchema.index({ employeeId: 1, date: -1 });

// Pre-save hook to calculate total work hours
attendanceSchema.pre('save', function(next) {
  if (this.checkIn && this.checkIn.time && this.checkOut && this.checkOut.time) {
    const checkInTime = new Date(this.checkIn.time);
    const checkOutTime = new Date(this.checkOut.time);
    const totalMinutes = (checkOutTime - checkInTime) / (1000 * 60);
    
    // Subtract break durations
    let breakMinutes = 0;
    if (this.breaks && this.breaks.length > 0) {
      this.breaks.forEach(breakItem => {
        if (breakItem.duration) {
          breakMinutes += breakItem.duration;
        } else if (breakItem.startTime && breakItem.endTime) {
          const breakStart = new Date(breakItem.startTime);
          const breakEnd = new Date(breakItem.endTime);
          breakMinutes += (breakEnd - breakStart) / (1000 * 60);
        }
      });
    }
    
    this.totalWorkHours = (totalMinutes - breakMinutes) / 60;
  }
  next();
});

module.exports = mongoose.model('Attendance', attendanceSchema);

