/**
 * Authentication Middleware
 */

const { verifyAccessToken } = require('../utils/jwt');
const { errorResponse } = require('../utils/response');
const User = require('../models/User.model');
const Tenant = require('../models/Tenant.model');
const Employee = require('../models/Employee.model');
const logger = require('../utils/logger');

/**
 * Authenticate JWT token
 */
const authenticateToken = async (req, res, next) => {
  try {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN

    if (!token) {
      return errorResponse(res, 'Access token required', 401);
    }

    const decoded = verifyAccessToken(token);
    
    // Debug logging
    logger.info('Token decoded', {
      userId: decoded.userId,
      role: decoded.role,
      email: decoded.email,
      tenantId: decoded.tenantId,
      url: req.url
    });
    
    // Check if it's Super Admin (from users collection)
    if (decoded.role === 'SUPER_ADMIN') {
      const user = await User.findById(decoded.userId)
        .populate('tenantId')
        .select('-password');

      if (!user) {
        return errorResponse(res, 'User not found', 404);
      }

      if (user.status !== 'ACTIVE') {
        return errorResponse(res, 'User account is not active', 403);
      }

      // Attach user to request
      req.user = user;
      req.tenantId = user.tenantId?._id || user.tenantId;
      req.userId = user._id;
      req.userRole = user.role;

      return next();
    }

    // Check if it's a Customer (from customers collection) - CHECK THIS FIRST
    if (decoded.role === 'CUSTOMER') {
      const Customer = require('../models/Customer.model');
      const customer = await Customer.findById(decoded.userId)
        .populate('tenantId')
        .select('-password');

      if (!customer) {
        return errorResponse(res, 'Customer not found', 404);
      }

      if (customer.status !== 'ACTIVE') {
        return errorResponse(res, 'Customer account is not active', 403);
      }

      // Check if company is active
      if (customer.tenantId) {
        const tenant = customer.tenantId._id ? customer.tenantId : await Tenant.findById(customer.tenantId);
        if (!tenant || !tenant.isActive || tenant.status !== 'ACTIVE') {
          return errorResponse(res, 'Company account is not active', 403);
        }
      }

      // Create user-like object for customer
      const customerUser = {
        _id: customer._id,
        email: customer.email,
        name: customer.name,
        role: 'CUSTOMER',
        tenantId: customer.tenantId?._id || customer.tenantId,
        status: customer.status,
        customerId: customer.customerId,
        contactNumber: customer.contactNumber,
        address: customer.address,
        products: customer.products,
      };

      // Attach customer as user to request
      req.user = customerUser;
      req.tenantId = customerUser.tenantId;
      req.userId = customerUser._id;
      req.userRole = customerUser.role;

      return next();
    }

    // Check if it's an Employee (from employees collection)
    if (decoded.role === 'EMPLOYEE') {
      const employee = await Employee.findById(decoded.userId)
        .populate('tenantId')
        .select('-password');

      if (!employee) {
        return errorResponse(res, 'Employee not found', 404);
      }

      if (employee.status !== 'ACTIVE') {
        return errorResponse(res, 'Employee account is not active', 403);
      }

      // Check if company is active
      if (employee.tenantId) {
        const tenant = employee.tenantId._id ? employee.tenantId : await Tenant.findById(employee.tenantId);
        if (!tenant || !tenant.isActive || tenant.status !== 'ACTIVE') {
          return errorResponse(res, 'Company account is not active', 403);
        }
      }

      // Populate role to get permissions
      await employee.populate('roleId', 'name description permissions');

      // Get role name from Role collection
      const roleName = employee.roleId?.name || 'Employee';
      
      // If employee has "Manager" role, check for corresponding User account
      let effectiveRole = 'EMPLOYEE';
      if (roleName && roleName.toLowerCase() === 'manager') {
        // Check if there's a corresponding User account with MANAGER role
        const managerUser = await User.findOne({ 
          email: employee.email, 
          tenantId: employee.tenantId?._id || employee.tenantId,
          role: 'MANAGER' 
        });
        if (managerUser) {
          effectiveRole = 'MANAGER';
        }
      }

      // Create user-like object for employee
      const employeeUser = {
        _id: employee._id,
        email: employee.email,
        name: `${employee.firstName} ${employee.lastName}`,
        role: effectiveRole, // Use effective role (MANAGER if roleName is Manager and User exists)
        tenantId: employee.tenantId?._id || employee.tenantId,
        status: employee.status,
        permissions: employee.roleId?.permissions || {},
        roleName: roleName, // Store the actual role name from Role collection
        assignedClients: employee.assignedClients,
        assignedJobs: employee.assignedJobs,
      };

      // Attach employee as user to request
      req.user = employeeUser;
      req.tenantId = employeeUser.tenantId;
      req.userId = employeeUser._id;
      req.userRole = effectiveRole; // Use effective role for filtering
      req.roleName = roleName; // Store roleName for reference

      return next();
    }

    // Company login (from companies collection) - CHECK THIS BEFORE REGULAR USER
    // If role is ADMIN, check companies collection first
    if (decoded.role === 'ADMIN') {
      logger.info('ADMIN role detected, checking companies collection', {
        userId: decoded.userId,
        url: req.url
      });
      
      // First try to find in companies collection (most common case)
      let company = await Tenant.findById(decoded.userId).select('-password');
      
      if (company) {
        logger.info('Company found in companies collection', {
          companyId: company._id,
          companyName: company.name,
          isActive: company.isActive,
          status: company.status
        });
        
        // Found in companies collection - this is a company login
        if (!company.isActive || company.status !== 'ACTIVE') {
          return errorResponse(res, 'Company account is not active', 403);
        }

        // Create user-like object for company (for compatibility)
        // IMPORTANT: Company owners (from companies collection) have ADMIN role
        // ADMIN role has FULL ACCESS to all endpoints - this is the DEFAULT
        const companyUser = {
          _id: company._id,
          email: company.email,
          name: company.name,
          role: 'ADMIN', // ADMIN role = FULL ACCESS to all endpoints
          tenantId: company._id,
          status: company.status === 'ACTIVE' ? 'ACTIVE' : 'INACTIVE',
        };

        // Attach company as user to request
        req.user = companyUser;
        req.user.role = 'ADMIN'; // Ensure role is set on user object too
        req.tenantId = company._id;
        req.userId = company._id;
        req.userRole = 'ADMIN'; // Explicitly set userRole - ADMIN has all permissions

        logger.info('✅ Company authentication successful - ADMIN role with FULL ACCESS', {
          userRole: req.userRole,
          userId: req.userId,
          tenantId: req.tenantId,
          userObjectRole: req.user.role,
          email: company.email,
          message: 'ADMIN role has full access to all endpoints - no permission checks needed'
        });

        return next();
      }
      
      logger.warn('ADMIN role but company not found in companies collection, checking users collection', {
        userId: decoded.userId
      });
      
      // If not found in companies, try users collection (for backward compatibility)
      // This handles cases where ADMIN users exist in users collection
      const user = await User.findById(decoded.userId).select('-password');
      
      if (user) {
        // Found in users collection - regular ADMIN user
        if (user.tenantId) {
          await user.populate('tenantId');
        }

        if (user.status !== 'ACTIVE') {
          return errorResponse(res, 'User account is not active', 403);
        }

        // Attach user to request
        req.user = user;
        req.tenantId = user.tenantId?._id || user.tenantId;
        req.userId = user._id;
        req.userRole = user.role;

        return next();
      }
      
      // Not found in either collection
      return errorResponse(res, 'Company or user not found', 404);
    }

    // Regular user login (from users collection) - for non-ADMIN roles
    const user = await User.findById(decoded.userId).select('-password');

    // Regular user login (from users collection)
    // Populate tenantId if it exists
    if (user.tenantId) {
      await user.populate('tenantId');
    }

    if (user.status !== 'ACTIVE') {
      return errorResponse(res, 'User account is not active', 403);
    }

    // Attach user to request
    req.user = user;
    req.tenantId = user.tenantId?._id || user.tenantId;
    req.userId = user._id;
    req.userRole = user.role;

    next();
  } catch (error) {
    logger.error('Authentication error:', error);
    if (error.name === 'JsonWebTokenError') {
      return errorResponse(res, 'Invalid token', 401);
    }
    if (error.name === 'TokenExpiredError') {
      return errorResponse(res, 'Token expired', 401);
    }
    return errorResponse(res, 'Authentication failed', 401);
  }
};

/**
 * Role-based authorization middleware
 * ADMIN role (company owner) has FULL ACCESS to all endpoints
 * Other roles are checked against required roles
 */
const authorize = (...roles) => {
  return (req, res, next) => {
    if (!req.user) {
      logger.warn('Authorization failed: No user in request', { 
        url: req.url,
        method: req.method 
      });
      return errorResponse(res, 'Authentication required', 401);
    }

    // Debug logging
    logger.info('Authorization check', {
      url: req.url,
      method: req.method,
      userRole: req.userRole,
      requiredRoles: roles,
      userEmail: req.user?.email,
      userId: req.userId,
      userPermissions: req.user?.permissions
    });

    // ADMIN role (company owner from companies collection) - ALWAYS HAS FULL ACCESS
    // This is the DEFAULT - ADMIN has all permissions to all endpoints
    if (req.userRole === 'ADMIN') {
      logger.info('✅ Authorization granted: ADMIN role (Company Owner) - FULL ACCESS', {
        url: req.url,
        method: req.method,
        userRole: req.userRole,
        userEmail: req.user?.email,
        userId: req.userId,
        message: 'ADMIN role has full access to all endpoints - no permission checks needed'
      });
      return next(); // Grant access immediately - ADMIN has all permissions
    }

    // Check if user role is in allowed roles
    if (roles.includes(req.userRole)) {
      logger.info('Authorization granted: Role match', {
        url: req.url,
        userRole: req.userRole
      });
      return next();
    }

    logger.warn('❌ Authorization failed: Insufficient permissions', {
      url: req.url,
      method: req.method,
      userRole: req.userRole,
      requiredRoles: roles,
      userEmail: req.user?.email,
      userPermissions: req.user?.permissions,
      message: `User role '${req.userRole}' is not in required roles: [${roles.join(', ')}]. Only ADMIN role has full access.`
    });
    
    // Provide helpful error message
    let errorMessage = 'Insufficient permissions';
    if (req.userRole === 'EMPLOYEE') {
      errorMessage = 'This endpoint requires ADMIN access. Please log in as the company owner.';
    } else if (!roles.includes(req.userRole)) {
      errorMessage = `This endpoint requires one of these roles: ${roles.join(', ')}. Your role: ${req.userRole}`;
    }
    
    return errorResponse(res, errorMessage, 403);
  };
};

/**
 * Manager isolation middleware - ensures manager can only see their assigned agents' jobs
 */
const managerIsolation = async (req, res, next) => {
  try {
    if (req.userRole !== 'MANAGER') {
      return next(); // Skip for non-managers
    }

    // Manager can only access jobs from their assigned agents
    // This will be enforced in the service layer
    req.managerId = req.userId;
    next();
  } catch (error) {
    logger.error('Manager isolation error:', error);
    return errorResponse(res, 'Authorization failed', 403);
  }
};

module.exports = {
  authenticateToken,
  authorize,
  managerIsolation,
};

