/**
 * Periodic Maintenance Job Generator
 * Cron job to automatically generate jobs from schedules
 */

const cron = require('node-cron');
const PeriodicMaintenanceSchedule = require('../models/PeriodicMaintenanceSchedule.model');
const Job = require('../models/Job.model');
const User = require('../models/User.model');
const Notification = require('../models/Notification.model');
const { calculateSLADeadline } = require('../utils/sla');
const logger = require('../utils/logger');

/**
 * Calculate next generation date based on recurrence
 */
const calculateNextGenerationDate = (startDate, interval) => {
  const date = new Date(startDate);
  
  switch (interval) {
    case 'DAILY':
      date.setDate(date.getDate() + 1);
      break;
    case 'WEEKLY':
      date.setDate(date.getDate() + 7);
      break;
    case 'MONTHLY':
      date.setMonth(date.getMonth() + 1);
      break;
    case 'QUARTERLY':
      date.setMonth(date.getMonth() + 3);
      break;
    case 'SEMI_ANNUAL':
      date.setMonth(date.getMonth() + 6);
      break;
    case 'YEARLY':
      date.setFullYear(date.getFullYear() + 1);
      break;
    default:
      date.setDate(date.getDate() + 1);
  }

  return date;
};

/**
 * Generate jobs for all active schedules
 * Runs daily at 2 AM
 */
const generatePeriodicMaintenanceJobs = async () => {
  try {
    logger.info('Starting periodic maintenance job generation...');
    
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);

    // Find all active schedules that need job generation
    const schedules = await PeriodicMaintenanceSchedule.find({
      isActive: true,
      assignedAgentId: { $exists: true, $ne: null }, // Only schedules with assigned agents
      $or: [
        { nextGenerationDate: { $lte: tomorrow } },
        { nextGenerationDate: { $exists: false } },
      ],
    }).populate('assignedAgentId', 'name email phone managerId')
      .populate('customerId', 'name email phone address');

    let totalJobsGenerated = 0;

    for (const schedule of schedules) {
      try {
        // Check if jobs need to be generated (day before scheduled date)
        const shouldGenerate = schedule.nextGenerationDate && 
          new Date(schedule.nextGenerationDate) <= tomorrow;

        if (!shouldGenerate) {
          continue;
        }

        const jobs = [];

        // Check if schedule has assigned agent
        if (!schedule.assignedAgentId) {
          logger.warn(`Schedule ${schedule._id} has no assigned agent, skipping`);
          continue;
        }

        const assignedAgent = schedule.assignedAgentId;
        
        // Get customer - schedule uses Customer model (not User)
        const Customer = require('../models/Customer.model');
        let customer = null;
        
        if (schedule.customerId) {
          // Populate customerId if it's already populated
          if (typeof schedule.customerId === 'object' && schedule.customerId._id) {
            customer = schedule.customerId;
          } else {
            // If it's just an ID, fetch the customer
            customer = await Customer.findById(schedule.customerId);
          }
        }

        if (!customer) {
          logger.warn(`Schedule ${schedule._id} has no customer, skipping`);
          continue;
        }

          // Get customer address - Customer model uses different structure
          let jobLocationLat = 0;
          let jobLocationLng = 0;
          let jobLocationAddress = schedule.address?.fullAddress || schedule.address?.native || '';

          // Try to get coordinates from customer if available
          if (customer.address) {
            if (customer.address.coordinates) {
              jobLocationLat = customer.address.coordinates.lat || 0;
              jobLocationLng = customer.address.coordinates.lng || 0;
            }
            if (!jobLocationAddress && customer.address.fullAddress) {
              jobLocationAddress = customer.address.fullAddress;
            }
          }

          // Create job
          const job = new Job({
            tenantId: schedule.tenantId,
            ticketType: 'PERIODIC',
            customerId: customer._id,
            customerName: customer.name || schedule.customerName,
            customerPhone: customer.contactNumber || customer.phone,
            customerEmail: customer.email,
            assignedAgentId: assignedAgent._id,
            managerId: assignedAgent.managerId || schedule.managerId || null,
            status: 'NEW',
            jobLocationLat: jobLocationLat,
            jobLocationLng: jobLocationLng,
            jobLocationAddress: jobLocationAddress,
            scheduledDate: schedule.nextGenerationDate,
            scheduleId: schedule._id,
            slaDeadline: calculateSLADeadline('PERIODIC', null, schedule.nextGenerationDate),
          });

          await job.save();

          // Create notification for agent
          const notification = new Notification({
            tenantId: schedule.tenantId,
            userId: assignedAgent._id,
            jobId: job._id,
            title: 'New Periodic Maintenance Job',
            message: `New ${schedule.serviceType} job assigned for ${customer.name}`,
            notificationType: 'JOB_ASSIGNED',
          });
          await notification.save();

          // Notify manager if assigned
          if (assignedAgent.managerId) {
            const managerNotification = new Notification({
              tenantId: schedule.tenantId,
              userId: assignedAgent.managerId,
              jobId: job._id,
              title: 'New Periodic Maintenance Job',
              message: `New ${schedule.serviceType} job assigned to ${assignedAgent.name}`,
              notificationType: 'JOB_ASSIGNED',
            });
            await managerNotification.save();
          }

          jobs.push(job);
          totalJobsGenerated++;
        }

        // Update schedule
        schedule.lastGeneratedAt = new Date();
        // Use frequency field (preferred) or fallback to recurrenceInterval
        const frequency = schedule.frequency || schedule.recurrenceInterval || 'DAILY';
        schedule.nextGenerationDate = calculateNextGenerationDate(
          schedule.nextGenerationDate,
          frequency
        );
        await schedule.save();

        logger.info(`Generated ${jobs.length} jobs for schedule ${schedule.name} (${schedule._id})`);
      } catch (error) {
        logger.error(`Error generating jobs for schedule ${schedule._id}:`, error);
      }
    }

    logger.info(`Periodic maintenance job generation completed. Total jobs generated: ${totalJobsGenerated}`);
    return { success: true, jobsGenerated: totalJobsGenerated };
  } catch (error) {
    logger.error('Error in periodic maintenance job generation:', error);
    throw error;
  }
};

/**
 * Start cron job
 * Runs daily at 2 AM
 */
const startPeriodicMaintenanceCron = () => {
  // Run daily at 2 AM
  cron.schedule('0 2 * * *', async () => {
    await generatePeriodicMaintenanceJobs();
  });

  logger.info('Periodic maintenance cron job started (runs daily at 2 AM)');
};

/**
 * Manual trigger for testing
 */
const triggerJobGeneration = async () => {
  return await generatePeriodicMaintenanceJobs();
};

module.exports = {
  startPeriodicMaintenanceCron,
  generatePeriodicMaintenanceJobs,
  triggerJobGeneration,
};

