/**
 * User Controller
 */

const userService = require('../services/user.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult, query } = require('express-validator');

/**
 * Get users
 */
const getUsers = async (req, res) => {
  try {
    const filters = {
      role: req.query.role,
    };

    const users = await userService.getUsers(
      filters,
      req.tenantId,
      req.userRole,
      req.userId
    );

    return successResponse(res, users, 'Users retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve users', 500);
  }
};

/**
 * Get user by ID
 */
const getUserById = async (req, res) => {
  try {
    const { id } = req.params;

    const user = await userService.getUserById(
      id,
      req.tenantId,
      req.userRole,
      req.userId
    );

    return successResponse(res, user, 'User retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve user', 404);
  }
};

/**
 * Create user
 */
const createUser = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const user = await userService.createUser(
      req.body,
      req.tenantId,
      req.userRole
    );

    return successResponse(res, user, 'User created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create user', 400);
  }
};

/**
 * Update user
 */
const updateUser = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;

    const user = await userService.updateUser(
      id,
      req.body,
      req.tenantId,
      req.userRole,
      req.userId
    );

    return successResponse(res, user, 'User updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update user', 400);
  }
};

/**
 * Delete user
 */
const deleteUser = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await userService.deleteUser(
      id,
      req.tenantId,
      req.userRole
    );

    return successResponse(res, result, 'User deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete user', 400);
  }
};

/**
 * Assign agents to manager
 */
const assignAgentsToManager = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { managerId } = req.params;
    const { agentIds } = req.body;

    const manager = await userService.assignAgentsToManager(
      managerId,
      agentIds,
      req.tenantId
    );

    return successResponse(res, manager, 'Agents assigned successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to assign agents', 400);
  }
};

/**
 * Update user location activity
 * This endpoint is accessible to ALL authenticated users (EMPLOYEE, CUSTOMER, ADMIN, etc.)
 */
const updateLocation = async (req, res) => {
  try {
    // Debug logging
    console.log('📍 [UPDATE LOCATION] Request received');
    console.log('   User ID:', req.userId);
    console.log('   User Role:', req.userRole);
    console.log('   Tenant ID:', req.tenantId);
    console.log('   URL:', req.url);
    console.log('   Method:', req.method);

    const { latitude, longitude, locationName, timestamp } = req.body;

    if (!latitude || !longitude) {
      return errorResponse(res, 'Latitude and longitude are required', 400);
    }

    const result = await userService.updateUserLocation(
      req.userId,
      req.userRole,
      req.tenantId,
      {
        latitude: parseFloat(latitude),
        longitude: parseFloat(longitude),
        locationName: locationName || null,
        timestamp: timestamp || new Date().toISOString(),
        ipAddress: req.ip || req.connection.remoteAddress,
        userAgent: req.get('user-agent'),
      }
    );

    console.log('✅ [UPDATE LOCATION] Success for role:', req.userRole);
    return successResponse(res, result, 'Location updated successfully');
  } catch (error) {
    console.error('❌ [UPDATE LOCATION] Error:', error);
    return errorResponse(res, error.message || 'Failed to update location', 400);
  }
};

/**
 * Validation rules
 */
const validateCreateUser = [
  body('email').isEmail().withMessage('Valid email is required'),
  body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
  body('name').notEmpty().withMessage('Name is required'),
  body('role').isIn(['ADMIN', 'MANAGER', 'SERVICE_AGENT', 'CUSTOMER']).withMessage('Valid role is required'),
];

const validateUpdateUser = [
  body('email').optional().isEmail().withMessage('Valid email is required'),
  body('password').optional().isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
];

/**
 * Get manager's assigned agents
 */
const getManagerAssignedAgents = async (req, res) => {
  try {
    console.log('🔵 [GET MANAGER AGENTS] Request received');
    console.log('   User ID:', req.userId);
    console.log('   User Role:', req.userRole);
    console.log('   Role Name:', req.roleName);
    console.log('   Tenant ID:', req.tenantId);
    
    // Only managers can access their own assigned agents
    // Check if user is a manager (by roleName or userRole)
    const isManager = (req.roleName && req.roleName.toLowerCase() === 'manager') || req.userRole === 'MANAGER';
    
    if (!isManager) {
      console.log('❌ [GET MANAGER AGENTS] Access denied - not a manager');
      return errorResponse(res, 'Only managers can access assigned agents', 403);
    }

    console.log('✅ [GET MANAGER AGENTS] Manager access confirmed');

    // Use the manager's ID (could be Employee ID or User ID)
    // The service will handle finding the correct User account
    const managerId = req.userId;
    
    console.log('📱 [GET MANAGER AGENTS] Calling service with managerId:', managerId);
    const agents = await userService.getManagerAssignedAgents(managerId, req.tenantId);

    console.log('✅ [GET MANAGER AGENTS] Service returned', agents.length, 'agents');
    return successResponse(res, agents, 'Assigned agents retrieved successfully');
  } catch (error) {
    console.error('❌ [GET MANAGER AGENTS] Error:', error);
    return errorResponse(res, error.message || 'Failed to retrieve assigned agents', 500);
  }
};

const validateAssignAgents = [
  body('agentIds').isArray().withMessage('agentIds must be an array'),
  body('agentIds.*').isMongoId().withMessage('Each agentId must be a valid MongoDB ID'),
];

module.exports = {
  getUsers,
  getUserById,
  createUser,
  updateUser,
  deleteUser,
  assignAgentsToManager,
  updateLocation,
  getManagerAssignedAgents,
  validateCreateUser,
  validateUpdateUser,
  validateAssignAgents,
};

