/**
 * Tenant Controller
 */

const tenantService = require('../services/tenant.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get tenants
 */
const getTenants = async (req, res) => {
  try {
    const filters = {
      isActive: req.query.isActive !== undefined ? req.query.isActive === 'true' : undefined,
      status: req.query.status || undefined,
      search: req.query.search || undefined,
    };

    const companies = await tenantService.getTenants(filters);

    return successResponse(res, companies, 'Companies retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve companies', 500);
  }
};

/**
 * Get tenant by ID
 */
const getTenantById = async (req, res) => {
  try {
    const { id } = req.params;

    const tenant = await tenantService.getTenantById(id);

    return successResponse(res, tenant, 'Tenant retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve tenant', 404);
  }
};

/**
 * Create tenant
 */
const createTenant = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const tenant = await tenantService.createTenant(req.body);

    return successResponse(res, tenant, 'Tenant created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create tenant', 400);
  }
};

/**
 * Update tenant
 */
const updateTenant = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const updateData = { ...req.body };

    // Validate password if provided
    if (updateData.adminPassword && updateData.adminPassword.trim() !== '') {
      if (updateData.adminPassword.length < 8) {
        return errorResponse(res, 'Password must be at least 8 characters long', 400);
      }
      if (!/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/.test(updateData.adminPassword)) {
        return errorResponse(res, 'Password must contain at least one uppercase letter, one lowercase letter, and one number', 400);
      }
    }

    const company = await tenantService.updateTenant(id, updateData);

    return successResponse(res, company, 'Company updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update company', 400);
  }
};

/**
 * Update white-label configuration
 */
const updateWhiteLabelConfig = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;

    const tenant = await tenantService.updateWhiteLabelConfig(id, req.body);

    return successResponse(res, tenant, 'White-label configuration updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update white-label config', 400);
  }
};

/**
 * Activate tenant
 */
const activateTenant = async (req, res) => {
  try {
    const { id } = req.params;

    const tenant = await tenantService.activateTenant(id);

    return successResponse(res, tenant, 'Tenant activated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to activate tenant', 400);
  }
};

/**
 * Deactivate tenant
 */
const deactivateTenant = async (req, res) => {
  try {
    const { id } = req.params;

    const tenant = await tenantService.deactivateTenant(id);

    return successResponse(res, tenant, 'Tenant deactivated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to deactivate tenant', 400);
  }
};

/**
 * Delete tenant
 */
const deleteTenant = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await tenantService.deleteTenant(id);

    return successResponse(res, result, 'Tenant deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete tenant', 400);
  }
};

/**
 * Validation rules
 */
/**
 * Validate create tenant request
 */
const validateCreateTenant = [
  body('name').notEmpty().withMessage('Name is required'),
  body('email').isEmail().withMessage('Valid email is required'),
  body('adminPassword')
    .notEmpty().withMessage('Password is required')
    .isLength({ min: 8 }).withMessage('Password must be at least 8 characters long')
    .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/).withMessage('Password must contain at least one uppercase letter, one lowercase letter, and one number'),
];

/**
 * Create default roles for all existing companies
 */
const createDefaultRolesForAll = async (req, res) => {
  try {
    const result = await tenantService.createDefaultRolesForAllCompanies();
    return successResponse(res, result, result.message);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create default roles', 500);
  }
};

const validateWhiteLabelConfig = [
  body('logoUrl').optional().isURL().withMessage('Valid URL is required'),
  body('primaryColor').optional().matches(/^#[0-9A-Fa-f]{6}$/).withMessage('Valid hex color is required'),
  body('secondaryColor').optional().matches(/^#[0-9A-Fa-f]{6}$/).withMessage('Valid hex color is required'),
];

module.exports = {
  getTenants,
  getTenantById,
  createTenant,
  updateTenant,
  updateWhiteLabelConfig,
  activateTenant,
  deactivateTenant,
  deleteTenant,
  createDefaultRolesForAll,
  validateCreateTenant,
  validateWhiteLabelConfig,
};

