/**
 * Settings Controller
 * Handles company settings including logo and branding
 */

const Tenant = require('../models/Tenant.model');
const { successResponse, errorResponse } = require('../utils/response');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const logger = require('../utils/logger');

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = 'uploads/logos';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, `logo-${req.tenantId}-${uniqueSuffix}${path.extname(file.originalname)}`);
  },
});

const fileFilter = (req, file, cb) => {
  const allowedTypes = /jpeg|jpg|png|gif|svg|webp/;
  const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
  const mimetype = allowedTypes.test(file.mimetype);

  if (mimetype && extname) {
    return cb(null, true);
  } else {
    cb(new Error('Only image files are allowed (jpeg, jpg, png, gif, svg, webp)'));
  }
};

const upload = multer({
  storage: storage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB limit
  fileFilter: fileFilter,
});

/**
 * Get company settings
 */
const getSettings = async (req, res) => {
  try {
    const company = await Tenant.findById(req.tenantId).select('name email phone address logoUrl primaryColor secondaryColor');

    if (!company) {
      return errorResponse(res, 'Company not found', 404);
    }

    return successResponse(res, company, 'Settings retrieved successfully');
  } catch (error) {
    logger.error('Get settings error:', error);
    return errorResponse(res, error.message || 'Failed to retrieve settings', 500);
  }
};

/**
 * Update company settings
 */
const updateSettings = async (req, res) => {
  try {
    const company = await Tenant.findById(req.tenantId);

    if (!company) {
      return errorResponse(res, 'Company not found', 404);
    }

    // Update allowed fields
    const allowedFields = ['name', 'email', 'phone', 'address', 'primaryColor', 'secondaryColor'];
    allowedFields.forEach(field => {
      if (req.body[field] !== undefined) {
        company[field] = req.body[field];
      }
    });

    await company.save();

    logger.info(`Settings updated for company: ${company.name}`, { companyId: company._id });

    return successResponse(res, company, 'Settings updated successfully');
  } catch (error) {
    logger.error('Update settings error:', error);
    return errorResponse(res, error.message || 'Failed to update settings', 400);
  }
};

/**
 * Upload logo
 */
const uploadLogo = async (req, res) => {
  try {
    if (!req.file) {
      return errorResponse(res, 'No file uploaded', 400);
    }

    const company = await Tenant.findById(req.tenantId);

    if (!company) {
      // Delete uploaded file if company not found
      if (req.file.path) {
        fs.unlinkSync(req.file.path);
      }
      return errorResponse(res, 'Company not found', 404);
    }

    // Delete old logo if exists
    if (company.logoUrl && company.logoUrl.startsWith('/uploads/logos/')) {
      const oldLogoPath = path.join(process.cwd(), company.logoUrl);
      if (fs.existsSync(oldLogoPath)) {
        fs.unlinkSync(oldLogoPath);
      }
    }

    // Update logo URL
    company.logoUrl = `/uploads/logos/${req.file.filename}`;
    await company.save();

    logger.info(`Logo uploaded for company: ${company.name}`, { companyId: company._id, logoUrl: company.logoUrl });

    return successResponse(res, { logoUrl: company.logoUrl }, 'Logo uploaded successfully');
  } catch (error) {
    logger.error('Upload logo error:', error);
    // Delete uploaded file on error
    if (req.file && req.file.path) {
      fs.unlinkSync(req.file.path);
    }
    return errorResponse(res, error.message || 'Failed to upload logo', 400);
  }
};

/**
 * Delete logo
 */
const deleteLogo = async (req, res) => {
  try {
    const company = await Tenant.findById(req.tenantId);

    if (!company) {
      return errorResponse(res, 'Company not found', 404);
    }

    // Delete logo file if exists
    if (company.logoUrl && company.logoUrl.startsWith('/uploads/logos/')) {
      const logoPath = path.join(process.cwd(), company.logoUrl);
      if (fs.existsSync(logoPath)) {
        fs.unlinkSync(logoPath);
      }
    }

    // Clear logo URL
    company.logoUrl = undefined;
    await company.save();

    logger.info(`Logo deleted for company: ${company.name}`, { companyId: company._id });

    return successResponse(res, { success: true }, 'Logo deleted successfully');
  } catch (error) {
    logger.error('Delete logo error:', error);
    return errorResponse(res, error.message || 'Failed to delete logo', 400);
  }
};

module.exports = {
  getSettings,
  updateSettings,
  uploadLogo: [upload.single('logo'), uploadLogo],
  deleteLogo,
};

