/**
 * Role Controller
 */

const roleService = require('../services/role.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get roles
 */
const getRoles = async (req, res) => {
  try {
    const filters = {
      status: req.query.status || undefined,
      search: req.query.search || undefined,
    };

    const roles = await roleService.getRoles(req.tenantId, filters);

    return successResponse(res, roles, 'Roles retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve roles', 500);
  }
};

/**
 * Get role by ID
 */
const getRoleById = async (req, res) => {
  try {
    const { id } = req.params;

    const role = await roleService.getRoleById(id, req.tenantId);

    return successResponse(res, role, 'Role retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve role', 404);
  }
};

/**
 * Create role
 */
const createRole = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const role = await roleService.createRole(
      req.body,
      req.tenantId,
      req.userId
    );

    return successResponse(res, role, 'Role created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create role', 400);
  }
};

/**
 * Update role
 */
const updateRole = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;

    const role = await roleService.updateRole(id, req.body, req.tenantId);

    return successResponse(res, role, 'Role updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update role', 400);
  }
};

/**
 * Delete role
 */
const deleteRole = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await roleService.deleteRole(id, req.tenantId);

    return successResponse(res, result, 'Role deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete role', 400);
  }
};

/**
 * Validation rules
 */
const validateCreateRole = [
  body('name').notEmpty().withMessage('Role name is required'),
  body('permissions').optional().isObject().withMessage('Permissions must be an object'),
];

const validateUpdateRole = [
  body('name').optional().notEmpty().withMessage('Role name cannot be empty'),
  body('permissions').optional().isObject().withMessage('Permissions must be an object'),
];

module.exports = {
  getRoles,
  getRoleById,
  createRole,
  updateRole,
  deleteRole,
  validateCreateRole,
  validateUpdateRole,
};

