/**
 * Product Controller
 */

const productService = require('../services/product.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get products
 */
const getProducts = async (req, res) => {
  try {
    const filters = {
      status: req.query.status || undefined,
      productType: req.query.productType || undefined,
      search: req.query.search || undefined,
    };

    const products = await productService.getProducts(req.tenantId, filters);

    return successResponse(res, products, 'Products retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve products', 500);
  }
};

/**
 * Get product by ID
 */
const getProductById = async (req, res) => {
  try {
    const { id } = req.params;

    const product = await productService.getProductById(id, req.tenantId);

    return successResponse(res, product, 'Product retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve product', 404);
  }
};

/**
 * Get customer's assigned products
 */
const getCustomerProducts = async (req, res) => {
  try {
    const products = await productService.getCustomerProducts(req.userId, req.tenantId);
    return successResponse(res, products, 'Customer products retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve customer products', 500);
  }
};

/**
 * Create product
 */
const createProduct = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const product = await productService.createProduct(
      req.body,
      req.tenantId,
      req.userId
    );

    return successResponse(res, product, 'Product created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create product', 400);
  }
};

/**
 * Update product
 */
const updateProduct = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;

    const product = await productService.updateProduct(id, req.body, req.tenantId);

    return successResponse(res, product, 'Product updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update product', 400);
  }
};

/**
 * Delete product
 */
const deleteProduct = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await productService.deleteProduct(id, req.tenantId);

    return successResponse(res, result, 'Product deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete product', 400);
  }
};

/**
 * Validation rules
 */
const validateCreateProduct = [
  body('name').notEmpty().withMessage('Product name is required'),
  body('productType').isIn(['product', 'spare', 'accessories']).withMessage('Invalid product type'),
];

const validateUpdateProduct = [
  body('name').optional().notEmpty().withMessage('Product name cannot be empty'),
  body('productType').optional().isIn(['product', 'spare', 'accessories']).withMessage('Invalid product type'),
];

module.exports = {
  getProducts,
  getProductById,
  getCustomerProducts,
  createProduct,
  updateProduct,
  deleteProduct,
  validateCreateProduct,
  validateUpdateProduct,
};

